use 5.014;
use utf8;
use strict;
use warnings;


package NNIS::MessageBus::Connector;

our $VERSION = '1.08';

use Params::Validate qw(validate);
use Carp qw(croak);
use AnyEvent;
use Moo;

use NNIS::MessageBus::Connection;

has heartbeat => (
    is          => 'ro',
    default     => sub { 580 },
);

has timeout => (
    is      => 'rw',
    default => sub { 20 },
);

has source_service => (
    is          => 'rw',
    required    => 1,
);

has response_address => (
    is          => 'lazy',
);

has config      => (
    is          => 'rw',
    default     => sub { { } },
);

has debug       => (
    is          => 'rw',
    default     => sub { 0 },
);

sub _next_host {
    my ( $self ) = @_;
    my $host_spec = $self->config->{host};
    return $host_spec unless ref $host_spec;
    my $host_count = scalar @$host_spec;
    $self->{_host_index} //= 0;
    return $host_spec->[ $self->{_host_index}++ % $host_count];
}

sub _create_connction {
    my $self = shift;
    # TODO: less hacky way
    my %args = %$self;
    $args{config} = { %{ $args{config} } };
    $args{config}{host} = $self->_next_host;
    delete $args{_consumer_connections};
    delete $args{_producer_connections};

    return NNIS::MessageBus::Connection->new( connector => $self, %args );
}

sub _discard_dead_connections {
    my $self = shift;
    for my $pool (qw(_consumer_connections _producer_connections)) {
        my @alive;
        next unless $self->{$pool};
        for my $connection (@{ $self->{$pool} }) {
            if ($connection->is_alive) {
                push @alive, $connection;
            }
            else {
                $connection->teardown;
            }
        }
        @{$self->{$pool}} = @alive;
    }
    return $self;
}

sub consumer_connection {
    my $self = shift;
    $self->_discard_dead_connections;
    $self->{_consumer_connections}[0] //= $self->_create_connction();
};

sub producer_connection {
    my $self = shift;
    $self->_discard_dead_connections;
    $self->{_producer_connections}[0] //= $self->_create_connction();
};

sub background_connect {
    my $self = shift;
    $self->consumer_connection->background_connect;
    return $self;
}

sub ask {
    my $self = shift;
    $self->producer_connection->ask(@_);
}
sub ask_async {
    my $self = shift;
    $self->producer_connection->ask_async(@_);
}

sub send {
    my $self = shift;
    $self->producer_connection->send(@_);
}
sub send_async {
    my $self = shift;
    $self->producer_connection->send_async(@_);
}

sub register_consumer {
    my $self = shift;
    if ($self->{_is_consuming}) {
        croak "Cannot call ->register_consumer() after ->consume().\n";
    }
    validate(@_, {
        queue           => 0,
        has_response    => 1,
        callback        => 1,
    });

    require NNIS::MessageBus::Consumer;
    my $consumer = NNIS::MessageBus::Consumer->new(@_);
    push @{ $self->{_consumers} }, $consumer;

    $self;
}

sub start_consumption {
    my $self = shift;
    if ($self->{_is_consuming}) {
        croak "Cannot call start_consumption: already consuming.\n";
    }
    $self->{_is_consuming} = 1;

    $self->_runloop();
}

sub _ensure_consumer {
    my $self = shift;
    $self->_discard_dead_connections;

    # make sure we have at least one connection:
    scalar $self->consumer_connection();

    # the consumers (hopefully) calls this when they fail
    my $error_callback = sub {
        $self->_ensure_consumer;
    };

    for my $connection ( @{$self->{_consumer_connections}} ) {
        unless ($connection->is_consuming) {
            $connection->set_consumers(@{ $self->{_consumers} });
            $connection->start_consumption($error_callback);
        }
    }
}

sub _runloop {
    my $self = shift;
    my $time_constant = 0.5 * $self->heartbeat;

    $self->_ensure_consumer();

    $self->{_runloop_timer} = AnyEvent->timer(
        after           => $time_constant,
        interval        => $time_constant,
        cb              => sub {
            $self->_ensure_consumer;
        },
    );

    # infinite loop, let the event loop do the rest
    if (AnyEvent::detect() eq 'AnyEvent::Impl::EV') {
        require EV;
        EV::run;
    }
    else {
        warn "WARNING: Runloop $AnyEvent::MODEL detected; you should really be using EV ('use EV; as the first line in your program) instead\n";
        my $cv = AnyEvent->condvar;
        $cv->recv;
    }
}

sub DESTROY {
    my $self = shift;
    say "DESTROYing " . ref($self) if $self->debug;
    for my $connection (@{ $self->{_consumer_connections} // []},
                        @{ $self->{_producer_connections} // [] }) {
        # during global destruction, things can be undef that normally
        # wouldn't, so need to check if that's the case here:
        $connection->teardown() if defined $connection;
    }
    say "Done DESTROYing " . ref($self) if $self->debug;
}


1;
__END__

=encoding UTF-8

=head1 NAME

NNIS::MessageBus::Connector - Connect to the message bus and communicate in NMF

=head1 SYNOPSIS

    use EV;     # Force the right event loop
    use NNIS::MessageBus::Connector;
    use Data::Dumper;

    my $connector = NNIS::MessageBus::Connector->new(
        source_service  => 'nnis.mlenz-workstation.kunde',
        config          => {
            reply_exchange  => 'reply',
            tls             => 0,
        },
    );


    # producer API: send a request, wait for response
    my $response = $connector->ask(
        routing_key => 'nnis.netapp.na-cl1-nbg6b.list_volumes',
        payload     => {},
        type        => 'ListVolumes',
    );

    print Dumper $response->{'nmf-body'}{volumes}[0];

    # or using the consumer API: provide a service
    $connector->register_consumer(
        queue           => 'nnis.netapp.na-cl1-nbg6b.list_volumes',
        has_response    => 1,
        callback        => sub {
            my $request_body = shift;
            # go crazy here!

            my $payload = {
                all => ['those', 'happy', 'people'],
            };

            return ($payload, 'success');
        },
    );
    $connector->start_consumption;


=head1 DESCRIPTION

NNIS::MessageBus::Connector is a (mostly-)synchronous layer over
L<AnyEvent::RabbitMQ>. It can be used to talk to other NNIS services over the
RabbitMQ broker.

=head1 ATTRIBUTES

The follow attributes can be set via the C<new> method, and queried (and some
of them set) via methods of the same name.

=head2 config

A hash reference with some of these keys:

    key                 default         meaning
    ===============     =========       =================================
    reply_exchange      reply           Exchange to which replies are sent
    host                localhost       Host of the RabbitMQ broker
    port                5672,           Port of the RabbitMQ broker
    user                guest           Username for broker authentication
    password            guest           Password for broker authentication
    vhost               /               Broker vhost
    tls                 1               Whether to use TLS (SSL)

Note that C<host> can be a an array reference of host names in a cluster, in which
case the hosts are used in a round-robin fashion.

=head2 connection

The L<AnyEvent::RabbitMQ> connection object; built automatically from the
config values.

=head2 source_service

The name of the application that uses this library.

Mandatory.

=head2 timeout

How many seconds to wait for a response.

If the timout is exceeded, throws an exception.

Defaults to 20.

=head1 METHODS

=head2 new

    my $connector = NNIS::MessageBus::Connector->new(
        config => { tls => 1, user => $user, password => $password },
        source_service => 'nnis.weird.thingy',
    );

Builds a new connector object.

=head2 ask

    my $anser = $connection->ask(
        routing_key     => 'nnis.netapp.nbg4.ListVolumes',      # mandatory
        type            => 'ListVolumes',                       # mandatory
        payload         => { },                                 # mandatory
        class           => 'Query'                              # optional
        exchange        => 'nnnis.netapp.nbg4',                 # optional
        timeout         => 20,                                  # optional
        version         => '1.0',                               # optional
    );

Sends a C<Query> (defaul) or a C<Command> message of the given C<type>,
declares a reply queue, and waits for a reply. The result is a hash ref
with the full NMF data structure, so with keys C<nmf-header> and C<nmf-body>.

=head2 ask_async

Same as C<ask>, but instead of waiting for a reply, returns an AnyEvent
condition variable (condvar) which will be fullfilled once the answer arives,
or breaks/croaks on timeout or connection loss.

=head2 register_consumer

Adds a consumer which will be called whenever a new message arrives in the
queue. You must pass three key/value pairs: C<queue> for the name of the queue
that the callback subscribes to, C<callback>, which must be a code reference,
and which will be called for each new message in the queue, and C<has_reply>.
If C<has_reply> is true, then the return value from the callback must be a
list of two elements: the payload and a status string, which can be any of
C<success>, C<warning> or C<error>. A response message will be constructed
from those values, and sent to the original requestor.

=head2 start_consumption

Connects to the message bus and starts consuming from the queues.

This command won't return.

=cut
