#===============================================================#
#                                                               #
# $ID$                                                          #
#                                                               #
# multistore.pl                                                 #
#                                                               #
# Copyright (c) 2009 NetApp, Inc. All rights reserved.          #
# Specifications subject to change without notice.              #
#                                                               #
# Sample code to demonstrate how to:                            #
#        - create/destroy/setup a multistore                    #
#                                                               #
# This Sample code is supported from DataFabric Manager 3.8     #
# onwards.                                                      #
# However few of the functionalities of the sample code may     #
# work on older versions of DataFabric Manager.                 #
#===============================================================#
require 5.6.1;

use lib '../../../../../../../lib/perl/NetApp';
use NaServer;
use NaElement;
use strict;

# Variables declaration
my $args      = $#ARGV + 1;
my $dfmserver = shift;
my $dfmuser   = shift;
my $dfmpw     = shift;
my $command   = shift;

# check for valid number of parameters
if ( $args < 4 ) {
	print_usage();
}

sub print_usage() {
	print <<MSG;
Usage:
 multistore.pl <dfmserver> <user> <password> create <vfiler> <ip> <protocol> <rpool>
 multistore.pl <dfmserver> <user> <password> destroy <vfiler>
 multistore.pl <dfmserver> <user> <password> setup <vfiler> <if> <ip> <nm> [-c]

 <dfmserver>  -- Name/IP Address of the DFM server
 <user>       -- DFM server User name
 <password>   -- DFM server User Password
 <vfiler>     -- Vfiler name to be created or setup
 <ip>         -- IP Address to be assigned to the vfiler
 <protocol>   -- nas - for NFS & CIFS
				 san - for iSCSI
				 all - for both NFS & CIFS
 <rpool>      -- Resource pool in which vfiler will be created
 <if>         -- interface on the vfiler to be used, for e.g e0a, e0b
 <nm>         -- netmask on the vfiler to be used, for e.g 255.255.255.0
 -c           -- specify this flag to run cifs setup for nas & all protocols
MSG
	exit 1;
}

# Setup DFM server connection
my $s = NaServer->new( $dfmserver, 1, 0 );
$s->set_style("LOGIN");
$s->set_transport_type("HTTP");
$s->set_server_type("DFM");
$s->set_port(8088);
$s->set_admin_user( $dfmuser, $dfmpw );

# Create a new Multistore
if ( $command eq "create" ) {
	if ( $args < 8 ) {
		print_usage();
	}
	my $vfiler_name = shift;
	my $ip          = shift;
	my $protocols   = shift;
	my $rpool       = shift;

	# Create multistore with ip and choose the resourcepool to be created from.
	my $in = NaElement->new("vfiler-create");
	$in->child_add_string( "name",                $vfiler_name );
	$in->child_add_string( "ip-address",          $ip );
	$in->child_add_string( "resource-name-or-id", $rpool );

	# Based on the option on the commandline setup the right protocols
	# needed on the multistore
	my $allproto = NaElement->new("allowed-protocols");
	if( $protocols eq "all" ) {
		$allproto->child_add_string( "protocols", "nfs" );
		$allproto->child_add_string( "protocols", "cifs" );
		$allproto->child_add_string( "protocols", "iscsi" );
	} elsif( $protocols eq "nas" ) {
		$allproto->child_add_string( "protocols", "nfs" );
		$allproto->child_add_string( "protocols", "cifs" );
	} elsif( $protocols eq "san" ) {
		$allproto->child_add_string( "protocols", "iscsi" );
	} else { 
		print "Protocols allowed are: nfs, cifs, all\n"; exit 1; 
	};
	$in->child_add($allproto);

	my $out = $s->invoke_elem($in);
	if ( $out->results_status() eq "failed" ) {
		print( "Error : " . $out->results_reason() . "\n" );
		exit(-2);
	}

	print "VFiler '"
	  . $vfiler_name
	  . "' created on "
	  . $out->child_get_string("filer-name") . ":"
	  . $out->child_get_string("root-volume-name");
}
# After a multistore creation, you need to set it up with right IPs and CIFS
# as needed.  With out this step Multistore is not completely operational.
elsif ( $command eq "setup" ) {
	if ( $args < 8 ) {
		print_usage();
	}
	my $vfiler_name = shift;
	my $interface   = shift;
	my $ip          = shift;
	my $netmask     = shift;
	my $cifs        = shift;

	my $in = NaElement->new("vfiler-setup");
	$in->child_add_string( "vfiler-name-or-id", $vfiler_name );
	if ( $cifs eq "-c" ) {
		$in->child_add_string( "run-cifs-setup", "true" );
	}

	# Have to manually choose the interface on the storage system,
	# that needs to host the IP of this multistore
	my $ipbind     = NaElement->new("ip-bindings");
	my $ipbindinfo = NaElement->new("ip-binding-info");
	$ipbindinfo->child_add_string( "interface",  $interface );
	$ipbindinfo->child_add_string( "ip-address", $ip );
	$ipbindinfo->child_add_string( "netmask",    $netmask );

	$ipbind->child_add($ipbindinfo);
	$in->child_add($ipbind);

	my $out = $s->invoke_elem($in);
	if ( $out->results_status() eq "failed" ) {
		print( "Error : " . $out->results_reason() . "\n" );
		exit(-2);
	}
	print "VFiler '" . $vfiler_name . "' setup!\n";
} elsif ( $command eq "destroy" ) {
	# Destroy an existing multistore.  This will stop and delete the multistore.
	if ( $args < 5 ) {
		print_usage();
	}
	my $vfiler_name = shift;

	my $out = $s->invoke( "vfiler-destroy", "vfiler-name-or-id", $vfiler_name );
	if ( $out->results_status() eq "failed" ) {
		print( "Error : " . $out->results_reason() . "\n" );
		exit(-2);
	}
	print "VFiler " . $vfiler_name . " destroyed! \n";
} else {
	print_usage();
}
