#===============================================================#
#                                                               #
# $ID$                                                          #
#                                                               #
# resource_pool.pl                                              #
#                                                               #
# Copyright (c) 2009 NetApp, Inc. All rights reserved.          #
# Specifications subject to change without notice.              #
#                                                               #
# Sample code to demonstrate how to:                            #
#        - list/create/delete a resource pool                   #
#        - list/add/delete members from a resource pool         #
#                                                               #
# This Sample code is supported from DataFabric Manager 3.8     #
# onwards.                                                      #
# However few of the functionalities of the sample code may     #
# work on older versions of DataFabric Manager.                 #
#===============================================================#
require 5.6.1;

use lib '../../../../../../../lib/perl/NetApp';
use NaServer;
use NaElement;
use strict;

# Variables declaration
my $args      = $#ARGV + 1;
my $dfmserver = shift;
my $dfmuser   = shift;
my $dfmpw     = shift;
my $command   = shift;

# Prints the usage of this program
sub print_usage() {
	print <<MSG;
Usage:
  resource_pool.pl <dfmserver> <user> <password> list [ResPoolName]
  resource_pool.pl <dfmserver> <user> <password> create ResPoolName [ResourceTag]
  resource_pool.pl <dfmserver> <user> <password> destroy ResPoolName
  resource_pool.pl <dfmserver> <user> <password> member list ResPoolName
  resource_pool.pl <dfmserver> <user> <password> member [add|del] ResPoolName MemberName

  <dfmserver>       -- Name/IP Address of the DFM server
  <user>            -- DFM server User name
  <password>        -- DFM server User Password
  <ResPoolName>     -- Resource pool name, mandatory for create & destroy options
  <MemberName>      -- Member to be added/removed from a resource pool, mandatory for add  & del options
MSG
	exit 1;
}

# check for valid number of parameters
if ( $args < 4 ) {
	print_usage();
}

# Setup DFM server connection
my $s = NaServer->new( $dfmserver, 1, 0 );
$s->set_style("LOGIN");
$s->set_transport_type("HTTP");
$s->set_server_type("DFM");
$s->set_port(8088);
$s->set_admin_user( $dfmuser, $dfmpw );

# List all resourcepools on the server
if ( $command eq "list" ) {
	my $rname = shift;
	my $out   = $s->invoke( "resourcepool-list-info-iter-start",
		"object-name-or-id", $rname );
	if ( $out->results_status() eq "failed" ) {
		print( "Error : " . $out->results_reason() . "\n" );
		exit(-2);
	}

	my $in = NaElement->new("resourcepool-list-info-iter-next");
	$in->child_add_string( "maximum", $out->child_get_int("records") );
	$in->child_add_string( "tag",     $out->child_get_string("tag") );

	my $out = $s->invoke_elem($in);
	if ( $out->results_status() eq "failed" ) {
		print( "Error : " . $out->results_reason() . "\n" );
		exit(-2);
	}

	# Loop through the iteration records and print details
	print "\nRESOURCEPOOLS:\n";
	print
	  "===================================================================\n";
	my $rps = $out->child_get("resourcepools");
	if ( $rps eq "" ) {
		print "Error: No Resourcepools!\n";
		exit;
	}
	my @rpInfos = $rps->children_get("resourcepools-info");
	foreach my $rpi (@rpInfos) {
		print "Name\t\t:" . $rpi->child_get_string("resourcepool-name");
		print "\n";
		print "Status\t\t:" . $rpi->child_get_string("resourcepool-status");
		print "\n";
		print "# of Members\t:"
		  . $rpi->child_get_string("resourcepool-member-count");
		print "\n";
		print "Tag\t\t:" . $rpi->child_get_string("resource-tag");
		print "\n";
		print
"===================================================================\n";
	}
} elsif ( $command eq "create" ) { # Below section creates a new resource pool
	my $rpName = shift;
	my $rTag   = shift;

	my $rpCreate = NaElement->new("resourcepool-create");
	my $rp       = NaElement->new("resourcepool");

	my $rpInfo = NaElement->new("resourcepool-info");
	$rpInfo->child_add_string( "resourcepool-name", $rpName );
	$rpInfo->child_add_string( "resource-tag",      $rTag );

	$rp->child_add($rpInfo);
	$rpCreate->child_add($rp);

	my $out = $s->invoke_elem($rpCreate);

	if ( $out->results_status() eq "failed" ) {
		print( "Error : " . $out->results_reason() . "\n" );
		exit(-2);
	}

	print "Resourcepool " 
	  . $rpName
	  . " created with ID : "
	  . $out->child_get_int("resourcepool-id");

	# Iterate through the commandline args to see if you have mentioned
	# any members to be added, if yes add them iteratively.
	foreach my $memName (@ARGV) {
		my $out =
		  $s->invoke( "resourcepool-add-member", "resourcepool-name-or-id",
			$rpName, "member-name-or-id", $memName, "resource-tag", $rTag );
		if ( $out->results_status() eq "failed" ) {
			print( "Error : " . $out->results_reason() . "\n" );
			exit(-2);
		}
		print "\nAdded member " 
		  . $memName
		  . " to Resourcepool "
		  . $rpName . "\n";
	}
} elsif ( $command eq "destroy" ) { 
	# Destroy a resourcepool, but it needs to be empty.
	my $rpName = shift;
	my $out =
	  $s->invoke( "resourcepool-destroy", "resourcepool-name-or-id", $rpName );
	if ( $out->results_status() eq "failed" ) {
		print( "Error : " . $out->results_reason() . "\n" );
		exit(-2);
	}
	print "Resourcepool " . $rpName . " destroyed!\n";
} elsif ( $command eq "member" ) { 
	# Member operations  on a resource pool
	my $subCommand = shift;
	my $rpName     = shift;
	my $memName    = shift;
	my $rTag       = shift;
	if ( $subCommand eq "list" ) {
		if ( $args < 6 ) {
			print_usage();
		}

		# Begin the iterative API
		my $out = $s->invoke( "resourcepool-member-list-info-iter-start",
			"resourcepool-name-or-id", $rpName );
		if ( $out->results_status() eq "failed" ) {
			print( "Error : " . $out->results_reason() . "\n" );
			exit(-2);
		}

	 # Use the tag and records returned by prev API call and retrive all records
		my $in = NaElement->new("resourcepool-member-list-info-iter-next");
		$in->child_add_string( "maximum", $out->child_get_int("records") );
		$in->child_add_string( "tag",     $out->child_get_string("tag") );

		my $out = $s->invoke_elem($in);
		if ( $out->results_status() eq "failed" ) {
			print( "Error : " . $out->results_reason() . "\n" );
			exit(-2);
		}

		# Print necessary details from retrieved records
		print "\nRESOURCEPOOL : " . $rpName;
		print
"\n===================================================================\n";
		my $rpms = $out->child_get("resourcepool-members");
		if ( $rpms eq "" ) {
			print "Error: No Members in Resourcepool!\n";
			exit;
		}
		my @rpmInfos = $rpms->children_get("resourcepool-member-info");
		foreach my $rpmi (@rpmInfos) {
			print "Name\t:" . $rpmi->child_get_string("member-name");
			print "\n";
			print "Status\t:" . $rpmi->child_get_string("member-status");
			print "\n";
			print "Type\t:" . $rpmi->child_get_string("member-type");
			print "\n";
			print "Tag\t:" . $rpmi->child_get_string("resource-tag");
			print "\n";
			print
"===================================================================\n";
		}
	} elsif ( $subCommand eq "add" ) { 
		# Add a new member into an existing resource pool
		if ( $args < 7 ) {
			print_usage();
		}
		my $out =
		  $s->invoke( "resourcepool-add-member", "resourcepool-name-or-id",
			$rpName, "member-name-or-id", $memName, "resource-tag", $rTag );
		if ( $out->results_status() eq "failed" ) {
			print( "Error : " . $out->results_reason() . "\n" );
			exit(-2);
		}
		print "Added member " . $memName . " to Resourcepool " . $rpName . "\n";
	} elsif ( $subCommand eq "del" ) { 
		# Remove a  member from an existing resource pool
		if ( $args < 7 ) {
			print_usage();
		}
		my $out =
		  $s->invoke( "resourcepool-remove-member", "resourcepool-name-or-id",
			$rpName, "member-name-or-id", $memName );
		if ( $out->results_status() eq "failed" ) {
			print( "Error : " . $out->results_reason() . "\n" );
			exit(-2);
		}
		print "Removed member " 
		  . $memName
		  . " from Resourcepool "
		  . $rpName . "\n";
	} else {
		print_usage();
	}
} else {
	print_usage();
}

