#! /usr/bin/env python
# encoding: utf-8
'''
Created on Apr 2, 2015

@author: Sebastian Wiesinger <sebastian@noris.net>
'''
import sys
import types
import argparse
try:
    import netsnmp
except ImportError:
    print 'ERROR: Python netsnmp module needed'
    sys.exit(3)


OID = {
    'dot3adAggPortSelectedAggID': '.1.2.840.10006.300.43.1.2.1.1.12',
    'ifDescr': '.1.3.6.1.2.1.2.2.1.2',
    'dot3adAggPortActorOperState': '.1.2.840.10006.300.43.1.2.1.1.21',
}

# LacpState ::= TEXTUAL-CONVENTION
#     STATUS      current
#     DESCRIPTION
#         "The Actor and Partner State values from the LACPDU."
#     SYNTAX      BITS {
#                 lacpActivity(0),
#                 lacpTimeout(1),
#                 aggregation(2),
#                 synchronization(3),
#                 collecting(4),
#                 distributing(5),
#                 defaulted(6),
#                 expired(7)
#                 }
LACP_STATE = {
    0: 'lacpActivity',
    1: 'lacpTimeout',
    2: 'aggregation',
    3: 'synchronization',
    4: 'collecting',
    5: 'distributing',
    6: 'defaulted',
    7: 'expired',
}

NAGIOS_STATE = {
    0: 'OK',
    1: 'Warning',
    2: 'Critical',
    3: 'Unknown'
}


def sget(self, oid):
    return self.get(netsnmp.VarList(netsnmp.Varbind(oid)))


def swalk(self, oid):
    return self.walk(netsnmp.VarList(netsnmp.Varbind(oid)))


class NagiosThreshold(object):
    def __init__(self, tstring):
        self.tstring = tstring

    @property
    def tstring(self):
        return self._build(self.start, self.end, self.invert)

    @tstring.setter
    def tstring(self, string):
        self.start, self.end, self.invert = self._parse(string)

    @classmethod
    def _parse(cls, string):
        invert = False
        if string is None:
            start = float('-inf')
            end = float('inf')
        else:
            if string[0] == '@':
                invert = True
                string = string[1:]
            if ':' in string:
                one, two = string.split(':')
                if one == '~':
                    start = float('-inf')
                else:
                    start = float(one)
                if len(two):
                    end = float(two)
                else:
                    end = float('inf')
            else:
                start = 0
                end = float(string)
        assert(start <= end)
        return start, end, invert

    @classmethod
    def _build(cls, start, end, invert):
        assert(start <= end)
        string = ''
        if invert:
            string += '@'
        if start == float('-inf'):
            if end == float('inf'):
                return None
            string += '~:'
        elif start == 0:
            pass
        else:
            string += str(start)
            string += ':'
        if end < float('inf'):
            string += str(end)
        return string

    def match(self, value):
        if value < self.start:
            return False ^ self.invert
        if value > self.end:
            return False ^ self.invert
        return True ^ self.invert


class Port(object):
    def __init__(self, snmp, ifindex):
        self.snmp = snmp
        self.ifindex = ifindex
        self.name = snmp.sget('%s.%s' % (OID['ifDescr'], ifindex))[0]
        memberstate = snmp.sget('%s.%s' % (OID['dot3adAggPortActorOperState'], ifindex))[0]
        try:
            self.memberstate = [int(x) for x in list((format(ord(memberstate), 'b')))]
        except TypeError:
            self.memberstate = None

    def __str__(self):
        return str(self.ifindex)

    def __repr__(self):
        return '<Port %s: %s>' % (self.ifindex, self.name)


def main():
    rcode = 0

    p = argparse.ArgumentParser(
        description='Check LACP interface status via SNMP IEEE8023-LAG-MIB',
        epilog="""
The number used for warning/critical thresholds is the number of member links that are NOT
syncronized in LACP. This means they cannot be used to carry traffic.

For example: -w 0 -c 1 means that you will get a warning when one link is down and a critical state
when more than one link is down.

For questions ask <sebastian@noris.net>"""
    )
    p.add_argument('--community', '-C', help='SNMP community')
    p.add_argument('--hostname', '-H', help='Hostname to query')
    p.add_argument('--timeout', '-t', help='SNMP timeout in seconds', type=int, default=5)
    p.add_argument('--retries', '-e', help='SNMP retries', type=int, default=3)
    p.add_argument('-w', dest='threshwarn', metavar='THRESHOLD',
                   help='Nagios warning threshold', default='0')
    p.add_argument('-c', dest='threshcrit', metavar='THRESHOLD',
                   help='Nagios critical threshold', default='0')
    p.add_argument('--verbose', '-v', action='count',
                   help='Increase verbosity (can be used more than once)')
    p.add_argument('ifindex', help='SNMP ifIndex of the LACP interface to check', type=int)
    args = p.parse_args()

    warn = NagiosThreshold(args.threshwarn)
    crit = NagiosThreshold(args.threshcrit)

    snmp = netsnmp.Session(Version=2, DestHost=args.hostname, Community=args.community,
                           Timeout=args.timeout * 100000, Retries=args.retries)
    snmp.sget = types.MethodType(sget, snmp)
    snmp.swalk = types.MethodType(swalk, snmp)

    aggport = Port(snmp, args.ifindex)
    aggports = netsnmp.VarList(netsnmp.Varbind(OID['dot3adAggPortSelectedAggID']))
    snmp.walk(aggports)
    memberports = [Port(snmp, p.tag.split('.')[-1])
                   for p in aggports if str(p.val) == str(aggport.ifindex)]

    syncmembers = []
    nsyncmembers = []
    if args.verbose >= 2:
        print 'Channel %s (ifIndex %s):' % (aggport.name, aggport.ifindex)
    for p in memberports:
        if p.memberstate[3]:
            syncmembers.append(p)
        else:
            nsyncmembers.append(p)
        state = p.memberstate
        if args.verbose >= 2:
            print ' Member %s (ifIndex %s) LACP Flags: %s' % (p.name, p.ifindex,
                ', '.join(['%s(%s)' % (LACP_STATE[i], i) for i, j in enumerate(state) if j]))

    if args.verbose < 1:
        rstring = '%s: %s/%s Sync/Down' % (aggport.name, len(syncmembers), len(nsyncmembers))
    else:
        rstring = '%s: Sync(%s): %s | Down(%s): %s' % (aggport.name, len(syncmembers),
                                                       ', '.join([p.name for p in syncmembers]),
                                                       len(nsyncmembers),
                                                       ', '.join([p.name for p in nsyncmembers]))
    if not crit.match(len(nsyncmembers)):
        rcode = 2
    elif not warn.match(len(nsyncmembers)):
        rcode = 1

    print rstring
    return rcode

if __name__ == '__main__':
    sys.exit(main())
