package noris::SMS;

use 5.008;
use strict;
use warnings;

# $Id: SMS.pm,v 1.1 2004/08/20 13:37:04 fany Exp $

=head1 NAME

noris::SMS - versende SMs ber HTTP(S)

=head1 SYNOPSE

    use noris::SMS 'normalize_numbers';

    my $sms = new noris::SMS orig => 'fany';

    my $sent = $sms->send( data => 'Das ist der Rand von Ostermundigen.' );

=head1 BESCHREIBUNG

Dieses Modul soll einen mglichst einfachen Versand von SMs (Short Messages)
ber Anbieter, die eine HTTP(S)-Schnittstelle anbieten, bei der die SMs ber
POST-Requests eingeliefert werden knnen, ermglichen.
Standardmig ist als Anbieter L<End2End Mobile|http://www.end2endmobile.com/>
eingestellt, ebenso sind die hier bentigten Zugangsdaten bereits enthalten.

=cut

use Carp qw(croak);
use Encode qw(encode);
use HTTP::Request::Common qw(POST);
use LWP::UserAgent ();
use noris::Encode::OpenITGSM0338;
use Encode::GSM0338;
use Digest::MD5 qw(md5_hex);
use POSIX qw(ceil);

=head1 KONSTRUKTOR

C<-E<gt>new> erwartet (optional) eine Liste von Name-/Wert-Paaren, insbesondere
folgenden:

=over 4

=item debug

Wird die debug Variable auf L<1> angegeben, gibt L</->send> zu
Debugging-Zwecken Meldungen (auf das aktuelle L<select()ierte|perlfunc/select>
FileHandle) aus.

=item test

Damit keine SMs versendet werden, muss  der Parameter L<test> auf L<1> gesetzt
werden.

=item user

Benutzername beim SMS-Anbieter (vgl. L</params>)

=item max_sms

Wieviele SMs sollen pro Aufruf von L<-E<gt>send> maximal versandt werden?
(Default: 10)

=item re_ok

regulrer Ausdruck, auf den die vom POST-Request zurckgelieferte Seite (genau
dann) matcht, wenn der Versand erfolgreich war (Default: C</^+/>)

=item params

Welche Parameter sollem im POST-Reqeust an den SMS-Anbieter bergeben werden?
(Default: L<user>, L<pass>, L<orig>, L<dest>, L<data>, L<maxparts>, L<enc>)
Mehrere Parameternamen knnen als Array-Referenz bergeben werden.

=item pass

Passwort zur Authentifikation beim SMS-Anbieter (vgl. L</params>)

=item orig

fr die SMs zu verwendende Absenderkennung
(Ziffernfolge oder maximal elf alphanumerische Zeichen; Default: "noris.net")

=item dest

an welche Zielrnufnummer(n) soll(en) die SM(s) versandt werden?
Werden mehrere Zielrufnummern (als Array-Referenz) angegeben, so erfolgt der
Versand an alle Zielrnufnummern.

=item data

zu versendende(r) Text(e).
Mehrere Texte knnen als Array-Referenz bergeben werden; sie werden in jedem
Fall als getrennte SMs versandt.

=item enc

Das Encoding das beim bertragen der Daten an den Anbieter verwendet werden
soll.
(Default: gsm)

=item maxparts

Obergrenze der SMs, die das End2End-Gateway fr grere Nachrichten ( E<gt> 160
 Zeichen ) erzeugen soll.
(Default: 10)

=item timeout

Timeout fr die HTTP(S)-Requests, der an L<LWP::UserAgent> bergeben wird

=item urls

URL(s), an die die POST-Requests abgesetzt werden sollen.
Bei der Angabe mehrerer URLs werden diese nacheinander durchprobiert, bis eine
positive Rckmeldung erfolgt.

=back

Weitere Parameter knnen jederzeit bergeben werden, werden aber ignoriert,
werden aber, auer im Zusammenhang mit L</params>, ignoriert.

=cut

sub new {
    croak "->new() called with odd number of arguments." unless @_ & 1;
    my $self;
    if ( ref( my $package = shift ) ) {
        croak "->new() called with foreign object."
          unless $package->isa(__PACKAGE__);
        $self = bless {%$package}, ref $package;
    }
    else {
        $self = bless {
            debug     => 0,
            user      => 'noris',
            enc       => 'gsm',
            max_sms   => 10,
            params    => [qw(user debug orig pass maxparts enc dest data)],
            pass      => 'fd32it56',
            orig      => 'noris.net',
            dest      => [],
            data      => [],
            re_ok     => qr/^REQUEST OK.*/,
            sent      => 0,
            maxparts  => 10,
            timeout   => undef,
            urls      => ["https://sms.openit.de/put.php"],
        }, $package;
    }

    # Parameter die new() bergeben wrden werden in die $self Conf bernommen
    for ( my $i = 0 ; $i < $#_ ; $i += 2 ) {
        if ( ref $self->{ $_[$i] } ) {
            @{ $self->{ $_[$i] } } =
              ref $_[ $i + 1 ] ? @{ $_[ $i + 1 ] } : $_[ $i + 1 ];
        }
        else { $self->{ $_[$i] } = $_[ $i + 1 ] }
    }
    $self;
}

=head1 METHODEN

Zugriff auf alle Parameter ist ber lvalue-sub-Routinen mglich, z. B. kann
eine neue C<orig> so gesetzt werden:

	$object->orig = 'noris.net';

Der Versand von SMs erfolgt durch Aufruf der Methode C<-E<gt>send>, die ein
neues noris::SMS-Objekt zurckliefert; auerdem knnen hier optional dieselben
Parameter bergeben werden wie bei L<-E<gt>new|/KONSTRUKTOR>, die dann in das
neu erzeugte Objekt bernommen werden.

Ging beim Versand etwas schief, liefert C<-E<gt>errstr> eine Fehlermeldung
zurck, ansonsten L<undef|perlfunc/undef>.

Mit C<-E<gt>sent> kann abgefragt werden, wie viele SMs erfolgreich versandt
wurden.

C<-E<gt>todo> liefert eine Referenz auf eine Liste von noch abzuarbeitenden
Parameter-Kombinationen.

Mit C<-E<gt>retry> kann ein erneuter Versuch gestartet werden, diese noch
abzuarbeiten.

=cut

sub send {
    my $self = shift;
    croak "->send() called with foreign object." unless $self->isa(__PACKAGE__);
    $self = $self->new(@_);
    my @params = @{ $self->params };
    $self->{todo} = [
        map {
            my %param = @$_;
            $param{data}  = encode( 'openit_gsm0338', $param{data} );
            $param{debug} = '1' if $self->{test};
            \%param;
        }
        $self->_combine(@params)
    ];
    $self->retry;
}

sub _number_of_sms {
    my ($self, %param) = @_;

    my $n;
    if ( (my $length = length $param{data}) <= 160 ) {
        $n = 1;
    }
    else {
        $n = 2;
        $length -= 305;
        $n += ceil($length/152) if $length > 0;
    }
    $n = $param{maxparts} if $n > $param{maxparts};

    print "No of SMS $n\n" if $self->debug;

    $n;
}

sub retry {
    my $self = shift;
    croak "->send() called with foreign object." unless $self->isa(__PACKAGE__);
    croak "->retry() does not accept arguments." if @_;
    my $ua =
      new LWP::UserAgent $self->timeout ? ( timeout => $self->{timeout} ) : ()
      or return $self->error('cannot create LWP::UserAgent object');
  COMBINATION: while ( @{ $self->todo } ) {
        my %todo = %{ $self->todo->[0] };

        my $number_of_sms = $self->_number_of_sms(%todo);

        if ( defined $self->max_sms
            && $self->sent + $number_of_sms > $self->max_sms )
        {
            print 'Limit of '
              . $self->max_sms . ' SM'
              . ( $self->max_sms != 1 && 's' )
              . " would be exhausted.\n"
              if $self->debug;
            last COMBINATION;
        }
        my @errors;
        for my $url ( @{ $self->urls } ) {

            my $post_object = HTTP::Request::Common::POST( $url, \%todo );
            print "I am sending the following Request:\n" . $post_object->as_string()
              if $self->debug;
            my $response = $ua->request( $post_object, () )
              or return $self->error( "unknown error POST $url\n",
                pp($post_object) );
            my $error;
            if ( $response->is_success ) {
                my $content = $response->content;
                if ( $content =~ $self->re_ok ) {
                    $self->sent += $number_of_sms;
                    next COMBINATION;
                }
                $error = "request returned: $content";
            }
            else {
                $error =
                  'error during POST request: ' . $response->status_line;
            }
            push @errors, $error;
            print "$error\n" if defined $error && $self->debug;
        }
        return $self->error( join '; ', @errors ) if @errors;
    }
    continue { shift @{ $self->todo } }
    $self->error(undef);
    $self;
}

# Diese Funktion erstellt eine Liste aller Kombinationen der
# Werte der Attribute, die als params gelten, und somit eine
# TODO-Liste fr den Aufruf des Gateways.
sub _combine {
    my $self = shift;
    croak "->error() called with foreign object."
      unless $self->isa(__PACKAGE__);
    defined( my $key = shift ) or return;
    my @combinations = @_ ? $self->_combine(@_) : [];
    my $values = $self->$key;
    map {
        my $value = $_;
        map [ $key, $value, @$_ ], @combinations;
    } ref $values ? @$values : $values;
}

sub error {
    my $self = shift;
    croak "->error() called with foreign object."
      unless $self->isa(__PACKAGE__);
    if (@_) {

        # Aufruf mit (undef) lscht errstr:
        if ( @_ == 1 && !defined $_[0] ) { $self->{errstr} = undef }
        else {

            # Ansonsten wird das erste Element als Fehlermeldung betrachtet...
            $self->{errstr} = shift;

            # ... an die ggf. ein Dump aller weiteren bergebenen
            # Datenstrukturen angehngt wird:
            $self->{errstr} .= pp(@_) if @_;
        }
    }
    $self;
}

sub pp {
    local $^W;
    require Data::Dump and Data::Dump->import('pp');
    goto &pp;
}

sub AUTOLOAD : lvalue {
    my $self = shift;
    my ($function) = our $AUTOLOAD =~ /([^:]+)$/;
    croak "->$function() called with foreign object."
      unless $self->isa(__PACKAGE__);
    $self->{$function};
}

=head1 OVERLOADING

In einem Wahrheitskontext liefert jedes noris::SMS-Objekt wahr, falls sich kein
Fehler ereignet hat (entspricht C<!defined($object-E<gt>errstr)>), als String
die auch mit C<-E<gt>errstr> abfragbare Fehlermeldung und in einem numerischen
Kontext die Anzahl der erfolgreich versendeten SMs (entspricht
C<$object-E<gt>sent>).

=cut

use overload
  'bool' => sub { !defined shift->errstr },
  '0+'   => sub { shift->sent },
  '""'   => sub { shift->errstr };

=head1 ZUSATZFUNKTIONEN

=head2 normalize_number

C<normalize_number> versucht, bergebene Rufnummern nach folgendem Schema
in ein einheitliches Format zu bringen:

=over 4

=item *

Beginnt die Rufnummer mit mindestens zwei Nullen, so werden diese verworfen.

=item *

Beginnt die Rufnummer mit nur einer Null, so wird diese durch "49" ersetzt.

=item *

Sodann werden alle nicht-numerischen Zeichen entfernt.

=item *

Schlielich wird ein Pluszeichen vorangestellt.

=back

Beim Aufruf in einem Void-Kontext werden dabei die bergebenen Parameter direkt
verndert, andernfalls gibt die Funktion eine Liste der vereinheitlichten
Rufnummern zurck.

Die Funktion wird nur auf Anfrage exportiert.

=cut

sub normalize_numbers {
    local our @numbers;
    if ( defined wantarray ) { @numbers = @_ }
    else { *numbers = \@_ }
    for (@numbers) {
        s/^00// or s/^0/49/;
        y/0-9//cd;
        $_ = "+$_";
    }
    @numbers;
}

use base 'Exporter';
our @EXPORT_OK = 'normalize_numbers';

=head1 AUTOR

 Martin H. Sluka <fany@noris.net>
 Stelios Gikas <sgikas@noris.net>
 fr die noris network AG

=cut

1;
