#!/usr/local/bin/perl -w

package noris::NetSaint;

=head1 NAME

noris::NetSaint

=head1 BESCHREIBUNG

Library fuer NetSaint-Plugins,
um diesen ganzen Status- und Timeout-Krempel einheitlich zu erledigen.

=head1 DEBUGGING

In der Umgebungsvariable NORIS_NETSAINT_DEBUGLOG kann ein Dateiname angegeben
werden.
Sofern diese Datei entsprechend geffnet werden kann, werden dort dann alle
Methodenaufrufe dokumentiert.

=head1 AUTOR

Martin H. Sluka E<lt>sluka@noris.netE<gt>
fr die noris network AG

=head1 TICKET

RT#38488

=cut

use 5.006;
use strict;
use warnings;

use noris::NetSaint::Status;

use constant CUT_MESSAGE_2 => 256;

our $STATUS = new noris::NetSaint::Status 'Unknown';

sub new {
    my $self = bless { time => time }, shift;
    $self->{setSIG}{ALRM} = $SIG{ALRM} =
      sub { die "Timed out after $ENV{DEFAULT_SOCKET_TIMEOUT} seconds.\n" }
      unless $SIG{ALRM};
    $self->{setSIG}{__DIE__} = $SIG{__DIE__} = sub {
        return if $^S;
        $self->add_message(@_);
        exit( $STATUS = new noris::NetSaint::Status 'Critical' );
      }
      unless $SIG{__DIE__};

    #   $self->{setSIG}{PIPE} = $SIG{PIPE} = 'IGNORE' unless $SIG{PIPE};
    alarm $ENV{DEFAULT_SOCKET_TIMEOUT} if $ENV{DEFAULT_SOCKET_TIMEOUT};
    $STATUS = $self->{status} = new noris::NetSaint::Status;
    $self;
}

sub message {
    my $self = shift;
    if (@_) {
        @{ $self->{messages} } = ();
        return unless defined $_[0];
        $self->add_message($_) for @_;
    }
    else {
        map $_->[0] . ( $_->[1] != 1 && " ($_->[1]x)" ),
          @{ $self->{messages} ||= [] };
    }
}

use constant { I_MESSAGE => 0, I_COUNT => 1, I_STATUS => 2 };

sub add_message {
    my $self = shift;
    return unless @_;

    my $status;

    # Workaround: gleich numerische Werte speichern, weil
    # noris::netSaint::Status-Objekte bei DESTROY bereits ber den Jordan
    # gegangen sind.
    $status = sprintf '%d', shift
      if ref $_[0] && $_[0]->isa('noris::NetSaint::Status');

    my $messages = $self->{messages} ||= [];
    for (@_) {
        if ( @$messages && $_ eq $messages->[-1][I_MESSAGE]
            and !defined $status
            || !defined $messages->[-1][I_STATUS]
            || $status == $messages->[-1][I_STATUS] )
        {
            $messages->[-1][I_COUNT]++;
            $messages->[-1][I_STATUS] = $status
              unless defined $messages->[-1][I_STATUS];
        }
        else {
            my @new_message;
            @new_message[ I_MESSAGE, I_COUNT, I_STATUS ] = ( $_, 1, $status );
            push @$messages, \@new_message;
        }
    }
}

sub status {
    my $self = shift;
    if (@_) {
        $STATUS = $self->{status} = new noris::NetSaint::Status shift;
        ++$self->{set_status};
    }
    $self->{status};
}

sub set_status {
    my $self = shift;
    $self->{set_status};
}

sub atleast {
    my $self   = shift;
    my $status = new noris::NetSaint::Status shift;
    $STATUS = $self->status($status)
      if !$self->set_status || $self->status < $status;
}

sub update {
    my $self = shift;
    $self->atleast( my $status = noris::NetSaint::Status->new(shift) );
    $self->add_message( $status, @_ );
    $self->status;
}

sub DESTROY {
    my $self = shift;
    my $last_min_status;
    {
        if ( $self->message ) {
            ( my $message = join '; ', $self->message ) =~
              y/`~$&|'"<>//d;    # vgl. RT#74847, RT#165792, RT#354399

            # Nicht \s+, wegen RT#360673
            $message =~ s/[ \n\r\t]+/ /g;    # vgl. RT#109321
            debug("Message: $message") unless defined $last_min_status;

            if ( length $message > CUT_MESSAGE_2 ) {

                my $min_status;
                defined $_->[I_STATUS]
                  and !defined $min_status || $min_status > $_->[I_STATUS]
                  and $min_status = $_->[I_STATUS]
                  for @{ $self->{messages} };
                $min_status = $last_min_status + 1
                  if defined $last_min_status
                  && $last_min_status == $min_status;
                $last_min_status = $min_status;

                if ( defined $min_status ) {
                    my @messages = grep !defined $_->[I_STATUS]
                      || $_->[I_STATUS] > $min_status, @{ $self->{messages} };

                    if ( @messages && @{ $self->{messages} } > @messages ) {
                        my $min_status =
                          noris::NetSaint::Status->new($min_status);
                        $self->{messages} = \@messages;
                        $self->add_message(
                            $min_status,
                            '['
                              . (
                                my $removed =
                                    "Removed messages with status $min_status"
                                  . ( defined $last_min_status && ' and below' )
                                  . '.'
                              )
                              . ']'
                        );
                        debug($removed);
                        redo;
                    }
                }
                $message =
                    substr( $message, 0, CUT_MESSAGE_2 - 16 ) . ' ['
                  . ( length($message) - ( CUT_MESSAGE_2 - 16 ) )
                  . ' chars cut]';
            }
            print "$message\n";
        }
        else {
            my $runtime = time - $self->{time};
            printf "runtime: %d second%s, status: %s.\n", $runtime,
              $runtime != 1 && 's', $STATUS;
        }
    }
    defined $SIG{$_} && $self->{setSIG}{$_}
      and $SIG{$_} = 'DEFAULT'
      for keys %{ $self->{setSIG} };
}

END { $? = $STATUS }

my $debug_fh;

sub my_pp {
    my $args = pp(@_);
    if ( $args =~ s/\n\s*/ /g ) {
        $args =~ s/^\(\s+/ /;
        $args =~ s/, \)\z/ )/;
    }
    if   ( $args =~ /^\(/ ) { $args }
    else                    { "($args)" }
}

sub debug($) { }

if ( defined( $ENV{NORIS_NETSAINT_DEBUGLOG} ) and open $debug_fh,
    '>>', $ENV{NORIS_NETSAINT_DEBUGLOG} )
{
    {
        no warnings 'redefine';
        *debug = sub ($) { print $debug_fh localtime() . " [$$] @_\n" };
    }

    require Data::Dump    and Data::Dump->import('pp');
    require Hook::LexWrap and Hook::LexWrap->import('wrap');

    debug( "$0 started with arguments " . my_pp(@ARGV) );

    for my $sub (qw(add_message atleast message new set_status status update)) {
        wrap(
            $sub => pre => sub {
                debug( "->$sub" . my_pp( @_[ 1 .. $#_ - 1 ] ) . ' called.' );
            },
        );
    }
}

END { debug("$0 exiting with status $?.") }

1
