package CapMan::Tools;

use strict;
use warnings;

use Exporter;
use Data::Dumper;
use SNMP ();

use CapMan::Config qw(get_rrdpath get_host_config host_alias_to_address :rrdcreate);

=head USAGE

  use CapMan::Tools qw(do_snmp_walk do_versioned_snmp_walk write_to_rrd);

  do_snmp_walk ($hostalias, @oids);
  do_versioned_snmp_walk ($hostname, $snmpversion, @oids);
  write_to_rrd ($hostalias, $service, $filename, \&rrd_create_handler, @data);

=cut

@CapMan::Tools::EXPORT_OK = qw(do_snmp_walk do_versioned_snmp_walk write_to_rrd create_rrdfile get_n_colors);
@CapMan::Tools::ISA = qw(Exporter);

return (1);

sub do_snmp_walk
{
	confess ("Wrong number of arguments") if (@_ < 3);
	my $customer = shift;
	my $hostalias = shift;
	my $hostname  = host_alias_to_address ($customer, $hostalias);
	my @oids = @_;
	my @vers = get_host_config ($customer, $hostalias, 'snmpversion');
	my $comm = get_host_config ($customer, $hostalias, 'community');

	my $data = undef;

	if (!$hostname)
	{
		print STDERR "Unable to get hostname for hostalias ``$hostalias''\n";
		return;
	}
	elsif ($::DEBUG)
	{
		print STDOUT 'DEBUG: Doing SNMP ', join (', ', @vers),
		" with $hostalias ($hostname)\n";
	}

	$comm ||= 'public';

	for (@vers)
	{
		my $v = $_;

		$data = do_versioned_snmp_walk ($hostname, $comm, $v, @oids);

		last if (defined ($data) and $data);
	}

	return ($data);
}

sub do_versioned_snmp_walk ($$$@)
{
	my $hostname = shift;
	my $comm = shift;
	my $ver  = shift;
	my @oids = @_;
	my @data = ();
	my $vars;
	my $success = 0;

	my $sess = new SNMP::Session
	(
		'DestHost'  => $hostname,
		'Community' => $comm,
		'Version'   => $ver
	);

	if (!defined ($sess))
	{
		return (undef);
	}

	if ($ver == 1)
	{
		for (@oids)
		{
			my $fh;
			my @blah = ();
			my $cmd = "snmpwalk -v 1 -Oqn -c '$comm' '$hostname' '$_'";

			print STDERR "Trying: $cmd\n" if ($::DEBUG);

			if (!open ($fh, "$cmd |"))
			{
				warn ("open snmpwalk: $!");
				return;
			}

			while (<$fh>)
			{
				chomp;

				if (m/^(\S+)\.(\d+) (.+)$/)
				{
					my $oid  = $1;
					my $id   = $2;
					my $val  = $3;
					my $type = 'INTEGER';

					if ($val =~ m/^"([^"]*)"$/)
					{
						$val = $1;
						$type = 'STRING';
					}

					push (@blah, [$oid, $id, $val, $type]);
				}
				elsif ($::DEBUG)
				{
					print STDOUT "SNMPv1: No match: $_\n";
				}
			}

			if (@blah)
			{
				push (@data, \@blah);
				$success++;
			}
			else
			{
				push (@data, undef);
			}
		}
	}
	else
	{
		for (@oids)
		{
			my $oid = $_;
			my @tmp;
			
			$vars = new SNMP::VarList ([$oid]);
			@tmp = $sess->bulkwalk (0, 10, $vars);

			if (defined ($tmp[0]))
			{
				push (@data, $tmp[0]);
				$success++;
			}
			else
			{
				push (@data, undef);
			}
		}
	}

	return (undef) if (!$success);
#	if ($::DEBUG)
#	{
#		require Data::Dumper;
#		print Data::Dumper->Dump ([\@data], ['data']);
#	}
	return (\@data);
}

sub write_to_rrd
{
	confess ("Wrong number of arguments") if (@_ < 5);

	my $customer = shift;
	my $hostalias = shift;
	my $service = shift;
	my $filename = shift;
	my @data = @_;
	my $rrdpath = get_rrdpath ();

	$hostalias =~ s#/#_#g;

	$customer ||= '_default_';
	$customer =~ s#/#_#g;

	if (!-d "$rrdpath/$customer")
	{
		mkdir ("$rrdpath/$customer") or die ("mkdir: $!");
	}
	if (!-d "$rrdpath/$customer/$hostalias")
	{
		mkdir ("$rrdpath/$customer/$hostalias") or die ("mkdir: $!");
	}

	if ($filename ne '')
	{
		$filename =~ s/\W+/_/g;
		$filename =~ s/__+/_/g;
		$filename = "$rrdpath/$customer/$hostalias/$service-$filename.rrd";
	}
	else
	{
		$filename = "$rrdpath/$customer/$hostalias/$service.rrd";
	}

	if (!-e $filename)
	{
		# This needs to be done at runtime..
		require CapMan::Services;
		CapMan::Services->import ('run_create_callback');

		run_create_callback ($service, $filename);
	}

	if (!-e $filename)
	{
		return;
	}

	print STDOUT "DEBUG: RRDs::update ($filename, N:", join (':', @data), ")\n" if ($::DEBUG);
	
	RRDs::update ($filename, 'N:' . join (':', @data));

	if (RRDs::error ())
	{
		print STDERR "RRDs::update ($filename): " . RRDs::error () . "\n";
	}
}

sub create_rrdfile
{
	my $file = shift;
	my @ds = @_;

	RRDs::create ($file, '-s', $Step,
		@ds,
		"RRA:AVERAGE:0.5:1:$DailyRows",
		"RRA:AVERAGE:0.5:6:$WeeklyRows",
		"RRA:AVERAGE:0.5:24:$MonthlyRows",
		"RRA:AVERAGE:0.5:288:$YearlyRows"
	);

	if (RRDs::error ())
	{
		print STDERR "RRDs::create ($file, ...): " . RRDs::error () . "\n";
	}
}

sub get_n_colors
{
	my $num = shift;
	my @ret = ();

	for (my $i = 0; $i < $num; $i++)
	{
		my $pos = 6 * $i / $num;
		my $n = int ($pos);
		my $p = $pos - $n;
		my $q = 1 - $p;

		my $red   = 0;
		my $green = 0;
		my $blue  = 0;

		if ($n == 0)
		{
			$red  = 255;
			$blue = 255 * $p;
		}
		elsif ($n == 1)
		{
			$red  = 255 * $q;
			$blue = 255;
		}
		elsif ($n == 2)
		{
			$green = 255 * $p;
			$blue  = 255;
		}
		elsif ($n == 3)
		{
			$green = 255;
			$blue  = 255 * $q;
		}
		elsif ($n == 4)
		{
			$red   = 255 * $p;
			$green = 255;
		}
		elsif ($n == 5)
		{
			$red   = 255;
			$green = 255 * $q;
		}
		else { die; }

		push (@ret, sprintf ("%02x%02x%02x", $red, $green, $blue));
	}

	return (@ret);
}
