package CapMan::Plugins::Windows;

use strict;
use warnings;

=head1 NAME

CapMan::Plugins::Windows - CapMan plugin for Microsoft Windows specific stuff

=head1 DESCRIPTION

This plugin collects data on CPU-usage and logged in users, provided by Windows
in extra MIBs.

=cut

use RRDs;

use CapMan::Config qw(:rrdcreate get_rrdpath);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd create_rrdfile);

our $GraphDefCPU = ['-v', 'Percent', '-u', '100',
	'DEF:u={filename}:used:AVERAGE',
	'DEF:i={filename}:idle:AVERAGE',
	'HRULE:100#FF0000',
	'AREA:u#FF0000:Used',
	'GPRINT:u:AVERAGE:%4.1lf%% Average,',
	'GPRINT:u:MAX:%4.1lf%% Max,',
	'GPRINT:u:LAST:%4.1lf%% Last'];

our $GraphDefUsers = ['-v', 'Users',
	'DEF:u={filename}:users:AVERAGE',
	'AREA:u#00FF00:Users',
	'GPRINT:u:AVERAGE:%4.1lf Average,',
	'GPRINT:u:MAX:%4.1lf Max,',
	'GPRINT:u:LAST:%4.1lf Last'];

=head1 SERVICES PROVIDED

=over 4

=item windows_cpu

Handler for CPU-usage.

=item windows_users

Number of users currently logged in.

=back

=cut

register_input_handler ('windows_cpu',   '.1.3.6.1.2.1.25.3.3.1.2', \&windows_cpu_handler);
register_input_handler ('windows_users', '.1.3.6.1.2.1.25.1.5',     \&windows_users_handler);

register_create_handler ('users',  \&create_users_handler);
register_create_handler ('wincpu', \&create_windows_cpu_handler);

register_graph_handler ('users',  $GraphDefUsers);
register_graph_handler ('wincpu', $GraphDefCPU, \&print_meta_graph_wincpu);

return (1);

sub windows_cpu_handler ($$$)
{
	my $host = shift;
	my $srv = shift;
	my $data = shift;

	my @data = ();
	my $i;

	for (@$data)
	{
		next unless ($_->[0] =~ m/1.3.6.1.2.1.25.3.3.1.2|hrProcessorLoad/);
		next if ($_->[2] =~ m/\D/);
		next unless ($_->[2] >= 0 and $_->[2] <= 100);
		
		push (@data, $_->[2]);
	}

	for ($i = 0; $i < scalar (@data); $i++)
	{
		write_to_rrd ($host, 'wincpu', $i, $data[$i], 100 - $data[$i]);
	}
}

sub create_windows_cpu_handler ($)
{
	my $file = shift;

	create_rrdfile ($file,
		"DS:used:GAUGE:$HeartBeat:0:110",
		"DS:idle:GAUGE:$HeartBeat:0:110"
	);
}

sub windows_users_handler ($$$)
{
	my $host = shift;
	my $srv = shift;
	my $data = shift;

	my $val;

	if (!defined ($data->[0][2]))
	{
		return;
	}

	$val = $data->[0][2];

	write_to_rrd ($host, 'users', '', $val);
}

sub create_users_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:users:GAUGE:$HeartBeat:0:100");
}

sub print_meta_graph_wincpu
{
	my $host = shift;
	my $begin = shift;
	my $end = shift;
	my @files = grep { $_ ne 'cpu.rrd' } (@_);

	my @cmd = ('-', '-s', $begin, '-e', $end, '-a', 'PNG');

	my $num = 0;
	my $rrdpath = get_rrdpath ();

	my $cdef_used = 'CDEF:used=used0';
	my $cdef_idle = 'CDEF:idle=idle0';

	for (@files)
	{
		my $file = "$rrdpath/$host/$_";

		push (@cmd, qq(DEF:used$num=$file:used:AVERAGE));
		push (@cmd, qq(DEF:idle$num=$file:idle:AVERAGE));

		$num++;
	}

	for (my $i = 1; $i < $num; $i++)
	{
		$cdef_used .= ",used$i,+";
		$cdef_idle .= ",idle$i,+";
	}

	if ($num > 1)
	{
		$cdef_used .= ",$num,/";
		$cdef_idle .= ",$num,/";
	}

	push (@cmd,
		'-v', 'Percent', '-u', '100',
		'-t', $num > 1 ? "Combined CPU Usage on $host ($num CPUs)" : "CPU Usage on $host",
		$cdef_used, $cdef_idle,
		'HRULE:100#FF0000',
		'AREA:used#FF0000:Used',
		'GPRINT:used:AVERAGE:%4.1lf%% Average,',
		'GPRINT:used:MAX:%4.1lf%% Max,',
		'GPRINT:used:LAST:%4.1lf%% Last');

	RRDs::graph (@cmd);

	die (RRDs::error ()) if (RRDs::error ());
}

=head1 SEE ALSO

L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
