package CapMan::Plugins::TCPSessions;

use strict;
use warnings;

=head1 NAME

CapMan::Plugins::TCPSessions - CapMan plugin for querying the number of
TCP-sessions.

=head1 DESCRIPTION

This plugin queries the actively opened (outgoing), passively opened (incoming)
and actively closed TCP-sessions.

=cut

use CapMan::Config qw(:rrdcreate);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd create_rrdfile);

our $GraphDef = ['-v', 'Sessions/s', '-l', '0',
	'DEF:i={filename}:incoming:AVERAGE',
	'DEF:o={filename}:outgoing:AVERAGE',
	'DEF:c={filename}:closed:AVERAGE',
	'AREA:o#00FF00:Outgoing',
	'GPRINT:o:AVERAGE:%5.1lf Average,',
	'GPRINT:o:MAX:%5.1lf Max,',
	'GPRINT:o:LAST:%5.1lf Last\n',
	'STACK:i#0000FF:Incoming',
	'GPRINT:i:AVERAGE:%5.1lf Average,',
	'GPRINT:i:MAX:%5.1lf Max,',
	'GPRINT:i:LAST:%5.1lf Last\n',
	'LINE2:c#FF0000:Closed  ',
	'GPRINT:c:AVERAGE:%5.1lf Average,',
	'GPRINT:c:MAX:%5.1lf Max,',
	'GPRINT:c:LAST:%5.1lf Last'];

=head1 SERVICES PROVIDED

=over 4

=item tcp_sessions

Significant numbers concerning TCP-sessions.

=back

=cut

register_input_handler ('tcpsessions', '.1.3.6.1.2.1.6', \&tcp_sessions_handler);
register_create_handler ('tcpsessions', \&tcp_sessions_create_handler);
register_graph_handler ('tcpsessions', $GraphDef);

return (1);

sub tcp_sessions_handler ($$$)
{
	my $host = shift;
	my $srv = shift;
	my $data = shift;

	my %tt =
	(
		'.1.3.6.1.2.1.6.5'	=> 'tcpActiveOpens',
		'.1.3.6.1.2.1.6.6'	=> 'tcpPassiveOpens',
		'.1.3.6.1.2.1.6.8'	=> 'tcpEstabResets'
	);
	
	my $open_act = 0;
	my $open_psv = 0;
	my $close = 0;

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;

		$name = $tt{$name} if (defined ($tt{$name}));

		if ($name eq 'tcpActiveOpens')     { $open_act = $val; }
		elsif ($name eq 'tcpPassiveOpens') { $open_psv = $val; }
		elsif ($name eq 'tcpEstabResets')  { $close    = $val; }
	}

	write_to_rrd ($host, 'tcpsessions', '', $open_act, $open_psv, $close);
}

sub tcp_sessions_create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file,
		"DS:outgoing:COUNTER:$HeartBeat:0:U",
		"DS:incoming:COUNTER:$HeartBeat:0:U",
		"DS:closed:COUNTER:$HeartBeat:0:U"
	);
}

=head1 SEE ALSO

L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
