package CapMan::Plugins::SysStat;

use strict;
use warnings;
#Use of implicit split to @_ is deprecated at /home/octo/capman/lib/CapMan/Plugins/SysStat.pm line 192.
#no warnings qw(deprecated);

=head1 NAME

CapMan::Plugins::SysStat - CapMan plugin which queries detailed Linux
statistics.

=head1 DESCRIPTION

This plugin queries numberous statistics provided by the Linux kernel. To use
this plugin you have to install a script on the host being queried. The
following information is collected:

    CPU-Usage
    Disk-IO
    Swap- and Page-IO
    Interrupts
    Forkrate
    Contextswitches
    Memory-Information

=cut

use RRDs;

use CapMan::Config qw(:rrdcreate get_rrdpath);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd create_rrdfile);


our $GraphDefMeminfo = ['-b', '1024', '-l', '0', '-v', 'Bytes',
	'DEF:used={filename}:used:AVERAGE',
	'DEF:buffer={filename}:buffer:AVERAGE',
	'DEF:page={filename}:page:AVERAGE',
	'DEF:free={filename}:free:AVERAGE',
	'AREA:used#0000FF:Used        ',
	'GPRINT:used:AVERAGE:%5.1lf%sByte Average,',
	'GPRINT:used:MAX:%5.1lf%sByte Max,',
	'GPRINT:used:LAST:%5.1lf%sByte Last\n',
	'STACK:buffer#FFFF00:Buffer cache',
	'GPRINT:buffer:AVERAGE:%5.1lf%sByte Average,',
	'GPRINT:buffer:MAX:%5.1lf%sByte Max,',
	'GPRINT:buffer:LAST:%5.1lf%sByte Last\n',
	'STACK:page#FF0000:Page cache  ',
	'GPRINT:page:AVERAGE:%5.1lf%sByte Average,',
	'GPRINT:page:MAX:%5.1lf%sByte Max,',
	'GPRINT:page:LAST:%5.1lf%sByte Last\n',
	'STACK:free#00FF00:Free        ',
	'GPRINT:free:AVERAGE:%5.1lf%sByte Average,',
	'GPRINT:free:MAX:%5.1lf%sByte Max,',
	'GPRINT:free:LAST:%5.1lf%sByte Last'];

our $GraphDefCPU = ['-v', 'Percent', '-u', '100',
	'DEF:u={filename}:user:AVERAGE',
	'DEF:n={filename}:nice:AVERAGE',
	'DEF:s={filename}:syst:AVERAGE',
	'DEF:i={filename}:idle:AVERAGE',
	'HRULE:100#FF0000',
	'AREA:s#FF0000:System',
	'GPRINT:s:AVERAGE:%4.1lf%% Average,',
	'GPRINT:s:MAX:%4.1lf%% Max,',
	'GPRINT:s:LAST:%4.1lf%% Last\n',
	'STACK:u#0000FF:User  ',
	'GPRINT:u:AVERAGE:%4.1lf%% Average,',
	'GPRINT:u:MAX:%4.1lf%% Max,',
	'GPRINT:u:LAST:%4.1lf%% Last\n',
	'STACK:n#FFFF00:Nice  ',
	'GPRINT:n:AVERAGE:%4.1lf%% Average,',
	'GPRINT:n:MAX:%4.1lf%% Max,',
	'GPRINT:n:LAST:%4.1lf%% Last'];

our $GraphDefDiskIO = ['-v', 'Operations/Blocks',
	'DEF:ori={filename}:opts_read:AVERAGE',
	'DEF:owi={filename}:opts_wrte:AVERAGE',
	'DEF:bri={filename}:blks_read:AVERAGE',
	'DEF:bwi={filename}:blks_wrte:AVERAGE',
	'CDEF:oro=ori,-1,*',
	'CDEF:owo=owi,-1,*',
	'AREA:bwi#FF0000:Write',
	'GPRINT:bwi:AVERAGE:%6.1lf Blk/s Average,',
	'GPRINT:bwi:MAX:%6.1lf Blk/s Max,',
	'GPRINT:bwi:LAST:%6.1lf Blk/s Last\n',
	'AREA:owo#FF0000:     ',
	'GPRINT:owi:AVERAGE:%6.1lf Ops/s Average,',
	'GPRINT:owi:MAX:%6.1lf Ops/s Max,',
	'GPRINT:owi:LAST:%6.1lf Ops/s Last\n',
	'HRULE:0#000000',
	'LINE1:bri#0000FF:Read ',
	'GPRINT:bri:AVERAGE:%6.1lf Blk/s Average,',
	'GPRINT:bri:MAX:%6.1lf Blk/s Max,',
	'GPRINT:bri:LAST:%6.1lf Blk/s Last\n',
	'LINE1:oro#0000FF:     ',
	'GPRINT:ori:AVERAGE:%6.1lf Ops/s Average,',
	'GPRINT:ori:MAX:%6.1lf Ops/s Max,',
	'GPRINT:ori:LAST:%6.1lf Ops/s Last'];

our $GraphDefIO = ['-v', 'Pages/s',
	'DEF:o={filename}:outgoing:AVERAGE',
	'DEF:i={filename}:incoming:AVERAGE',
	'AREA:o#FF0000:Swapout',
	'GPRINT:o:AVERAGE:%6.1lf Average,',
	'GPRINT:o:MAX:%6.1lf Max,',
	'GPRINT:o:LAST:%6.1lf Last\n',
	'LINE1:i#0000FF:Swapin ',
	'GPRINT:i:AVERAGE:%6.1lf Average,',
	'GPRINT:i:MAX:%6.1lf Max,',
	'GPRINT:i:LAST:%6.1lf Last'];

our $GraphDefCntxt = ['-v', 'Contextswitches/s',
	'DEF:v={filename}:value:AVERAGE',
	'AREA:v#00FF00:Contextswitches/s',
	'GPRINT:v:AVERAGE:%5.1lf Average,',
	'GPRINT:v:MAX:%5.1lf Max,',
	'GPRINT:v:LAST:%5.1lf Last'];

our $GraphDefIntr = ['-v', 'Interrupts/s',
	'DEF:v={filename}:value:AVERAGE',
	'AREA:v#00FF00:Interrupts/s',
	'GPRINT:v:AVERAGE:%5.1lf Average,',
	'GPRINT:v:MAX:%5.1lf Max,',
	'GPRINT:v:LAST:%5.1lf Last'];

our $GraphDefFork = ['-v', 'forks/s',
	'DEF:v={filename}:value:AVERAGE',
	'AREA:v#00FF00:Forks/s',
	'GPRINT:v:AVERAGE:%5.1lf Average,',
	'GPRINT:v:MAX:%5.1lf Max,',
	'GPRINT:v:LAST:%5.1lf Last'];

our $GraphDefBind = ['-v', 'Requests/s',
	'DEF:raw_success={filename}:success:AVERAGE',
	'DEF:raw_referral={filename}:referral:AVERAGE',
	'DEF:raw_nxrrset={filename}:nxrrset:AVERAGE',
	'DEF:raw_nxdomain={filename}:nxdomain:AVERAGE',
	'DEF:raw_failure={filename}:failure:AVERAGE',
	'CDEF:failure=raw_success,raw_referral,raw_nxrrset,raw_nxdomain,raw_failure,+,+,+,+',
	'CDEF:nxdomain=raw_success,raw_referral,raw_nxrrset,raw_nxdomain,+,+,+',
	'CDEF:nxrrset=raw_success,raw_referral,raw_nxrrset,+,+',
	'CDEF:referral=raw_success,raw_referral,+',
	'CDEF:success=raw_success',
	'AREA:failure#FF0000:Failure   ',
	'GPRINT:raw_failure:AVERAGE:%5.1lf Average,',
	'GPRINT:raw_failure:MAX:%5.1lf Max,',
	'GPRINT:raw_failure:LAST:%5.1lf Last\l',
	'AREA:nxdomain#FF00FF:NX Domain ',
	'GPRINT:raw_nxdomain:AVERAGE:%5.1lf Average,',
	'GPRINT:raw_nxdomain:MAX:%5.1lf Max,',
	'GPRINT:raw_nxdomain:LAST:%5.1lf Last\l',
	'AREA:nxrrset#00FFFF:NX RRSet  ',
	'GPRINT:raw_nxrrset:AVERAGE:%5.1lf Average,',
	'GPRINT:raw_nxrrset:MAX:%5.1lf Max,',
	'GPRINT:raw_nxrrset:LAST:%5.1lf Last\l',
	'AREA:referral#0000FF:Referral  ',
	'GPRINT:raw_referral:AVERAGE:%5.1lf Average,',
	'GPRINT:raw_referral:MAX:%5.1lf Max,',
	'GPRINT:raw_referral:LAST:%5.1lf Last\l',
	'AREA:success#00A000:Success   ',
	'GPRINT:raw_success:AVERAGE:%5.1lf Average,',
	'GPRINT:raw_success:MAX:%5.1lf Max,',
	'GPRINT:raw_success:LAST:%5.1lf Last\l'];


=head1 SERVICES PROVIDED

=over 4

=item sysstat

Run the remote script and get all of the above information.

=back

=cut

register_input_handler ('sysstat', '.1.3.6.1.4.1.2021.50.101', \&sysstat_handler);

register_create_handler ('contextswitches', \&create_general_counter_handler);
register_create_handler ('cpu',             \&sysstat_create_cpu_handler);
register_create_handler ('disk_io',         \&sysstat_create_diskio_handler);
register_create_handler ('forkrate',        \&create_general_counter_handler);
register_create_handler ('interrupts',      \&create_general_counter_handler);
register_create_handler ('io',              \&sysstat_create_general_io_handler);
register_create_handler ('meminfo',         \&sysstat_create_mem_handler);
register_create_handler ('bind',            \&sysstat_create_bind_handler);

register_graph_handler ('contextswitches', $GraphDefCntxt);
register_graph_handler ('cpu',             $GraphDefCPU, \&print_meta_graph_cpu);
register_graph_handler ('disk_io',         $GraphDefDiskIO);
register_graph_handler ('forkrate',        $GraphDefFork);
register_graph_handler ('interrupts',      $GraphDefIntr);
register_graph_handler ('io',              $GraphDefIO);
register_graph_handler ('meminfo',         $GraphDefMeminfo);
register_graph_handler ('bind',            $GraphDefBind);

return (1);

sub sysstat_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $data = shift;

	my $sequence = '';
	
	my $stats = {};
	my %ids = ();

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;

		if ($val =~ m/^(\S+)=([\d,]+)$/)
		{
			$stats->{$1} = $2;
		}
	}

	for (sort (keys (%$stats)))
	{
		my $key = $_;

		if ($key =~ m/^cpu(\d*)$/)
		{
			write_to_rrd ($customer, $host, 'cpu', $1, split (m/,/, $stats->{$key}));
		}
		elsif ($key =~ m/^disk_io_([\d,]+)$/)
		{
			my $hd = $1;
			$hd =~ tr/,/_/;

			write_to_rrd ($customer, $host, 'disk_io', $hd, split (m/,/, $stats->{$key}));
		}
		elsif ($key eq 'swap' or $key eq 'page')
		{
			write_to_rrd ($customer, $host, 'io', $key, split (m/,/, $stats->{$key}));
		}
		elsif ($key eq 'intr' or $key eq 'ctxt'
				or $key eq 'processes')
		{
			my $name = 'forkrate';

			$name = 'interrupts'      if ($key eq 'intr');
			$name = 'contextswitches' if ($key eq 'ctxt');
			
			write_to_rrd ($customer, $host, $name, '', $stats->{$key});
		}
		elsif ($key eq 'meminfo')
		{
			write_to_rrd ($customer, $host, 'meminfo', '', split (m/,/, $stats->{$key}));
		}
		elsif ($key eq 'mysql_queries')
		{
			write_to_rrd ($customer, $host, 'mysql', 'queries', $stats->{$key});
		}
		elsif ($key =~ m/^suncpu(\d*)$/)
		{
			write_to_rrd ($customer, $host, 'suncpu', $1, split (m/,/, $stats->{$key}));
		}
		elsif ($key eq 'bind')
		{
			write_to_rrd ($customer, $host, 'bind', '', split (m/,/, $stats->{$key}));
		}
	}

	#print STDERR '', Data::Dumper->Dump ([$stats], ['stats']);

}

sub sysstat_create_cpu_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, 
		"DS:user:COUNTER:$HeartBeat:0:110",
		"DS:nice:COUNTER:$HeartBeat:0:110",
		"DS:syst:COUNTER:$HeartBeat:0:110",
		"DS:idle:COUNTER:$HeartBeat:0:110",
	);
}

sub sysstat_create_mem_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, 
		"DS:used:GAUGE:$HeartBeat:0:U",
		"DS:buffer:GAUGE:$HeartBeat:0:U",
		"DS:page:GAUGE:$HeartBeat:0:U",
		"DS:free:GAUGE:$HeartBeat:0:U"
	);
}

sub sysstat_create_diskio_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, 
		"DS:opts_ninf:COUNTER:$HeartBeat:0:U",
		"DS:opts_read:COUNTER:$HeartBeat:0:U",
		"DS:blks_read:COUNTER:$HeartBeat:0:U",
		"DS:opts_wrte:COUNTER:$HeartBeat:0:U",
		"DS:blks_wrte:COUNTER:$HeartBeat:0:U"
	);
}

sub sysstat_create_general_io_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, 
		"DS:incoming:COUNTER:$HeartBeat:0:U",
		"DS:outgoing:COUNTER:$HeartBeat:0:U"
	);
}

sub create_general_counter_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, 
		"DS:value:COUNTER:$HeartBeat:0:U"
	);
}

sub sysstat_create_bind_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, 
		"DS:success:COUNTER:$HeartBeat:0:U",
		"DS:referral:COUNTER:$HeartBeat:0:U",
		"DS:nxrrset:COUNTER:$HeartBeat:0:U",
		"DS:nxdomain:COUNTER:$HeartBeat:0:U",
		"DS:failure:COUNTER:$HeartBeat:0:U"
	);
}

sub print_meta_graph_cpu ($$$@)
{
	my $customer = shift;
	my $host = shift;
	my $begin = shift;
	my $end = shift;
	my @files = grep { $_ ne 'cpu.rrd' } (@_);

	my @cmd = ('-', '-s', $begin, '-e', $end, '-a', 'PNG');

	my $num = 0;
	my $rrdpath = get_rrdpath ();

	my $ds = { user => [], nice => [], syst => [], idle => [] };

	$num = scalar (@files);

	push (@cmd,
		'-v', 'Percent', '-u', '100',
		'-t', $num > 1 ? "Combined CPU Usage on $host ($num CPUs)" : "CPU Usage on $host");

	for (my $i = 0; $i < $num; $i++)
	{
		my $file = "$rrdpath/$customer/$host/" . $files[$i];

		push (@cmd, qq(DEF:user${i}_raw=$file:user:AVERAGE));
		push (@cmd, qq(DEF:nice${i}_raw=$file:nice:AVERAGE));
		push (@cmd, qq(DEF:syst${i}_raw=$file:syst:AVERAGE));
		push (@cmd, qq(DEF:idle${i}_raw=$file:idle:AVERAGE));
	}

	for (my $i = 0; $i < $num; $i++)
	{
		push (@cmd, qq(CDEF:user${i}_notnull=user${i}_raw,UN,0,user${i}_raw,IF));
		push (@cmd, qq(CDEF:nice${i}_notnull=nice${i}_raw,UN,0,nice${i}_raw,IF));
		push (@cmd, qq(CDEF:syst${i}_notnull=syst${i}_raw,UN,0,syst${i}_raw,IF));
		push (@cmd, qq(CDEF:idle${i}_notnull=idle${i}_raw,UN,0,idle${i}_raw,IF));

		push (@{$ds->{'user'}}, "user${i}_notnull");
		push (@{$ds->{'nice'}}, "nice${i}_notnull");
		push (@{$ds->{'syst'}}, "syst${i}_notnull");
		push (@{$ds->{'idle'}}, "idle${i}_notnull");
	}

	for (qw(user nice syst idle))
	{
		my $type = $_;
		my $cdef;
		$cdef = "CDEF:${type}_total=" . $ds->{$type}[0];
		for (my $i = 1; $i < $num; $i++)
		{
			$cdef .= ',' . $ds->{$type}[$i] . ',+';
		}
		push (@cmd, $cdef);
	}
	push (@cmd, 'CDEF:total=user_total,nice_total,+,syst_total,+,idle_total,+');
	push (@cmd, 'CDEF:user=100,user_total,*,total,/');
	push (@cmd, 'CDEF:nice=100,nice_total,*,total,/');
	push (@cmd, 'CDEF:syst=100,syst_total,*,total,/');
	push (@cmd, 'CDEF:idle=100,idle_total,*,total,/');

	push (@cmd,
		'HRULE:100#FF0000',
		'AREA:syst#FF0000:System',
		'GPRINT:syst:AVERAGE:%4.1lf%% Average,',
		'GPRINT:syst:MAX:%4.1lf%% Max,',
		'GPRINT:syst:LAST:%4.1lf%% Last\n',
		'STACK:user#0000FF:User  ',
		'GPRINT:user:AVERAGE:%4.1lf%% Average,',
		'GPRINT:user:MAX:%4.1lf%% Max,',
		'GPRINT:user:LAST:%4.1lf%% Last\n',
		'STACK:nice#00FF00:Nice  ',
		'GPRINT:nice:AVERAGE:%4.1lf%% Average,',
		'GPRINT:nice:MAX:%4.1lf%% Max,',
		'GPRINT:nice:LAST:%4.1lf%% Last');

	#print STDERR "RRDTool graph: " . join (' ', map { "'$_'" } (@cmd)) . "\n";
	RRDs::graph (@cmd);

	die (RRDs::error ()) if (RRDs::error ());
}

=head1 SEE ALSO

L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
