package CapMan::Plugins::NetApp;

use strict;
use warnings;

use Math::BigInt;

=head1 NAME

CapMan::Plugins::NetApp - CapMan plugin for monitoring NetApps

=head1 DESCRIPTION

This plugin monitors CPU usage, Network and Disk traffic, NFS and CIFS
operations, and other aspects of a NetApp.. L<http://netapp.com/>

=cut

use CapMan::Config qw(:rrdcreate get_rrdpath);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd create_rrdfile get_n_colors);

our $IOGraphDef = ['-v', 'Operations/s',
	'DEF:c={filename}:ops',
	'LINE1:c#0000FF:Operations/s',
	'GPRINT:c:MIN:%5.1lf%s Min,',
	'GPRINT:c:AVERAGE:%5.1lf%s Average,',
	'GPRINT:c:MAX:%5.1lf%s Max,',
	'GPRINT:c:LAST:%5.1lf%s Last'];

our $CapacityGraphDef = ['-v', 'Percent', '--lower-limit', '0',
	'DEF:free={filename}:free:AVERAGE',
	'DEF:used={filename}:used:AVERAGE',
	'CDEF:free_pct=free,free,used,+,/,100,*',
	'CDEF:used_pct=used,free,used,+,/,100,*',
	'AREA:used_pct#FF0000:Used',
	'GPRINT:used:MIN:%5.1lf%s Min,',
	'GPRINT:used:AVERAGE:%5.1lf%s Average,',
	'GPRINT:used:MAX:%5.1lf%s Max,',
	'GPRINT:used:LAST:%5.1lf%s Last\l',
	'STACK:free_pct#00FF00:Free',
	'GPRINT:free:MIN:%5.1lf%s Min,',
	'GPRINT:free:AVERAGE:%5.1lf%s Average,',
	'GPRINT:free:MAX:%5.1lf%s Max,',
	'GPRINT:free:LAST:%5.1lf%s Last\l'];

our $CapacityBytesGraphDef = ['-v', 'Percent', '--lower-limit', '0',
	'DEF:free={filename}:free:AVERAGE',
	'DEF:used={filename}:used:AVERAGE',
	'CDEF:free_pct=free,free,used,+,/,100,*',
	'CDEF:used_pct=used,free,used,+,/,100,*',
	'CDEF:free_byte=free,1000,*',
	'CDEF:used_byte=used,1000,*',
	'AREA:used_pct#FF0000:Used',
	'GPRINT:used_byte:MIN:%5.1lf%s Min,',
	'GPRINT:used_byte:AVERAGE:%5.1lf%s Average,',
	'GPRINT:used_byte:MAX:%5.1lf%s Max,',
	'GPRINT:used_byte:LAST:%5.1lf%s Last\l',
	'STACK:free_pct#00FF00:Free',
	'GPRINT:free_byte:MIN:%5.1lf%s Min,',
	'GPRINT:free_byte:AVERAGE:%5.1lf%s Average,',
	'GPRINT:free_byte:MAX:%5.1lf%s Max,',
	'GPRINT:free_byte:LAST:%5.1lf%s Last\l'];

our $NFSv3GraphDef = ['-v', 'Procedures/s', '--logarithmic',
	'DEF:i={filename}:issues:AVERAGE',
	'DEF:c={filename}:cached:AVERAGE',
	'AREA:i#0000FF:Issues/s',
	'GPRINT:i:MIN:%5.1lf%s Min,',
	'GPRINT:i:AVERAGE:%5.1lf%s Average,',
	'GPRINT:i:MAX:%5.1lf%s Max,',
	'GPRINT:i:LAST:%5.1lf%s Last\l',
	'AREA:c#00FF00:Cached  ',
	'GPRINT:c:MIN:%5.1lf%s Min,',
	'GPRINT:c:AVERAGE:%5.1lf%s Average,',
	'GPRINT:c:MAX:%5.1lf%s Max,',
	'GPRINT:c:LAST:%5.1lf%s Last\l'];

=head1 SERVICES PROVIDED

=over 4

=item name

This is the service which would be provided if this was a real plugin.

=back

=cut

register_graph_handler  ('io_ops', $IOGraphDef);
register_graph_handler  ('netapp_nfsv3_ops', $NFSv3GraphDef, \&nfsv3ops_graph_handler);
register_graph_handler  ('netapp_capacity_bytes', $CapacityBytesGraphDef);
register_graph_handler  ('netapp_capacity_files', $CapacityGraphDef);
register_graph_handler  ('netapp_capacity_inodes', $CapacityGraphDef);

register_create_handler ('io_ops', \&io_ops_create_handler);
register_create_handler ('netapp_capacity_bytes', \&capacity_create_handler);
register_create_handler ('netapp_capacity_inodes', \&capacity_create_handler);
register_create_handler ('netapp_capacity_files', \&capacity_create_handler);
register_create_handler ('netapp_nfsv3_ops', \&nfsv3ops_create_handler);

register_input_handler ('netapp_cpu', '.1.3.6.1.4.1.789.1.2.1', \&netapp_cpu_handler);
register_input_handler ('netapp_io', '.1.3.6.1.4.1.789.1.2.2', \&netapp_misc_handler);
register_input_handler ('netapp_capacity', '.1.3.6.1.4.1.789.1.5.4.1', \&netapp_capacity_handler);
register_input_handler ('netapp_nfsv3', '.1.3.6.1.4.1.789.1.3.1.2.4', \&netapp_nfsv3_handler);

# NETWORK-APPLIANCE-MIB::enclTempSensorsCurrentTemp.1 = STRING: "30C (86F)"
# NETWORK-APPLIANCE-MIB::enclTempSensorsCurrentTemp.2 = STRING: "26C (78F)"
# NETWORK-APPLIANCE-MIB::enclTempSensorsCurrentTemp.3 = STRING: "26C (78F)"
register_input_handler ('netapp_temp', '.1.3.6.1.4.1.789.1.21.1.2.1.25', \&load_handler);

#register_graph_handler ('load', $GraphDef);

return (1);

sub timeticks_to_seconds
{
	my $tt = shift;

	return (0) unless ($tt);

	if ($tt =~ m/^(\d+):(\d+):(\d+):(\d+)(?:\.\d+)?$/)
	{
		return ((86400 * $1) + (3600 * $2) + (60 * $3) + $4);
	}

	return (0);
}

sub netapp_cpu_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv  = shift;
	my $data = shift;
	my $hash = {};

	my $num;

	my $cpuBusyTime = '.1.3.6.1.4.1.789.1.2.1.2';
	my $cpuIdleTime = '.1.3.6.1.4.1.789.1.2.1.4';
	my $cpuCount = '.1.3.6.1.4.1.789.1.2.1.6';

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;
		$hash->{$name}[$seq] = $val;
	}

#	if ($::DEBUG)
#	{
#		require Data::Dumper;
#		print Data::Dumper->Dump ([$data], ['data']);
#	}

	return if (!$hash->{$cpuCount});
	($num) = @{$hash->{$cpuCount}};

	for (my $i = 0; $i < $num; $i++)
	{
		my $busy = timeticks_to_seconds ($hash->{$cpuBusyTime}[$i]) or next;
		my $idle = timeticks_to_seconds ($hash->{$cpuIdleTime}[$i]) or next;

		write_to_rrd ($customer, $host, 'cpu', $i, 'U', 'U', $busy, $idle);
	}
}

sub netapp_misc_handler # `misc' is what the MIB tells this section.. *shrug*
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv  = shift;
	my $data = shift;
	my $hash = {};

	my $num;

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;
		$hash->{$name}[$seq] = $val;
	}

	if (defined ($hash->{'.1.3.6.1.4.1.789.1.2.2.12'}) and defined ($hash->{'.1.3.6.1.4.1.789.1.2.2.14'}))
	{
		my $incoming = $hash->{'.1.3.6.1.4.1.789.1.2.2.12'}->[0];
		my $outgoing = $hash->{'.1.3.6.1.4.1.789.1.2.2.14'}->[0];
		write_to_rrd ($customer, $host, 'interfaces', 'net', $incoming, $outgoing);
	}
}

sub netapp_capacity_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv  = shift;
	my $data = shift;
	my $hash = {};

	my $dfIndex = '.1.3.6.1.4.1.789.1.5.4.1.1';
	my $dfMountedOn = '.1.3.6.1.4.1.789.1.5.4.1.10';

	my $dfMaxFilesAvail = '.1.3.6.1.4.1.789.1.5.4.1.11';
	my $dfMaxFilesUsed = '.1.3.6.1.4.1.789.1.5.4.1.12';

	my $dfInodesUsed = '.1.3.6.1.4.1.789.1.5.4.1.7';
	my $dfInodesFree = '.1.3.6.1.4.1.789.1.5.4.1.8';

	my $dfHighUsedKBytes = '.1.3.6.1.4.1.789.1.5.4.1.16';
	my $dfLowUsedKBytes = '.1.3.6.1.4.1.789.1.5.4.1.17';
	my $dfHighAvailKBytes = '.1.3.6.1.4.1.789.1.5.4.1.18';
	my $dfLowAvailKBytes = '.1.3.6.1.4.1.789.1.5.4.1.19';
	

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;
		$hash->{$name}[$seq] = $val;
	}

	for (my $i = 1; defined ($hash->{$dfIndex}[$i]); $i++)
	{
		my $inst = $hash->{$dfMountedOn}[$i] or next;
		$inst =~ s#^/##;
		$inst =~ s#\W#_#g;

		my $kbytes_used = Math::BigInt->new ($hash->{$dfHighUsedKBytes}[$i]);
		$kbytes_used->blsft (32);
		$kbytes_used->badd ($hash->{$dfLowUsedKBytes}[$i]);
		my $kbytes_free = Math::BigInt->new ($hash->{$dfHighAvailKBytes}[$i]);
		$kbytes_free->blsft (32);
		$kbytes_free->badd ($hash->{$dfLowAvailKBytes}[$i]);
		write_to_rrd ($customer, $host, 'netapp_capacity_bytes', $inst,
			$kbytes_used, $kbytes_free);

		my $inodes_used = $hash->{$dfInodesUsed}[$i];
		my $inodes_free = $hash->{$dfInodesFree}[$i];
		write_to_rrd ($customer, $host, 'netapp_capacity_inodes', $inst,
			$inodes_used, $inodes_free);

		my $files_used = $hash->{$dfMaxFilesUsed}[$i];
		my $files_free = $hash->{$dfMaxFilesAvail}[$i];
		write_to_rrd ($customer, $host, 'netapp_capacity_files', $inst,
			$files_used, $files_free);
	}
}

#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cNulls.0 = Counter32: 419592
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cGetattrs.0 = Counter32: 2645785615
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cSetattrs.0 = Counter32: 369660394
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cLookups.0 = Counter32: 2056427839
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cAccesss.0 = Counter32: 2990941367
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cReadlinks.0 = Counter32: 99885993
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cReads.0 = Counter32: 1619266444
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cWrites.0 = Counter32: 2027075962
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cCreates.0 = Counter32: 62635539
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cMkdirs.0 = Counter32: 620033
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cSymlinks.0 = Counter32: 257355
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cMknods.0 = Counter32: 501
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cRemoves.0 = Counter32: 52739997
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cRmdirs.0 = Counter32: 1119332
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cRenames.0 = Counter32: 216967398
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cLinks.0 = Counter32: 2376313
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cReaddirs.0 = Counter32: 527814521
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cReaddirPluss.0 = Counter32: 5684643
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cFsstats.0 = Counter32: 47095071
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cFsinfos.0 = Counter32: 18250866
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cPathconfs.0 = Counter32: 378
#.iso.org.dod.internet.private.enterprises.netapp.netapp1.nfs.curNfs.nfsServ.nfsV3.v3Calls.v3cCommits.0 = Counter32: 2
sub netapp_nfsv3_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv  = shift;
	my $data = shift;
	my $hash = {};

	my $v3Calls = '.1.3.6.1.4.1.789.1.3.1.2.4.1';
	my $v3CachedCalls = '.1.3.6.1.4.1.789.1.3.1.2.4.3';

	my @ops = (undef, qw(Null Getattrs Setattrs Lookups Accesss
		Readlinks Reads Writes Creates Mkdirs Symlinks Mknods
		Removes Rmdirs Renames Links Readdirs ReaddirPluss Fsstats
		Fsinfos Pathconfs Commits));

#	my %ops =
#	(
#		Null	     => '.1.3.6.1.4.1.789.1.3.1.2.4.1.1',
#		Getattrs     => '.1.3.6.1.4.1.789.1.3.1.2.4.1.2',
#		Setattrs     => '.1.3.6.1.4.1.789.1.3.1.2.4.1.3',
#		Lookups      => '.1.3.6.1.4.1.789.1.3.1.2.4.1.4',
#		Accesss      => '.1.3.6.1.4.1.789.1.3.1.2.4.1.5',
#		Readlinks    => '.1.3.6.1.4.1.789.1.3.1.2.4.1.6',
#		Reads        => '.1.3.6.1.4.1.789.1.3.1.2.4.1.7',
#		Writes       => '.1.3.6.1.4.1.789.1.3.1.2.4.1.8',
#		Creates      => '.1.3.6.1.4.1.789.1.3.1.2.4.1.9',
#		Mkdirs       => '.1.3.6.1.4.1.789.1.3.1.2.4.1.10',
#		Symlinks     => '.1.3.6.1.4.1.789.1.3.1.2.4.1.11',
#		Mknods       => '.1.3.6.1.4.1.789.1.3.1.2.4.1.12',
#		Removes      => '.1.3.6.1.4.1.789.1.3.1.2.4.1.13',
#		Rmdirs       => '.1.3.6.1.4.1.789.1.3.1.2.4.1.14',
#		Renames      => '.1.3.6.1.4.1.789.1.3.1.2.4.1.15',
#		Links        => '.1.3.6.1.4.1.789.1.3.1.2.4.1.16',
#		Readdirs     => '.1.3.6.1.4.1.789.1.3.1.2.4.1.17',
#		ReaddirPluss => '.1.3.6.1.4.1.789.1.3.1.2.4.1.18',
#		Fsstats      => '.1.3.6.1.4.1.789.1.3.1.2.4.1.19',
#		Fsinfos      => '.1.3.6.1.4.1.789.1.3.1.2.4.1.20',
#		Pathconfs    => '.1.3.6.1.4.1.789.1.3.1.2.4.1.21',
#		Commits      => '.1.3.6.1.4.1.789.1.3.1.2.4.1.22'
#	);

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;
		$hash->{$name}[$seq] = $val;
	}

	for (my $i = 1; $i < @ops; $i++)
	{
		my $op_name = $ops[$i];
		my $op_oid = $v3Calls . '.' . $i;
		my $cop_oid = $v3CachedCalls . '.' . $i;

		my $value;
		my $cvalue;

		next if (!defined ($hash->{$op_oid}));
		next if (!defined ($hash->{$cop_oid}));

		($value) = @{$hash->{$op_oid}};
		($cvalue) = @{$hash->{$cop_oid}};

		write_to_rrd ($customer, $host, 'netapp_nfsv3_ops', $op_name,
			$value, $cvalue);
	}
}

sub io_ops_create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:ops:COUNTER:$HeartBeat:0:U");
}

sub capacity_create_handler
{
	my $file = shift;
	create_rrdfile ($file,
		"DS:free:GAUGE:$HeartBeat:0:U",
		"DS:used:GAUGE:$HeartBeat:0:U");
}

sub nfsv3ops_create_handler
{
	my $file = shift;
	create_rrdfile ($file,
		"DS:issues:COUNTER:$HeartBeat:0:U",
		"DS:cached:COUNTER:$HeartBeat:0:U");
}

sub nfsv3ops_graph_handler
{
	my $customer = shift;
	my $host  = shift;
	my $begin = shift;
	my $end   = shift;
	my $title = shift;
	my @files = @_;

	my @colors = get_n_colors (scalar (@files));

	my @names;
	my $names_max = 0;
	my $num = 0;

	my @cmd = ('-', '-s', $begin, '-e', $end, '-a', 'PNG', '-t', $title, '-v', 'Procedures/s', '--height', '300', '--logarithmic');
	my $path = get_rrdpath ();

	for (@files)
	{
		my $file = "$path/" . lc ($customer) . "/$host/$_";
		my $name = (split (m/-/, $_))[1];

		$name =~ s/\.rrd$//;
		$name =~ s/_+/ /g;

		push (@cmd, qq(DEF:val$num=$file:issues:AVERAGE));
		$num++;

		push (@names, $name);
		if (length ($name) > $names_max)
		{
			$names_max = length ($name);
		}
	}

	my $format = "%-${names_max}s";
	for (my $i = 0; $i < $num; $i++)
	{
		my $line = sprintf ($format, $names[$i]);
		my $color = $colors[$i % scalar (@colors)];
		
		push (@cmd, 'LINE1:val' . $i . '#' . $color . ":$line");
		push (@cmd, "GPRINT:val$i:AVERAGE:\%6.1lf Average,",
			"GPRINT:val$i:MAX:\%6.1lf Max,",
			"GPRINT:val$i:LAST:\%6.1lf Last\\n");
	}

	RRDs::graph (@cmd);

	die (RRDs::error ()) if (RRDs::error ());
}

=head1 SEE ALSO

L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
