package CapMan::Plugins::Interfaces;

use strict;
use warnings;

=head1 NAME

CapMan::Plugins::Interfaces - Get interface statistics from nearly any SNMPd
implementation.

=head1 DESCRIPTION

This plugin uses the default IF-MIB to query interface statistics from pretty
much any device with a network connection.

=cut

use CapMan::Config qw(:rrdcreate);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd create_rrdfile);

our $GraphDef = ['-v', 'Bits/s',
	'DEF:out_avg_raw={filename}:incoming:AVERAGE',
	'DEF:inc_avg_raw={filename}:outgoing:AVERAGE',
	'CDEF:out_avg=out_avg_raw,8,*',
	'CDEF:inc_avg=inc_avg_raw,8,*',
	'CDEF:overlap=out_avg,inc_avg,GT,inc_avg,out_avg,IF',
	'CDEF:mytime=out_avg_raw,TIME,TIME,IF',
	'CDEF:sample_len_raw=mytime,PREV(mytime),-',
	'CDEF:sample_len=sample_len_raw,UN,0,sample_len_raw,IF',
	'CDEF:out_avg_sample=out_avg_raw,UN,0,out_avg_raw,IF,sample_len,*',
	'CDEF:out_avg_sum=PREV,UN,0,PREV,IF,out_avg_sample,+',
	'CDEF:inc_avg_sample=inc_avg_raw,UN,0,inc_avg_raw,IF,sample_len,*',
	'CDEF:inc_avg_sum=PREV,UN,0,PREV,IF,inc_avg_sample,+',
	"AREA:out_avg#B7EFB7",
	"AREA:inc_avg#B7B7F7",
	"AREA:overlap#89B3C9",
	"LINE1:out_avg#00E000:Outgoing",
	'GPRINT:out_avg:AVERAGE:%5.1lf%s Average,',
	'GPRINT:out_avg:MAX:%5.1lf%s Max,',
	'GPRINT:out_avg:LAST:%5.1lf%s Last',
	'GPRINT:out_avg_sum:LAST:(ca. %5.1lf%sB Total)\l',
	"LINE1:inc_avg#0000FF:Incoming",
	'GPRINT:inc_avg:AVERAGE:%5.1lf%s Average,',
	'GPRINT:inc_avg:MAX:%5.1lf%s Max,',
	'GPRINT:inc_avg:LAST:%5.1lf%s Last',
	'GPRINT:inc_avg_sum:LAST:(ca. %5.1lf%sB Total)\l'];

=head1 SERVICES PROVIDED

=over 4

=item interfaces

The incoming and outgoing bytes of a network-port. Graphed as Bits/s.

=back

=cut

register_input_handler  ('interfaces', '.1.3.6.1.2.1.2', \&input_handler);
register_create_handler ('interfaces', \&create_handler);
register_graph_handler  ('interfaces', $GraphDef);

return (1);

sub input_handler ($$$)
{
	my $host = shift;
	my $srv = shift;
	my $data = shift;

	my $i;
	my $hash = {};

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;

		if (!defined ($hash->{$name}))
		{
			$hash->{$name} = [$val];
		}
		else
		{
			push (@{$hash->{$name}}, $val);
		}
	}

	if (!defined ($hash->{'ifNumber'}[0]))
	{
		#print STDERR "Unable to query interfaces on host $host.\n";
		return;
	}

	for ($i = 0; $i < $hash->{'ifNumber'}[0]; $i++)
	{
		my $descr    = $hash->{'ifDescr'}[$i];
		my $incoming = $hash->{'ifInOctets'}[$i];
		my $outgoing = $hash->{'ifOutOctets'}[$i];

		if (!defined ($descr))
		{
			print STDERR "$host: Interface $i has no description.\n";
			next;
		}
		elsif (!defined ($incoming) and !defined ($outgoing))
		{
			print STDERR "$host: No or incomplete counters.\n";
			next;
		}

		write_to_rrd ($host, 'interfaces', $descr, $incoming, $outgoing);
	}
}

sub create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file,
		"DS:incoming:COUNTER:$HeartBeat:0:U",
		"DS:outgoing:COUNTER:$HeartBeat:0:U"
	);
}

=head1 SEE ALSO

L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
