package CapMan::Config;

use strict;
use warnings;

use Exporter;
use Fcntl (':flock');
use Carp (qw(cluck confess));
use Data::Dumper ();

=head1 USAGE

  use CapMan::Config qw(read_config get_host_config);

  read_config ($filename);
  my @hosts    = get_all_hosts ();
  my @settings = get_host_config ($hostname, $config_directive);

Functions do exactly what you expect them to. In case of ``get_host_config''
you expect it to return ('a', 'b', 'c') for 'services' if you did set
``services: a, b, c''..

=cut

our $Config = {};
our $ConfPath = '/etc/capman';
our $AddressToAlias = undef;

our $Step = 300;
our $HeartBeat = 900;
our $DailyRows   = int (  3024000 / $Step);          #   35 Days    ( 5 Minutes)
our $WeeklyRows  = int (  7948800 / ($Step * 6));    #   92 Days    (30 Minutes)
our $MonthlyRows = int ( 34560000 / ($Step * 24));   #  400 Days    ( 2 Hours)
our $YearlyRows  = int (158112000 / ($Step * 288));  #    5 Years   ( 1 Day)

@CapMan::Config::EXPORT_OK = qw(read_config
				get_all_customers get_all_hosts
				get_host_config
				get_rrdpath
				get_plugindir
				host_address_to_alias
				host_alias_to_address
				host_check_auth
				get_customer_names_by_user_name
				$Step $HeartBeat
				$DailyRows $WeeklyRows $MonthlyRows $YearlyRows);
%CapMan::Config::EXPORT_TAGS =
(
	rrdcreate => [qw($Step $HeartBeat $DailyRows $WeeklyRows $MonthlyRows $YearlyRows)]
);
@CapMan::Config::ISA = qw(Exporter);

sub read_config (;$);

=head1 CONFIG SYNTAX

Das folgende Beispiel gibt einen generellen Ueberblick ueber den Aufbau einer
Config fuer CapMan.

  [host _default_]
  services: load interfaces storage sysstat

  [host test0.example.com]
  # uses defaults

  [host mailserver]
  address: test1.example.com
  services: load interfaces sysstat

  [include backup_net.conf]
  # includes /etc/capman/backup_net.conf

  [include /usr/share/hosts.conf]
  # includes /usr/share/hosts.conf

  [general]
  plugindir: /usr/share/perl5/CapMan/Plugins
  destdir: /var/lib/rrd/capman

=head1 EXPORTED VARIABLES

    $Step
    $HeartBeat
    $DailyRows
    $WeeklyRows
    $MonthlyRows
    $YearlyRows

To import all of the above variables use the B<rrdcreate>-tag, e.g. by issueing
C<use CapMan::Config (qw(:rrdcreate));>.

=head1 EXPORTED FUNCTIONS

=over 4

=item B<read_config> ([I<$config_file>])

Read and parse the config file I<$config_file>. If I<$config_file> is not given
F</etc/capman/capman.conf> will be read.  Must be called before any of the
B<get_*> functions may be called.

=cut

{
	my $current_customer = '_default_';
	my $current_host = '_default_';
	my $current_section = '';
	my $current_section_opt = '';
	my %parsing_files = ();

	sub read_config (;$)
	{
		my $file = @_ ? shift : 'capman.conf';
		my $fh;

		my $counter = 0;
		my %default = ();

		$file = $ConfPath . '/' . $file unless ($file =~ m#^/#);

		if (!-e $file)
		{
			print STDERR "Could not read config from $file.\n";
			return;
		}

		if ($parsing_files{$file})
		{
			print STDERR "File $file was included recursively.\n";
			return;
		}
		$parsing_files{$file} = 1;

		if ($::DEBUG)
		{
			print STDOUT "DEBUG: Reading config file $file\n";
		}

		open ($fh, "< $file") or die ("open ($file): $!");
		flock ($fh, LOCK_SH) or die ("flock ($file): $!");

		while (<$fh>)
		{
			my $line;

			chomp;
			s/#.*//;
			s/^\s+//;
			s/\s+$//;
			s/\s+/ /g;

			$line = $_;
			$counter++;

			next unless ($line);

			if ($line =~ m/^\[(\w+)(?:\s+([^\]]+))?\]$/)
			{
				my $sec = lc ($1);
				my $sub = defined ($2) ? $2 : '';

				if ($sec eq 'include')
				{
					read_config ($sub);
				}
				elsif ($sec eq 'general')
				{
					$current_customer = '_default_';
					$current_host = '_default_';
					$current_section = $sec;
					$current_section_opt = $sub;
					$Config->{':general'} = {}
						if (!defined ($Config->{':general'}));
				}
				elsif (($sec eq 'customer') || ($sec eq 'host'))
				{
					$sub ||= '_default_';

					if ($sec eq 'customer')
					{
						$current_customer = lc ($sub);
						$current_host = '_default_';
					}
					else
					{
						$current_host = $sub;
					}
					$current_section = $sec;
					$current_section_opt = $sub;

					$Config->{':customer'}{$current_customer}{':host'}{$current_host} = {}
						if (!defined ($Config->{':customer'}{$current_customer}{':host'}{$current_host}));
				}
				else
				{
					$current_customer = '_default_';
					$current_host = '_default_';
					$current_section = '';
					$current_section_opt = '';
					print STDERR "Invalid section `$sec' in $file, line $counter.\n";
					next;
				}
			}
			elsif ($line =~ m/^(\w+)\s*:\s*(.+)/)
			{
				my $key = lc ($1);
				my @val = split (m/\s*,\s*/, $2);
				my $ptr;

				if (!$current_section)
				{
					print STDERR "Syntax error in $file, line $counter: Key $key not in a section.\n";
					next;
				}

				if ($current_section eq 'general')
				{
					$ptr = $Config->{':general'};
				}
				elsif ($current_section eq 'customer')
				{
					$ptr = $Config->{':customer'}{$current_customer};
				}
				elsif ($current_section eq 'host')
				{
					$ptr = $Config->{':customer'}{$current_customer}{':host'}{$current_host};
				}
				else
				{
					die ("You should not get here");
				}

				$ptr->{$key} = [] unless (defined ($ptr->{$key}));
				unshift (@{$ptr->{$key}}, @val);
			}
			else
			{
				print STDERR "Syntax error in $file, line $counter\n";
			}
		} # while (<$fh>)

		close ($fh);
		$parsing_files{$file} = 0;

		if ($::DEBUG)
		{
			my $o = Data::Dumper->Dump ([$file, $Config], [qw(file Config)]);
			$o =~ s/^/DEBUG: /gm;
			print STDOUT $o;
		}
	}
}

=item B<get_all_customers> ([I<$user>])

Returns a list of all customers. If the optional argument I<$user> is given
only customers which may be accessed by the given user are returned.

This function is different from B<get_customer_names_by_user_name> because it
only returnes actually configured customers for a given user, where
get_customer_names_by_user_name returns B<all> customers for the user.

=cut

sub get_all_customers
{
	confess ("Wrong number of arguments") if (@_ > 1);

	my $user = shift;

	if ($user)
	{
		return (grep { defined ($Config->{':customer'}{$_}) } (get_customer_names_by_user_name ($user)));
	}
	return (grep { ($_ ne '') and ($_ ne '_default_') } (keys %{$Config->{':customer'}}));
}


=item B<get_all_hosts> (I<$customer>, [I<$user>])

Returns a list of all hosts configured. Returns the aliases rather than the
actual addresses. If the optional I<$user> argument is passed returnes only
hosts the given user may access.

=cut

sub get_all_hosts
{
	confess ("Wrong number of arguments") if ((@_ != 1) && (@_ != 2));

	my $customer = shift;
	my $user = shift;

	if (!$user)
	{
		return (grep { $_ ne '' and $_ ne '_default_' } (sort (keys (%{$Config->{':customer'}{$customer}{':host'}}))));
	}

	return (grep { $_ ne '' and $_ ne '_default_' and host_check_auth ($customer, $_, $user) != 0 } (sort (keys (%{$Config->{':customer'}{$customer}{':host'}}))));
}

=item B<get_host_config> (I<$customer>, I<$host>, I<$param>)

Returns the value for I<$param> on I<$host> of customer I<$customer>. In list
context returns all configured values. If I<$param> is not configured for
I<$host> the default host's config will be used..

=cut

sub get_host_config
{
	confess ("Wrong number of arguments") if (@_ != 3);

	my $customer = shift;
	my $host   = shift;
	my $config = shift;
	my $ptr;

	if (defined ($Config->{':customer'}{$customer})
			and defined ($Config->{':customer'}{$customer}{':host'}{$host})
			and defined ($Config->{':customer'}{$customer}{':host'}{$host}{$config}))
	{
		$ptr = $Config->{':customer'}{$customer}{':host'}{$host}{$config};
	}
	elsif (defined ($Config->{':customer'}{$customer})
			and defined ($Config->{':customer'}{$customer}{':host'}{'_default_'})
			and defined ($Config->{':customer'}{$customer}{':host'}{'_default_'}{$config}))
	{
		$ptr = $Config->{':customer'}{$customer}{':host'}{'_default_'}{$config};
	}
	elsif (defined ($Config->{':customer'}{'_default_'})
			and defined ($Config->{':customer'}{'_default_'}{':host'}{$host})
			and defined ($Config->{':customer'}{'_default_'}{':host'}{$host}{$config}))
	{
		$ptr = $Config->{':customer'}{'_default_'}{':host'}{$host}{$config};
	}
	elsif (defined ($Config->{':customer'}{'_default_'})
			and defined ($Config->{':customer'}{'_default_'}{':host'}{'_default_'})
			and defined ($Config->{':customer'}{'_default_'}{':host'}{'_default_'}{$config}))
	{
		$ptr = $Config->{':customer'}{'_default_'}{':host'}{'_default_'}{$config};
	}

	return if (!$ptr);
	return (wantarray ? @$ptr : $ptr->[0]);
}

=item B<host_address_to_alias> (I<$host>)

Returns the alias to which I<$host> belongs. This is the reverse direction to
the config file, in which hostnames are assigned to aliases, not vice versa..

=cut

#sub host_address_to_alias ($)
#{
#	my $host = shift;
#
#	if (!defined ($AddressToAlias))
#	{
#		$AddressToAlias = {};
#
#		print STDOUT "DEBUG: Building AddressToAlias-List\n" if ($::DEBUG);
#
#		for (keys %{$Config->{':customer'}})
#		{
#			my $customer = $_;
#
#			for (keys %{$Config->{':customer'}{$customer}{':host'}})
#			{
#				my $alias = $_;
#
#				$AddressToAlias->{$alias} = $alias;
#
#				if (defined ($Config->{':customer'}{$customer}{':host'}{'address'}))
#				{
#					my $h = $Config->{':customer'}{$customer}{':host'}{'address'}[0];
#
#					$AddressToAlias->{$h} = $alias;
#				}
#			} # for hosts
#		} # for customers
#	}
#
#	return (defined ($AddressToAlias->{$host}) ? $AddressToAlias->{$host} : '');
#}

=item B<host_alias_to_address> (I<$customer>, I<$alias>)

Returns the host or address associated with I<$alias>. This is almost the same
as ``B<get_host_config> (I<$alias>, I<'address'>)'' (the default host's config
will not be used here).

=cut

sub host_alias_to_address
{
	confess ("Wrong number of arguments") if (@_ != 2);

	my $customer = shift;
	my $host = shift;

	if (!defined ($Config->{':customer'}{$customer}{':host'}{$host}))
	{
		return;
	}
	elsif (!defined ($Config->{':customer'}{$customer}{':host'}{$host}{'address'}))
	{
		return ($host);
	}
	else
	{
		$Config->{':customer'}{$customer}{':host'}{$host}{'address'}[0];
	}
}

=item B<host_check_auth> (I<$customer>, I<$host>, I<$user>)

Check wether user I<$user> is allowed to see/access host I<$host> of customer
I<$customer>. This is done using the B<user> and B<customer> options from the
config file. If access is granted a true value is returned, if access is denied
a false value is returned.

=cut

sub host_check_auth
{
	confess ("Wrong number of arguments") if (@_ != 3);

	my $customer = shift;
	my $host = shift;
	my $user = shift;
	my %customers = ();
	my @allowed_users;

	$customers{lc ($_)} = 1 for (get_customer_names_by_user_name ($user));

	$customer = lc ($customer);

	return (0) if (!$customers{$customer});
	return (0) if (!defined $Config->{':customer'}{$customer}{':host'}{$host});

	@allowed_users = get_host_config ($customer, $host, 'user');

	# If no users are configured anywhere, we will allow access.. After
	# all, the user has access to the customer's data.
	if (!@allowed_users)
	{
		print STDERR "No allowed users.\n";
		return (1);
	}

	print STDERR '@allowed_users = (' . join (', ', @allowed_users) . ");\n";

	if (grep { $_ eq $user } (@allowed_users))
	{
		return (1);
	}

	return (0);
}

=item B<get_rrdpath> ()

Returns the path where the RRD files are kept.

=cut

sub get_rrdpath
{
	if (defined ($Config->{':general'}{'destdir'}))
	{
		return ($Config->{':general'}{'destdir'}[-1]);
	}
	else
	{
		return ('/var/lib/rrd/capman');
	}
}

=item B<get_plugindir> ()

Returns the path where the plugins are installed.

=cut

sub get_plugindir
{
	if (defined ($Config->{':general'}{'plugindir'}))
	{
		return ($Config->{':general'}{'plugindir'}[-1]);
	}
	else
	{
		return ('/usr/share/perl5/CapMan/Plugins');
	}
}

{
	my $callback_ref;
	sub _get_user_to_customer_callback
	{
		if (defined ($callback_ref))
		{
			return ($callback_ref);
		}

		return if (!defined ($Config->{':general'}{'customer_callback'}));

		my $module = $Config->{':general'}{'customer_callback'}[-1];

		if (!($module =~ m/^(\w+::)*\w+$/))
		{
			print STDERR "`$module' is not a valid module name.\n";
			return;
		}

		$@ = '';
		eval <<EVAL;
	use $module (qw(username_to_customer_list));
EVAL
		if ($@)
		{
			print STDERR "Loading `$module' failed: $@\n";
			return;
		}

		$callback_ref = \&username_to_customer_list;
		return ($callback_ref);
	}
}

=item B<get_customer_names_by_user_name> (I<$user_name>)

Returns a list of all customers the given user belongs to. In scalar context a
array-ref is returnes, otherwise it's a list. Reqeuests are cached so calling
this function often is appropriate.

For this function to work you need to set the B<customer_callback> option in
the B<general> section. The given module is loaded (searching in @INC) and the
function B<username_to_customer_list> is imported from there. An error is
thrown if anything failes.

An example module could look like this:

  package noris::UserToCustomer;
  use Exporter;
  use Magic;

  @noris::UserToCustomer::EXPORT_OK = qw(username_to_customer_list);
  @noris::UserToCustomer::ISA = qw(Exporter);

  return (1);

  sub username_to_customer_list
  {
    my $user_name = shift;
    return (magic_find_customer ($user_name));
  }

The config entry for this module would look like this, assuming it's somewhere
in I<@INC>:

  [general]
  customer_callback noris::UserToCustomer

=cut

{
	my $user_name_to_customer_cache = {};
	sub get_customer_names_by_user_name
	{
		my $user_name = shift;
		my $ret = [];
		if (!defined ($user_name_to_customer_cache->{$user_name}))
		{
			my $cb = _get_user_to_customer_callback ();
			my @list = &$cb ($user_name);
			$user_name_to_customer_cache->{$user_name} = \@list;
		}
		$ret = $user_name_to_customer_cache->{$user_name};
		return (wantarray () ? (@$ret) : [@$ret]);
	}
}

=back

=head1 AUTHOR

Florian Forster E<lt>octo at noris.netE<gt> for the L<noris network AG|http://noris.net/>.

=cut
