package CapMan::Services;

use strict;
use warnings;

use Exporter;
use Carp (qw(cluck confess));
use Data::Dumper ();
use RRDs ();

use CapMan::Config qw(get_host_config get_plugindir);
use CapMan::Tools qw(write_to_rrd);

=head1 USAGE

  use CapMan::Config qw(read_config);
  use CapMan::Services qw(load_plugins);

  read_config ();
  load_plugins ();
  
  
  ### OR ###
  
  
  use CapMan::Services qw(:plugin);

  register_input_handler ('myservice', '.1.2.3.4.5.6.7.8.9', \&input_callback);
  register_graph_handler ('myservice', [qw(arguments passed to rrd)]);
  register_graph_handler ('mymetaservice', \@rrdargs, \&graph_callback);


  ### OR ###


  use CapMan::Services qw(:frontend);

  run_input_callback ($hostname, $service, $data);
  
  my @rrdargs = map { s/{filename}/$filename/g; } (get_graph_definition ($service));

  if (is_meta_service ($service))
  {
    run_graph_callback ($service, $hostname, $begin, $end, @filenames)
  }

=cut

our %OIDMap = ();
our %InputCallbacks = ();
our %CreateCallbacks = ();
our %GraphReal = ();
our %GraphMeta = ();

=head1 EXPORTED SYMBOLS

The following functions and variables may be imported by other modules.

=head2 Exported Functions

=over 4

=item B<load_plugins> ()

Looks for B<*.pm> files in the I<plugindir> and loads all modules it can find.

=item B<register_input_handler> (I<$service>, I<$oid>, I<$callback>)

Registers a new service: The provided oid will be queried and the result will
be provided to the callback function (provided as code-ref). The arguments
passed to the callback function are (I<$hostname>, I<$service>, I<$data>),
where I<$hostname> and I<$service> are sclars and I<$data> is a two-dimensional
array-ref as provided by L<the SNMP module|SNMP>.

=item B<register_create_handler> (I<$service>, I<$create_handler>)

Registers a create-handler for the given service. The filename is passed to the
code-ref I<$create_handler> and it is expected to create a file. Create-handler
and graph-handler should, if possible, declared next to each other, so
unneccessary confusion can be avoided.

=item B<register_graph_handler> (I<$service>, I<$real_handler> [, I<$meta_handler>])

Registers a graph-handler for a service. A "real handler" is a handler for a
single RRD-file and is passed as a array-ref. The array is passed to
RRDs::graph mostly as-is, only occurences of literal B<{filename}> should be
substituted to the actual filename by the importing modules.

A meta service is a callback-function which, in whatever way, combined several
RRD-files in one graph. The callback is passed as code-ref. The arguments
passed to the callback-function are (I<$hostname>, I<$begin>, I<$end>,
I<@filenames>), where I<$begin> and I<$end> are seconds since the epoch.

=item B<service_to_oid> (I<$service>)

Returns the OID registered for this service or an empty string if there is no
such service.

=item B<run_input_callback> (I<$hostname>, I<$service>, I<$data>)

Runs the given service's callback function with the passed arguments. The
callback-function has to be registered with B<register_input_handler> before it
can be called.

=item B<run_create_callback> (I<$service>, I<$filename>)

Calls the callback function for service I<$service> which creates the RRD file
for this service. Returns true upon success, false otherwise.

=item B<run_graph_callback> (I<$service>, I<$hostname>, I<$begin>, I<$end>, I<@filenames>)

Calls the callback function registered with B<register_graph_handler>. See
there for a more detailed description.

=item B<get_graph_definition> (I<$service>)

Returns the RRD-arguments supplied for this service by a plugin. This is the
I<$real_handler> argument of B<register_graph_handler>. In scalar context
returns an arrayref, a list of scalars is return otherwise.

=item B<is_meta_service> (I<$service>)

Returns true if the given service is a "meta service" (a graph callback
exists). If this function returns true it's save to call B<run_graph_callback>.

=back

=head2 Exporter Tags

=over 4

=item :plugin

    register_input_handler
    register_graph_handler

=item :frontend

    run_input_callback
    run_graph_callback
    get_graph_definition
    is_meta_service

=back

=cut

@CapMan::Services::EXPORT_OK =
qw(
	load_plugins
	register_input_handler
	register_create_handler
	register_graph_handler
	service_to_oid
	run_input_callback
	run_create_callback
	run_graph_callback
	get_graph_definition
	is_meta_service
);
@CapMan::Services::ISA = qw(Exporter);
%CapMan::Services::EXPORT_TAGS = 
(
	plugin		=> [qw(register_input_handler register_create_handler
				register_graph_handler)],
	frontend	=> [qw(run_input_callback run_graph_callback
			get_graph_definition is_meta_service)]
);

return (1);

sub load_plugins ()
{
	my $dir = get_plugindir ();
	my $dh;
	my @files = ();

	$dir =~ s#/$##;

	die ("Plugin-directory $dir does not exist")     unless (-e $dir);
	die ("Plugin-directory $dir is not a directory") unless (-d $dir);

	opendir ($dh, $dir) or die ("opendir($dir): $!");
	while ($_ = readdir ($dh))
	{
		push (@files, $_) if (m/\.pm$/);
	}
	closedir ($dh);

	for (@files)
	{
		require "$dir/$_";
	}

	die ("No plugins found in dir $dir") unless (keys %InputCallbacks);
}


sub register_input_handler
{
	confess ("Wrong number of arguments") if (@_ != 3);

	my $service = shift;
	my $oid = shift;
	my $handler = shift;

	if (ref ($handler) ne 'CODE')
	{
		my $f = ref ($handler) || 'NOREF';
		cluck ("Illegal input handler for service $service: $f");
		return;
	}

	if (defined ($InputCallbacks{$service}))
	{
		warn ("More than one plugin tries to register input-handler for '$service'");
	}

	$InputCallbacks{$service} = $handler;
	$OIDMap{$service} = $oid;
}

sub register_create_handler ($$)
{
	my $service = shift;
	my $handler = shift;

	if (ref ($handler) ne 'CODE')
	{
		my $f = ref ($handler) || 'NOREF';
		cluck ("Illegal create handler for service $service: $f");
		return;
	}

	if (defined ($CreateCallbacks{$service}) and $::DEBUG)
	{
		print STDOUT "DEBUG: More than one plugin tries to register create-handler for '$service'";
	}

	$CreateCallbacks{$service} = $handler;
}

sub register_graph_handler ($$;$)
{
	my $service = shift;
	my $real_hd = shift;
	my $meta_hd = @_ ? shift : undef;

	if (ref ($real_hd) ne 'ARRAY')
	{
		my $f = ref ($real_hd) || 'NOREF';
		cluck ("Illegal graph handler for service $service: $real_hd ($f)");
		return;
	}

	if (defined ($meta_hd) and ref ($meta_hd) ne 'CODE')
	{
		my $f = ref ($meta_hd) || 'NOREF';
		cluck ("Illegal graph handler for meta service $service: $meta_hd ($f)");
		$meta_hd = undef;
	}

	if (defined ($GraphReal{$service}) and $::DEBUG)
	{
		print STDOUT "DEBUG: More than one plugin tries to register graph-handler for '$service'\n";
	}

	$GraphReal{$service} = $real_hd;
	$GraphMeta{$service} = $meta_hd if (defined ($meta_hd));
}

sub service_to_oid ($)
{
	my $service = shift;

	if (!defined ($OIDMap{$service}))
	{
		return ('');
	}

	return ($OIDMap{$service});
}

sub run_input_callback
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $service = shift;
	my $data = shift;

	if (!defined ($InputCallbacks{$service}))
	{
		cluck ("No input callback registered for service '$service'");
		return;
	}

	$InputCallbacks{$service}->($customer, $host, $service, $data);
}

sub run_create_callback
{
	my $service = shift;
	my $filename = shift;

	if (!defined ($CreateCallbacks{$service}))
	{
		print STDERR "ERROR: No create handler registered for service ``$service''\n";
		return (0);
	}

	return ($CreateCallbacks{$service}->($filename));
}

sub run_graph_callback
{
	if (@_ < 5) { confess ("Not enough arguments"); }

	my $service = shift;
	my @args = @_;

	if (!defined ($GraphMeta{$service}))
	{
		cluck ("No graph callback registered for service '$service'");
		return;
	}

	$GraphMeta{$service}->(@args);
}

sub get_graph_definition ($)
{
	my $service = shift;

	if (!defined ($GraphReal{$service}))
	{
		cluck ("No graph definition exists for service '$service'");
		return (wantarray ? () : []);
	}

	return (wantarray ? @{$GraphReal{$service}} : $GraphReal{$service});
}

sub is_meta_service ($)
{
	my $service = shift;

	return (defined ($GraphMeta{$service}) ? 1 : 0);
}

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
