package CapMan::Plugins::WuT;

use strict;
use warnings;

=head1 NAME

CapMan::Plugins::WuT - CapMan plugin for Wiesemann & Theis web-thermometers

=head1 DESCRIPTION

W&T build a series of web-thermometers with 1, 2 or 8 ports. These ports can be
queried via SNMP which is what this plugin does. 

=cut

use CapMan::Config qw(:rrdcreate);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd);

=head1 SERVICES PROVIDED

=over 4

=item wut_io12

Handler for twelve port web-io (57630 and 57631)

=item wut_io24

Handler for twenty-four port web-io (57632)

=item wut_thermo1

Handler for one port web-thermometers

=item wut_thermo2

Handler for two port web-thermometers

=item wut_thermo8

Handler for eight port web-thermometers

=item wut_an1graph

Handler for Web-Thermograph NTC/Pt100 (57610)

=item wut_an2graph

Handler for Web-IO 2x Thermometer NTC/Pt100 (57607)

=item wut_an8graph

Handler for Web-IO 8x Thermometer NTC/Pt100 (57608)

=back

=cut

register_input_handler ('wut_thermo8',  '.1.3.6.1.4.1.5040.1.2.1', \&wut_thermo8_handler);
register_input_handler ('wut_thermo2',  '.1.3.6.1.4.1.5040.1.2.2', \&wut_thermo2_handler);
register_input_handler ('wut_thermo1',  '.1.3.6.1.4.1.5040.1.2.3', \&wut_thermo1_handler);
register_input_handler ('wut_io12',     '.1.3.6.1.4.1.5040.1.2.4', \&wut_io12_handler);
register_input_handler ('wut_io24',     '.1.3.6.1.4.1.5040.1.2.5', \&wut_io24_handler);
register_input_handler ('wut_i12o6',    '.1.3.6.1.4.1.5040.1.2.19', \&wut_i12o6_handler);
register_input_handler ('wut_an8graph', '.1.3.6.1.4.1.5040.1.2.6', \&wut_an8graph_handler);
register_input_handler ('wut_an2graph', '.1.3.6.1.4.1.5040.1.2.7', \&wut_an2graph_handler);
register_input_handler ('wut_an1graph', '.1.3.6.1.4.1.5040.1.2.8', \&wut_an1graph_handler);

return (1);

sub wut_thermo8_handler
{
	wut_thermo_handler (@_, '.1.3.6.1.4.1.5040.1.2.1');
}

sub wut_thermo2_handler
{
	wut_thermo_handler (@_, '.1.3.6.1.4.1.5040.1.2.2');
}

sub wut_thermo1_handler
{
	wut_thermo_handler (@_, '.1.3.6.1.4.1.5040.1.2.3');
}

sub wut_an8graph_handler
{
	wut_thermo_handler (@_, '.1.3.6.1.4.1.5040.1.2.6');
}

sub wut_an2graph_handler
{
	wut_thermo_handler (@_, '.1.3.6.1.4.1.5040.1.2.7');
}

sub wut_an1graph_handler
{
	wut_thermo_handler (@_, '.1.3.6.1.4.1.5040.1.2.8');
}

sub wut_thermo_handler
{
	confess ("Wrong number of arguments") if (@_ != 5);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $data = shift;
	my $prefix = shift;

	my $i;
	my $hash = {};
	my %temp = ();
	
	my @names  = ();
	my @values = ();

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;

		if (!defined ($hash->{$name}))
		{
			$hash->{$name} = [];
		}

		$hash->{$name}[$seq] = $val;
	}

	if (!defined ($hash->{"$prefix.3.2.1.1.2"}))
	{
		print STDERR "Unable to get RMS-temp-name(s) from $host\n";
		return;
	}
	@names = @{$hash->{"$prefix.3.2.1.1.2"}};

	if (!defined ($hash->{"$prefix.1.4.1.1"}))
	{
		print STDERR "Unable to get RMS-temp-value(s) from $host\n";
		return;
	}
	@values = @{$hash->{"$prefix.1.4.1.1"}};

	for ($i = 0; $i <= $#names; $i++)
	{
		next unless (defined ($names[$i]));
		next unless (defined ($values[$i]));

		$_ = $names[$i];
		$temp{$_} = int ($values[$i]) / 10;
	}

	for (keys (%temp))
	{
		my $name = $_;
		my $temp = $temp{$name};

		next if (($temp < -50) or ($temp > 100));

		write_to_rrd ($customer, $host, 'temperature', $name, $temp);
	}
}

sub wut_io12_handler
{
	wut_io_handler (@_, '.1.3.6.1.4.1.5040.1.2.4');
}

sub wut_io24_handler
{
	wut_io_handler (@_, '.1.3.6.1.4.1.5040.1.2.5');
}

sub wut_i12o6_handler
{
	wut_io_handler (@_, '.1.3.6.1.4.1.5040.1.2.19');
}

sub wut_io_handler
{
	confess ("Wrong number of arguments") if (@_ != 5);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $data = shift;
	my $prefix = shift;

	my $i;
	my $hash = {};
	my %temp = ();

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;

		if (!defined ($hash->{$name}))
		{
			$hash->{$name} = [];
		}

		$hash->{$name}[$seq] = $val;
	}

	if ($hash->{"$prefix.1.1"})
	{
		my $input_num = $hash->{"$prefix.1.1"}->[0];
		my @input_names = ();
		my $input_values;

		if (defined ($hash->{"$prefix.3.2.1.1.2"}))
		{
			@input_names = @{$hash->{"$prefix.3.2.1.1.2"}};
		}
		else
		{
			@input_names = map { "input$_" } (0 .. $input_num);
		}

		if (!defined ($hash->{"$prefix.1.4"}))
		{
			print STDERR "Unable to get RMS-Input-Value(s) from $host\n";
			return;
		}
		$input_values = $hash->{"$prefix.1.4"}->[0];

		for (my $i = 0; $i < $input_num; $i++)
		{
			my $value = ($input_values >> $i) & 0x00000001;
			my $name = $input_names[$i + 1];
			
			next unless ($name);

			write_to_rrd ($customer, $host, 'binary', "Input $name", $value);
		}
	}

	if ($hash->{"$prefix.1.2"})
	{
		my $output_num = $hash->{"$prefix.1.2"}->[0];
		my @output_names = ();
		my $output_values;

		if (defined ($hash->{"$prefix.3.2.2.1.2"}))
		{
			@output_names = @{$hash->{"$prefix.3.2.2.1.2"}};
		}
		else
		{
			@output_names = map { "output$_" } (0 .. $output_num);
		}

		if (!defined ($hash->{"$prefix.1.6"}))
		{
			print STDERR "Unable to get RMS-Output-Value(s) from $host\n";
			return;
		}
		$output_values = $hash->{"$prefix.1.6"}->[0];

		for (my $i = 0; $i < $output_num; $i++)
		{
			my $value = ($output_values >> $i) & 0x00000001;
			my $name = $output_names[$i + 1];

			next unless ($name);

			write_to_rrd ($customer, $host, 'binary', "Output $name", $value);
		}
	}
}

=head1 SEE ALSO

L<http://www.wut.de/>, L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
