package CapMan::Plugins::UPS;

use strict;
use warnings;

=head1 NAME

CapMan::Plugins::UPS - CapMan plugin for UPSes

=head1 DESCRIPTION

This plugin queries various information from UPSes using the UPS-MIB. All
conforming UPSes should work..

=cut

use RRDs;

use CapMan::Config qw(:rrdcreate get_rrdpath);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd create_rrdfile get_n_colors);

our $GraphDefCharge = ['-v', 'Percent',
	'DEF:p={filename}:percentage:AVERAGE',
	'LINE1:p#000000:Charged',
	'GPRINT:p:AVERAGE:%5.1lf%% Average,',
	'GPRINT:p:MIN:%5.1lf%% Min,',
	'GPRINT:p:MAX:%5.1lf%% Max,',
	'GPRINT:p:LAST:%5.1lf%% Last'];

our $GraphDefDuration = ['-v', 'Minutes',
	'DEF:d={filename}:duration:AVERAGE',
	'LINE1:d#000000:Duration [min]',
	'GPRINT:d:AVERAGE:%4.1lf Average,',
	'GPRINT:d:MIN:%4.1lf Min,',
	'GPRINT:d:MAX:%4.1lf Max,',
	'GPRINT:d:LAST:%4.1lf Last'];

our $GraphDefStatus = ['-v', 'State',
	'DEF:d={filename}:status:AVERAGE',
	'LINE1:d#000000:Status  (2 = good, 1 = bad, 0 = nightmare)'];

our $GraphDefUtilisation = ['-v', 'Percent',
	'DEF:p={filename}:percentage:AVERAGE',
	'LINE1:p#000000:Utilisation',
	'GPRINT:p:AVERAGE:%5.1lf%% Average,',
	'GPRINT:p:MIN:%5.1lf%% Min,',
	'GPRINT:p:MAX:%5.1lf%% Max,',
	'GPRINT:p:LAST:%5.1lf%% Last'];

=head1 SERVICES PROVIDED

=over 4

=item ups-mib

Provides information about input/output voltage and current as well as battery
duration, system utilisation and system status.

=back

=cut

register_input_handler ('ups-mib', '.1.3.6.1.2.1.33.1', \&power_handler);

register_create_handler ('duration',    \&duration_create_handler);
register_create_handler ('percent',     \&percentage_create_handler);
register_create_handler ('charge',      \&percentage_create_handler);
register_create_handler ('upsstatus',   \&status_create_handler);
register_create_handler ('utilisation', \&percentage_create_handler);

register_graph_handler ('duration',    $GraphDefDuration);
register_graph_handler ('percent',     $GraphDefCharge);
register_graph_handler ('charge',      $GraphDefCharge);
register_graph_handler ('upsstatus',   $GraphDefStatus);
register_graph_handler ('utilisation', $GraphDefUtilisation, \&print_meta_graph_utilisation);

return (1);

sub power_handler ($$$)
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $data = shift;
	my %d = ();

	my %tt =
	(
		'2.1.0'		=> 'status_battery',
		'2.3.0'		=> 'duration_battery',
		'2.4.0'		=> 'charge_battery',
		'2.5.0'		=> 'voltage_battery',
		'2.6.0'		=> 'current_battery',
		'2.7.0'		=> 'temperature_battery',

		'3.2.0'		=> 'number_input',
		'3.3.1.2'	=> 'frequency_input',    # array
		'3.3.1.3'	=> 'voltage_input',      # array
		'3.3.1.4'	=> 'current_input',      # array
		'3.3.1.5'	=> 'power_input',        # array

		'4.2.0'		=> 'frequency_output',
		'4.3.0'		=> 'number_output',
		'4.4.1.2'	=> 'voltage_output',     # array
		'4.4.1.3'	=> 'current_output',     # array
		'4.4.1.4'	=> 'power_output',       # array
		'4.4.1.5'	=> 'utilisation_output', # array

		'5.1.0'		=> 'frequency_bypass',
		'5.2.0'		=> 'number_bypass',
		'5.3.1.2'	=> 'voltage_bypass',     # array
		'5.3.1.3'	=> 'current_bypass',     # array
		'5.3.1.4'	=> 'power_bypass'        # array

	);
	
	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;

		$name =~ s/^\.1\.3\.6\.1\.2\.1\.33\.1\.//;

		if (defined ($tt{"$name.$seq"}))
		{
			$name = $tt{"$name.$seq"};
			$d{$name} = $val;
		}
		elsif (defined ($tt{$name}))
		{
			$name = $tt{$name};
			$d{$name} = [] unless (defined ($d{$name}));
			$d{$name}->[$seq] = $val;
		}
	}

	if ($::DEBUG)
	{
		for (sort (keys (%d)))
		{
			my $key = $_;
			my $val = $d{$key};

			if (ref ($val))
			{
				print "DEBUG: $key = [", join (', ', map { defined ($_) ? $_ : 'NULL' } (@$val)), "]\n";
			}
			else
			{
				print "DEBUG: $key = $val\n";
			}
		}
	}

	if (defined ($d{'status_battery'}))
	{
		my $status = 4 - $d{'status_battery'};

		# 1 == unknown
		# 2 == good, 3 == bad, 4 == panic
		if ($status >= 0 and $status <= 2)
		{
			write_to_rrd ($customer, $host, 'upsstatus', '', $status);
		}
	}

	if (defined ($d{'duration_battery'}))
	{
		write_to_rrd ($customer, $host, 'duration', 'Battery', $d{'duration_battery'});
	}

	if (defined ($d{'charge_battery'}))
	{
		write_to_rrd ($customer, $host, 'charge', 'Battery', $d{'charge_battery'});
	}

	if (defined ($d{'voltage_battery'}))
	{
		write_to_rrd ($customer, $host, 'voltage', 'Battery', $d{'voltage_battery'});
	}

	if (defined ($d{'current_battery'}))
	{
		write_to_rrd ($customer, $host, 'current', 'Battery', $d{'current_battery'});
	}

	if (defined ($d{'voltage_battery'}) and defined ($d{'current_battery'}))
	{
		my $power = $d{'voltage_battery'} * $d{'current_battery'};
		write_to_rrd ($customer, $host, 'power', 'Battery', $power);
	}

	if (defined ($d{'temperature_battery'}))
	{
		write_to_rrd ($customer, $host, 'temperature', 'System', $d{'temperature_battery'});
	}

	if (defined ($d{'number_input'}))
	{
		my $i;
		for ($i = 1; $i <= $d{'number_input'}; $i++)
		{
			my $temp_current = 0;
			my $temp_voltage = 0;

			if (defined ($d{'frequency_input'}->[$i]))
			{
				write_to_rrd ($customer, $host, 'frequency', "Input $i", $d{'frequency_input'}->[$i] / 10);
			}
			if (defined ($d{'voltage_input'}->[$i]))
			{
				$temp_voltage = $d{'voltage_input'}->[$i];
				next unless ($temp_voltage);
				write_to_rrd ($customer, $host, 'voltage', "Input $i", $d{'voltage_input'}->[$i]);
			}
			if (defined ($d{'current_input'}->[$i]))
			{
				$temp_current = $d{'current_input'}->[$i];
				write_to_rrd ($customer, $host, 'current', "Input $i", $d{'current_input'}->[$i]);
			}

			if (defined ($d{'power_input'}->[$i]))
			{
				my $power = $d{'power_input'}->[$i];
				$power = $temp_current * $temp_voltage unless ($power);

				write_to_rrd ($customer, $host, 'power', "Input $i", $power);
			}
			elsif (defined ($d{'voltage_input'}->[$i]) and defined ($d{'current_input'}->[$i]))
			{
				my $power = $temp_current * $temp_voltage;
				write_to_rrd ($customer, $host, 'power', "Input $i", $power);
			}
		}
	}

	if (defined ($d{'frequency_output'}))
	{
		write_to_rrd ($customer, $host, 'frequency', 'Output', $d{'frequency_output'} / 10);
	}

	if (defined ($d{'number_output'}))
	{
		my $i;
		for ($i = 1; $i <= $d{'number_output'}; $i++)
		{
			my $temp_current = 0;
			my $temp_voltage = 0;

			if (defined ($d{'voltage_output'}->[$i]))
			{
				$temp_voltage = $d{'voltage_output'}->[$i];
				next unless ($temp_voltage);
				write_to_rrd ($customer, $host, 'voltage', "Output $i", $d{'voltage_output'}->[$i]);
			}
			if (defined ($d{'current_output'}->[$i]))
			{
				$temp_current = $d{'current_output'}->[$i];
				write_to_rrd ($customer, $host, 'current', "Output $i", $d{'current_output'}->[$i]);
			}
			if (defined ($d{'utilisation_output'}->[$i]))
			{
				write_to_rrd ($customer, $host, 'utilisation', "Output $i", $d{'utilisation_output'}->[$i]);
			}

			if (defined ($d{'power_output'}->[$i]))
			{
				my $power = $d{'power_output'}->[$i];
				$power = $temp_current * $temp_voltage unless ($power);

				write_to_rrd ($customer, $host, 'power', "Output $i", $power);
			}
			elsif (defined ($d{'voltage_output'}->[$i]) and defined ($d{'current_output'}->[$i]))
			{
				my $power = $temp_current * $temp_voltage;
				write_to_rrd ($customer, $host, 'power', "Output $i", $power);
			}
		}
	}

	if (defined ($d{'frequency_bypass'}))
	{
		write_to_rrd ($customer, $host, 'frequency', 'Bypass', $d{'frequency_bypass'} / 10);
	}

	if (defined ($d{'number_bypass'}))
	{
		my $i;
		for ($i = 1; $i <= $d{'number_bypass'}; $i++)
		{
			my $temp_current = 0;
			my $temp_voltage = 0;

			if (defined ($d{'voltage_bypass'}->[$i]))
			{
				$temp_voltage = $d{'voltage_bypass'}->[$i];
				next unless ($temp_voltage);
				write_to_rrd ($customer, $host, 'voltage', "Bypass $i", $d{'voltage_bypass'}->[$i]);
			}
			if (defined ($d{'current_bypass'}->[$i]))
			{
				$temp_current = $d{'current_bypass'}->[$i];
				write_to_rrd ($customer, $host, 'current', "Bypass $i", $d{'current_bypass'}->[$i]);
			}
			if (defined ($d{'power_bypass'}->[$i]))
			{
				my $power = $d{'power_bypass'}->[$i];
				$power = $temp_current * $temp_voltage unless ($power);

				write_to_rrd ($customer, $host, 'power', "Bypass $i", $power);
			}
			elsif (defined ($d{'voltage_bypass'}->[$i]) and defined ($d{'current_bypass'}->[$i]))
			{
				my $power = $temp_current * $temp_voltage;
				write_to_rrd ($customer, $host, 'power', "Bypass $i", $power);
			}
		}
	}
}

sub duration_create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:duration:GAUGE:$HeartBeat:0:U");
}

sub percentage_create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:percentage:GAUGE:$HeartBeat:0:100");
}

sub status_create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:status:GAUGE:$HeartBeat:0:2");
}

sub temperature_create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:temp:GAUGE:$HeartBeat:-273:U");
}

sub print_meta_graph_utilisation
{
	confess ("Wrong number of arguments") if (@_ < 5);

	my $customer = shift;
	my $host = shift;
	my $begin = shift;
	my $end = shift;
	my @files = @_;
	my @names = ();
	
	my @colors = get_n_colors (scalar (@files));
	
	my @cmd = ('-', '-s', $begin, '-e', $end, '-a', 'PNG', '-t', "Utilisation of $host", '-v', 'Percent');
	my $rrdpath = get_rrdpath ();

	my $num = 0;

	for (@files)
	{
		my $file = "$rrdpath/$customer/$host/$_";
		my $name = (split (m/-/, $_))[1];

		$name =~ s/\.rrd$//;
		$name =~ s/_+/ /g;

		push (@cmd, qq(DEF:val$num=$file:percentage:AVERAGE));
		push (@names, $name);
		$num++;
	}

	for (my $i = 0; $i < $num; $i++)
	{
		my $line = $names[$i];
		if ($line !~ m/\w/)
		{
			$line = "Line $line";
		}
		
		my $color = $colors[$i % scalar (@colors)];
		
		push (@cmd, 'LINE1:val' . $i . '#' . $color . ":$line");
		push (@cmd, "GPRINT:val$i:AVERAGE:\%5.1lf%% Average,",
			"GPRINT:val$i:MIN:\%5.1lf%% Min,",
			"GPRINT:val$i:MAX:\%5.1lf%% Max,",
			"GPRINT:val$i:LAST:\%5.1lf%% Last\\n");
	}

	RRDs::graph (@cmd);

	die (RRDs::error ()) if (RRDs::error ());
}

=head1 SEE ALSO

L<CapMan::Services>, L<http://www.aps-ups.de/>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
