package CapMan::Plugins::SysStat;

use strict;
use warnings;
#Use of implicit split to @_ is deprecated at /home/octo/capman/lib/CapMan/Plugins/SysStat.pm line 192.
#no warnings qw(deprecated);

=head1 NAME

CapMan::Plugins::SysStat - CapMan plugin which queries detailed Linux
statistics.

=head1 DESCRIPTION

This plugin queries numberous statistics provided by the Linux kernel. To use
this plugin you have to install a script on the host being queried. The
following information is collected:

    CPU-Usage
    Disk-IO
    Swap- and Page-IO
    Interrupts
    Forkrate
    Contextswitches
    Memory-Information

=cut

use RRDs;

use CapMan::Config qw(:rrdcreate get_rrdpath get_host_config);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd create_rrdfile get_n_colors);


our $GraphDefMeminfo = ['-b', '1024', '-l', '0', '-v', 'Bytes',
	'DEF:used={filename}:used:AVERAGE',
	'DEF:buffer={filename}:buffer:AVERAGE',
	'DEF:page={filename}:page:AVERAGE',
	'DEF:free={filename}:free:AVERAGE',
	'AREA:used#0000FF:Used        ',
	'GPRINT:used:AVERAGE:%5.1lf%sByte Average,',
	'GPRINT:used:MAX:%5.1lf%sByte Max,',
	'GPRINT:used:LAST:%5.1lf%sByte Last\n',
	'STACK:buffer#FFFF00:Buffer cache',
	'GPRINT:buffer:AVERAGE:%5.1lf%sByte Average,',
	'GPRINT:buffer:MAX:%5.1lf%sByte Max,',
	'GPRINT:buffer:LAST:%5.1lf%sByte Last\n',
	'STACK:page#FF0000:Page cache  ',
	'GPRINT:page:AVERAGE:%5.1lf%sByte Average,',
	'GPRINT:page:MAX:%5.1lf%sByte Max,',
	'GPRINT:page:LAST:%5.1lf%sByte Last\n',
	'STACK:free#00FF00:Free        ',
	'GPRINT:free:AVERAGE:%5.1lf%sByte Average,',
	'GPRINT:free:MAX:%5.1lf%sByte Max,',
	'GPRINT:free:LAST:%5.1lf%sByte Last'];

our $GraphDefCPU24 = ['-v', 'Percent', '-u', '100',
	'DEF:ruser={filename}:user:AVERAGE',
	'DEF:rnice={filename}:nice:AVERAGE',
	'DEF:rsyst={filename}:syst:AVERAGE',
	'DEF:ridle={filename}:idle:AVERAGE',
	'CDEF:duser=ruser,UN,0,ruser,IF',
	'CDEF:dnice=rnice,UN,0,rnice,IF',
	'CDEF:dsyst=rsyst,UN,0,rsyst,IF',
	'CDEF:didle=ridle,UN,0,ridle,IF',
	'CDEF:total=duser,dnice,+,dsyst,+,didle,+',
	'CDEF:puser=ruser,100,*,total,/',
	'CDEF:pnice=rnice,100,*,total,/',
	'CDEF:psyst=rsyst,100,*,total,/',
	'CDEF:pidle=ridle,100,*,total,/',
	'HRULE:100#FF0000',
	'AREA:psyst#FF0000:System   ',
	'GPRINT:psyst:AVERAGE:%4.1lf%% Average,',
	'GPRINT:psyst:MAX:%4.1lf%% Max,',
	'GPRINT:psyst:LAST:%4.1lf%% Last\l',
	'STACK:puser#0000FF:User     ',
	'GPRINT:puser:AVERAGE:%4.1lf%% Average,',
	'GPRINT:puser:MAX:%4.1lf%% Max,',
	'GPRINT:puser:LAST:%4.1lf%% Last\l',
	'STACK:pnice#00FF00:Nice     ',
	'GPRINT:pnice:AVERAGE:%4.1lf%% Average,',
	'GPRINT:pnice:MAX:%4.1lf%% Max,',
	'GPRINT:pnice:LAST:%4.1lf%% Last\l',
	'STACK:pidle#FFFFFF:Idle     ',
	'GPRINT:pidle:AVERAGE:%4.1lf%% Average,',
	'GPRINT:pidle:MAX:%4.1lf%% Max,',
	'GPRINT:pidle:LAST:%4.1lf%% Last\l'];

our $GraphDefCPU26 = ['-v', 'Percent', '-u', '100',
	'DEF:ruser={filename}:user:AVERAGE',
	'DEF:rnice={filename}:nice:AVERAGE',
	'DEF:rsyst={filename}:syst:AVERAGE',
	'DEF:ridle={filename}:idle:AVERAGE',
	'DEF:rwait={filename}:wait:AVERAGE',
	'DEF:rintr={filename}:intr:AVERAGE',
	'DEF:rsitr={filename}:sitr:AVERAGE',
	'DEF:rstel={filename}:stel:AVERAGE',
	'CDEF:duser=ruser,UN,0,ruser,IF',
	'CDEF:dnice=rnice,UN,0,rnice,IF',
	'CDEF:dsyst=rsyst,UN,0,rsyst,IF',
	'CDEF:didle=ridle,UN,0,ridle,IF',
	'CDEF:dwait=rwait,UN,0,rwait,IF',
	'CDEF:dintr=rintr,UN,0,rintr,IF',
	'CDEF:dsitr=rsitr,UN,0,rsitr,IF',
	'CDEF:dstel=rstel,UN,0,rstel,IF',
	'CDEF:total=duser,dnice,+,dsyst,+,didle,+,dwait,+,dintr,+,dsitr,+,dstel,+',
	'CDEF:puser=ruser,100,*,total,/',
	'CDEF:pnice=rnice,100,*,total,/',
	'CDEF:psyst=rsyst,100,*,total,/',
	'CDEF:pidle=ridle,100,*,total,/',
	'CDEF:pwait=rwait,100,*,total,/',
	'CDEF:pintr=rintr,100,*,total,/',
	'CDEF:psitr=rsitr,100,*,total,/',
	'CDEF:pstel=rstel,100,*,total,/',
	'HRULE:100#FF0000',
	'AREA:pstel#000000:Steal    ',
	'GPRINT:pstel:AVERAGE:%4.1lf%% Average,',
	'GPRINT:pstel:MAX:%4.1lf%% Max,',
	'GPRINT:pstel:LAST:%4.1lf%% Last\l',
	'STACK:pintr#A000A0:Interrupt',
	'GPRINT:pintr:AVERAGE:%4.1lf%% Average,',
	'GPRINT:pintr:MAX:%4.1lf%% Max,',
	'GPRINT:pintr:LAST:%4.1lf%% Last\l',
	'STACK:psitr#FF00FF:Soft-IRQ ',
	'GPRINT:psitr:AVERAGE:%4.1lf%% Average,',
	'GPRINT:psitr:MAX:%4.1lf%% Max,',
	'GPRINT:psitr:LAST:%4.1lf%% Last\l',
	'STACK:psyst#FF0000:System   ',
	'GPRINT:psyst:AVERAGE:%4.1lf%% Average,',
	'GPRINT:psyst:MAX:%4.1lf%% Max,',
	'GPRINT:psyst:LAST:%4.1lf%% Last\l',
	'STACK:pwait#A0A000:IO-Wait  ',
	'GPRINT:pwait:AVERAGE:%4.1lf%% Average,',
	'GPRINT:pwait:MAX:%4.1lf%% Max,',
	'GPRINT:pwait:LAST:%4.1lf%% Last\l',
	'STACK:puser#0000FF:User     ',
	'GPRINT:puser:AVERAGE:%4.1lf%% Average,',
	'GPRINT:puser:MAX:%4.1lf%% Max,',
	'GPRINT:puser:LAST:%4.1lf%% Last\l',
	'STACK:pnice#00FF00:Nice     ',
	'GPRINT:pnice:AVERAGE:%4.1lf%% Average,',
	'GPRINT:pnice:MAX:%4.1lf%% Max,',
	'GPRINT:pnice:LAST:%4.1lf%% Last\l',
	'STACK:pidle#FFFFFF:Idle     ',
	'GPRINT:pidle:AVERAGE:%4.1lf%% Average,',
	'GPRINT:pidle:MAX:%4.1lf%% Max,',
	'GPRINT:pidle:LAST:%4.1lf%% Last\l'];

our $GraphDefDiskIO = ['-v', 'Operations/Blocks',
	'DEF:ori={filename}:opts_read:AVERAGE',
	'DEF:owi={filename}:opts_wrte:AVERAGE',
	'DEF:bri={filename}:blks_read:AVERAGE',
	'DEF:bwi={filename}:blks_wrte:AVERAGE',
	'CDEF:oro=ori,-1,*',
	'CDEF:owo=owi,-1,*',
	'AREA:bwi#FF0000:Write',
	'GPRINT:bwi:AVERAGE:%6.1lf Blk/s Average,',
	'GPRINT:bwi:MAX:%6.1lf Blk/s Max,',
	'GPRINT:bwi:LAST:%6.1lf Blk/s Last\n',
	'AREA:owo#FF0000:     ',
	'GPRINT:owi:AVERAGE:%6.1lf Ops/s Average,',
	'GPRINT:owi:MAX:%6.1lf Ops/s Max,',
	'GPRINT:owi:LAST:%6.1lf Ops/s Last\n',
	'HRULE:0#000000',
	'LINE1:bri#0000FF:Read ',
	'GPRINT:bri:AVERAGE:%6.1lf Blk/s Average,',
	'GPRINT:bri:MAX:%6.1lf Blk/s Max,',
	'GPRINT:bri:LAST:%6.1lf Blk/s Last\n',
	'LINE1:oro#0000FF:     ',
	'GPRINT:ori:AVERAGE:%6.1lf Ops/s Average,',
	'GPRINT:ori:MAX:%6.1lf Ops/s Max,',
	'GPRINT:ori:LAST:%6.1lf Ops/s Last'];

our $GraphDefIO = ['-v', 'Pages/s',
	'DEF:o={filename}:outgoing:AVERAGE',
	'DEF:i={filename}:incoming:AVERAGE',
	'AREA:o#FF0000:Swapout',
	'GPRINT:o:AVERAGE:%6.1lf Average,',
	'GPRINT:o:MAX:%6.1lf Max,',
	'GPRINT:o:LAST:%6.1lf Last\n',
	'LINE1:i#0000FF:Swapin ',
	'GPRINT:i:AVERAGE:%6.1lf Average,',
	'GPRINT:i:MAX:%6.1lf Max,',
	'GPRINT:i:LAST:%6.1lf Last'];

our $GraphDefCntxt = ['-v', 'Contextswitches/s',
	'DEF:v={filename}:value:AVERAGE',
	'AREA:v#00FF00:Contextswitches/s',
	'GPRINT:v:AVERAGE:%5.1lf Average,',
	'GPRINT:v:MAX:%5.1lf Max,',
	'GPRINT:v:LAST:%5.1lf Last'];

our $GraphDefIntr = ['-v', 'Interrupts/s',
	'DEF:v={filename}:value:AVERAGE',
	'AREA:v#00FF00:Interrupts/s',
	'GPRINT:v:AVERAGE:%5.1lf Average,',
	'GPRINT:v:MAX:%5.1lf Max,',
	'GPRINT:v:LAST:%5.1lf Last'];

our $GraphDefFork = ['-v', 'forks/s',
	'DEF:v={filename}:value:AVERAGE',
	'AREA:v#00FF00:Forks/s',
	'GPRINT:v:AVERAGE:%5.1lf Average,',
	'GPRINT:v:MAX:%5.1lf Max,',
	'GPRINT:v:LAST:%5.1lf Last'];

our $GraphDefBindStat = ['-v', 'Requests/s',
	'DEF:val={filename}:value:AVERAGE',
	'AREA:val#B7B7F7',
	'LINE1:val#0000FF:Requests/s',
	'GPRINT:val:AVERAGE:%5.1lf Average,',
	'GPRINT:val:MAX:%5.1lf Max,',
	'GPRINT:val:LAST:%5.1lf Last\l'];

our $GraphDefSwapinfo = ['-v', 'Bytes',
	'DEF:total={filename}:total:AVERAGE',
	'DEF:free={filename}:free:AVERAGE',
	'DEF:cached={filename}:cached:AVERAGE',
	'CDEF:used=total,free,-,cached,-',
	'CDEF:used_cached=used,cached,+',
	'AREA:total#B7EFB7',
	'AREA:used_cached#B7B7F7',
	'AREA:used#F7B7B7',
	'LINE1:total#00E000:Free      ',
	'GPRINT:free:AVERAGE:%6.2lf%s Average,',
	'GPRINT:free:MAX:%6.2lf%s Max,',
	'GPRINT:free:LAST:%6.2lf%s Last\l',
	'LINE1:used_cached#0000FF:Cached    ',
	'GPRINT:cached:AVERAGE:%6.2lf%s Average,',
	'GPRINT:cached:MAX:%6.2lf%s Max,',
	'GPRINT:cached:LAST:%6.2lf%s Last\l',
	'LINE1:used#FF0000:Used      ',
	'GPRINT:used:AVERAGE:%6.2lf%s Average,',
	'GPRINT:used:MAX:%6.2lf%s Max,',
	'GPRINT:used:LAST:%6.2lf%s Last\l'];

our $GraphIOBlocks = ['-v', 'Blocks/s',
	'DEF:o={filename}:outgoing:AVERAGE',
	'DEF:i={filename}:incoming:AVERAGE',
	'CDEF:overlap=o,i,GT,i,o,IF',
	'AREA:o#B7EFB7',
	'AREA:i#B7B7F7',
	'AREA:overlap#89B3C9',
	'LINE1:o#00E000:Written',
	'GPRINT:o:AVERAGE:%6.1lf Average,',
	'GPRINT:o:MAX:%6.1lf Max,',
	'GPRINT:o:LAST:%6.1lf Last\n',
	'LINE1:i#0000FF:Read   ',
	'GPRINT:i:AVERAGE:%6.1lf Average,',
	'GPRINT:i:MAX:%6.1lf Max,',
	'GPRINT:i:LAST:%6.1lf Last'];

our $GraphIOOps = ['-v', 'Operations/s',
	'DEF:o={filename}:outgoing:AVERAGE',
	'DEF:i={filename}:incoming:AVERAGE',
	'CDEF:overlap=o,i,GT,i,o,IF',
	'AREA:o#B7EFB7',
	'AREA:i#B7B7F7',
	'AREA:overlap#89B3C9',
	'LINE1:o#00E000:Write',
	'GPRINT:o:AVERAGE:%6.1lf Average,',
	'GPRINT:o:MAX:%6.1lf Max,',
	'GPRINT:o:LAST:%6.1lf Last\n',
	'LINE1:i#0000FF:Read ',
	'GPRINT:i:AVERAGE:%6.1lf Average,',
	'GPRINT:i:MAX:%6.1lf Max,',
	'GPRINT:i:LAST:%6.1lf Last'];

our $GraphIOTime = ['-v', 'Seconds',
	'DEF:ms_o={filename}:outgoing:AVERAGE',
	'DEF:ms_i={filename}:incoming:AVERAGE',
	'CDEF:o=ms_o,1000,/',
	'CDEF:i=ms_i,1000,/',
	'CDEF:overlap=o,i,GT,i,o,IF',
	'AREA:o#B7EFB7',
	'AREA:i#B7B7F7',
	'AREA:overlap#89B3C9',
	'LINE1:o#00E000:Write',
	'GPRINT:o:AVERAGE:%6.1lf Average,',
	'GPRINT:o:MAX:%6.1lf Max,',
	'GPRINT:o:LAST:%6.1lf Last\n',
	'LINE1:i#0000FF:Read ',
	'GPRINT:i:AVERAGE:%6.1lf Average,',
	'GPRINT:i:MAX:%6.1lf Max,',
	'GPRINT:i:LAST:%6.1lf Last'];

=head1 SERVICES PROVIDED

=over 4

=item sysstat

Run the remote script and get all of the above information.

=back

=cut

register_input_handler ('sysstat', '.1.3.6.1.4.1.2021.50.101', \&sysstat_handler);

register_create_handler ('contextswitches', \&create_general_counter_handler);
register_create_handler ('cpu',             \&sysstat_create_cpu_handler24);
register_create_handler ('cpu_linux26',     \&sysstat_create_cpu_handler26);
register_create_handler ('disk_io',         \&sysstat_create_diskio_handler);
register_create_handler ('forkrate',        \&create_general_counter_handler);
register_create_handler ('interrupts',      \&create_general_counter_handler);
register_create_handler ('io',              \&sysstat_create_general_io_handler);
register_create_handler ('meminfo',         \&sysstat_create_mem_handler);
register_create_handler ('bind_stat',            \&sysstat_create_bind_stat_handler);
register_create_handler ('swapinfo',        \&create_swapinfo_handler);
register_create_handler ('disk_blocks',     \&sysstat_create_general_io_handler);
register_create_handler ('disk_ops',        \&sysstat_create_general_io_handler);
register_create_handler ('disk_time',       \&sysstat_create_general_io_handler);
register_create_handler ('partition_blocks', \&sysstat_create_general_io_handler);
register_create_handler ('partition_ops',   \&sysstat_create_general_io_handler);

register_graph_handler ('bind_stat',       $GraphDefBindStat, \&print_meta_graph_bind_stat);
register_graph_handler ('contextswitches', $GraphDefCntxt);
register_graph_handler ('cpu',             $GraphDefCPU24, \&print_meta_graph_cpu24);
register_graph_handler ('cpu_linux26',     $GraphDefCPU26, \&print_meta_graph_cpu26);
register_graph_handler ('disk_io',         $GraphDefDiskIO);
register_graph_handler ('forkrate',        $GraphDefFork);
register_graph_handler ('interrupts',      $GraphDefIntr);
register_graph_handler ('io',              $GraphDefIO);
register_graph_handler ('meminfo',         $GraphDefMeminfo);
register_graph_handler ('swapinfo',        $GraphDefSwapinfo);
register_graph_handler ('disk_blocks',     $GraphIOBlocks);
register_graph_handler ('disk_ops',        $GraphIOOps);
register_graph_handler ('disk_time',       $GraphIOTime);
register_graph_handler ('partition_blocks', $GraphIOBlocks);
register_graph_handler ('partition_ops',   $GraphIOOps);

return (1);

sub sysstat_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $data = shift;

	my $sequence = '';
	
	my $stats = {};
	my %ids = ();

	my %owfs_sensor_names = ();
	for (get_host_config ($customer, $host, "owfs_sensor_name"))
	{
		my ($key, $value) = split ('=', $_, 2);
		next if (!$key || !$value);
		$owfs_sensor_names{$key} = $value;
	}

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;

		if ($val =~ m/^(\S+)=([\d,\.]+)$/)
		{
			$stats->{$1} = $2;
		}
	}

	for (sort (keys (%$stats)))
	{
		my $key = $_;

		if ($key =~ m/^cpu(\d*)$/)
		{
			write_to_rrd ($customer, $host, 'cpu', $1, split (m/,/, $stats->{$key}));
		}
		elsif ($key =~ m/^cpu_linux26-(\d+)$/)
		{
			my @values = split (m/,/, $stats->{$key});
			push (@values, 'U') if (@values == 7);
			write_to_rrd ($customer, $host, 'cpu_linux26', $1, @values);
		}
		elsif ($key =~ m/^disk_io_([\d,]+)$/)
		{
			my $hd = $1;
			$hd =~ tr/,/_/;

			write_to_rrd ($customer, $host, 'disk_io', $hd, split (m/,/, $stats->{$key}));
		}
		elsif ($key eq 'swap' or $key eq 'page')
		{
			write_to_rrd ($customer, $host, 'io', $key, split (m/,/, $stats->{$key}));
		}
		elsif ($key eq 'intr' or $key eq 'ctxt'
				or $key eq 'processes')
		{
			my $name = 'forkrate';

			$name = 'interrupts'      if ($key eq 'intr');
			$name = 'contextswitches' if ($key eq 'ctxt');
			
			write_to_rrd ($customer, $host, $name, '', $stats->{$key});
		}
		elsif ($key eq 'meminfo')
		{
			write_to_rrd ($customer, $host, 'meminfo', '', split (m/,/, $stats->{$key}));
		}
		elsif ($key eq 'mysql_queries')
		{
			write_to_rrd ($customer, $host, 'mysql', 'queries', $stats->{$key});
		}
		elsif ($key =~ m/^suncpu(\d*)$/)
		{
			write_to_rrd ($customer, $host, 'suncpu', $1, split (m/,/, $stats->{$key}));
		}
		elsif ($key =~ m/^bind_stat-([a-zA-Z]+)$/)
		{
			my $inst = $1;
			write_to_rrd ($customer, $host, 'bind_stat', $inst, $stats->{$key});
		}
		elsif ($key eq 'swapinfo')
		{
			write_to_rrd ($customer, $host, 'swapinfo', '', split (m/,/, $stats->{$key}));
		}
		elsif ($key =~ m/^temperature-(.*)$/)
		{
			my $instance = exists ($owfs_sensor_names{$1}) ? $owfs_sensor_names{$1} : $1;
			write_to_rrd ($customer, $host, 'temperature', $instance, $stats->{$key});
		}
		elsif ($key =~ m/^disk_([A-Za-z]+)$/)
		{
			my $disk = $1;
			my @fields = split (m/,/, $stats->{$key});
			if (@fields == 11)
			{
				write_to_rrd ($customer, $host, 'disk_blocks', $disk, @fields[2,6]);
				write_to_rrd ($customer, $host, 'disk_ops',    $disk, @fields[0,4]);
				write_to_rrd ($customer, $host, 'disk_time',   $disk, @fields[3,7]);
			}
		}
		elsif ($key =~ m/^partition_([A-Za-z0-9]+)$/)
		{
			my $disk = $1;
			my @fields = split (m/,/, $stats->{$key});
			if (@fields == 4)
			{
				write_to_rrd ($customer, $host, 'partition_blocks', $disk, @fields[0,2]);
				write_to_rrd ($customer, $host, 'partition_ops',    $disk, @fields[1,3]);
			}
		}
	}

	#print STDERR '', Data::Dumper->Dump ([$stats], ['stats']);
}

sub sysstat_create_cpu_handler24 ($)
{
	my $file = shift;

	create_rrdfile ($file, 
		"DS:user:COUNTER:$HeartBeat:0:110",
		"DS:nice:COUNTER:$HeartBeat:0:110",
		"DS:syst:COUNTER:$HeartBeat:0:110",
		"DS:idle:COUNTER:$HeartBeat:0:110"
	);
}

sub sysstat_create_cpu_handler26 ($)
{
	my $file = shift;

	create_rrdfile ($file, 
		"DS:user:COUNTER:$HeartBeat:0:110",
		"DS:nice:COUNTER:$HeartBeat:0:110",
		"DS:syst:COUNTER:$HeartBeat:0:110",
		"DS:idle:COUNTER:$HeartBeat:0:110",
		"DS:wait:COUNTER:$HeartBeat:0:110",
		"DS:intr:COUNTER:$HeartBeat:0:110",
		"DS:sitr:COUNTER:$HeartBeat:0:110",
		"DS:stel:COUNTER:$HeartBeat:0:110"
	);
}

sub sysstat_create_mem_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, 
		"DS:used:GAUGE:$HeartBeat:0:U",
		"DS:buffer:GAUGE:$HeartBeat:0:U",
		"DS:page:GAUGE:$HeartBeat:0:U",
		"DS:free:GAUGE:$HeartBeat:0:U"
	);
}

sub sysstat_create_diskio_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, 
		"DS:opts_ninf:COUNTER:$HeartBeat:0:U",
		"DS:opts_read:COUNTER:$HeartBeat:0:U",
		"DS:blks_read:COUNTER:$HeartBeat:0:U",
		"DS:opts_wrte:COUNTER:$HeartBeat:0:U",
		"DS:blks_wrte:COUNTER:$HeartBeat:0:U"
	);
}

sub sysstat_create_general_io_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, 
		"DS:incoming:COUNTER:$HeartBeat:0:U",
		"DS:outgoing:COUNTER:$HeartBeat:0:U"
	);
}

sub create_general_counter_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, 
		"DS:value:COUNTER:$HeartBeat:0:U"
	);
}

sub sysstat_create_bind_stat_handler ($)
{
	my $file = shift;
	create_rrdfile ($file, "DS:value:COUNTER:$HeartBeat:0:50000");
}

sub create_swapinfo_handler
{
	my $file = shift;

	create_rrdfile ($file,
		"DS:total:GAUGE:$HeartBeat:0:1099511627776",
		"DS:free:GAUGE:$HeartBeat:0:1099511627776",
		"DS:cached:GAUGE:$HeartBeat:0:1099511627776"
	);
}

sub print_meta_graph_cpu24
{
	my $dses =
	[
		[qw(syst user nice idle)],
		[qw(FF0000 0000FF 00FF00 FFFFFF)],
		['System', 'User', 'Nice', 'Idle']
	];
	print_meta_graph_cpu ($dses, @_);
}

sub print_meta_graph_cpu26
{
	my $dses =
	[
		[qw(stel intr sitr syst wait user nice idle)],
		[qw(000000 A000A0 FF00FF FF0000 E0E000 0000FF 00FF00 FFFFFF)],
		['Steal Time', 'Interrupt', 'Soft-IRQ', 'System', 'IO-Wait', 'User', 'Nice', 'Idle']
	];
	print_meta_graph_cpu ($dses, @_);
}

sub print_meta_graph_cpu
{
	confess ('Wrong number of arguments') if (@_ < 6);

	my $dses = shift;;
	my $customer = shift;
	my $host = shift;
	my $begin = shift;
	my $end = shift;
	my @files = grep { $_ ne 'cpu.rrd' } (@_);

	my $ds_names  = $dses->[0];
	my $ds_colors = $dses->[1];
	my $ds_descs  = $dses->[2];

	my @cmd = ('-', '-s', $begin, '-e', $end, '-a', 'PNG');

	my $rrdpath = get_rrdpath ();

	my $files_num = scalar (@files);
	my $ds_num    = scalar (@$ds_names);

	push (@cmd,
		'-v', 'Percent', '-u', '100',
		'-t', ($files_num > 1)
		? "Combined CPU Usage on $host ($files_num CPUs)"
		: "CPU Usage on $host");

	for (my $i = 0; $i < $files_num; $i++)
	{
		my $file = "$rrdpath/$customer/$host/" . $files[$i];

		for (my $j = 0; $j < $ds_num; $j++)
		{
			my $ds_name = $ds_names->[$j];
			push (@cmd, qq(DEF:r${ds_name}${i}=$file:${ds_name}:AVERAGE));
		}
	}

	for (my $i = 0; $i < $files_num; $i++)
	{
		for (my $j = 0; $j < $ds_num; $j++)
		{
			my $ds_name = $ds_names->[$j];
			push (@cmd, qq(CDEF:d${ds_name}${i}=r${ds_name}${i},UN,0,r${ds_name}${i},IF));
		}
	}

	for (my $j = 0; $j < $ds_num; $j++)
	{
		my $ds_name = $ds_names->[$j];
		my $cmd = "CDEF:t${ds_name}=d${ds_name}0";
		for (my $i = 1; $i < $files_num; $i++)
		{
			$cmd .= ",d${ds_name}${i},+";
		}
		push (@cmd, $cmd);
	}

	{
		my $cmd = 'CDEF:total=t' . $ds_names->[0];
		for (my $j = 1; $j < $ds_num; $j++)
		{
			my $ds_name = $ds_names->[$j];
			$cmd .= ",t${ds_name},+";
		}
		push (@cmd, $cmd);
	}

	for (my $j = 0; $j < $ds_num; $j++)
	{
		my $ds_name = $ds_names->[$j];
		push (@cmd, "CDEF:p${ds_name}=t${ds_name},100,*,total,/");
	}

	push (@cmd, 'HRULE:100#FF0000');

	for (my $j = 0; $j < $ds_num; $j++)
	{
		my $ds_name = $ds_names->[$j];
		my $ds_color = $ds_colors->[$j];
		my $ds_desc = sprintf ("%-10s", $ds_descs->[$j]);
		my $type = ($j == 0) ? "AREA" : "STACK";
		push (@cmd,
			"${type}:p${ds_name}#${ds_color}:${ds_desc}",
			"GPRINT:p${ds_name}:AVERAGE:%4.1lf%% Average,",
			"GPRINT:p${ds_name}:MAX:%4.1lf%% Max,",
			"GPRINT:p${ds_name}:LAST:%4.1lf%% Last\\l");
	}

	RRDs::graph (@cmd);
	die (RRDs::error ()) if (RRDs::error ());
}

sub print_meta_graph_bind_stat
{
	confess ('Wrong number of arguments') if (@_ < 5);

	my $customer = shift;
	my $host = shift;
	my $begin = shift;
	my $end = shift;
	my @files = grep { $_ ne 'cpu.rrd' } (@_);

	my @cmd = ('-', '-s', $begin, '-e', $end, '-a', 'PNG');

	my $rrdpath = get_rrdpath ();

	my $files_num = scalar (@files);
	my @colors = get_n_colors ($files_num);

	my @total = ();

	my @titles = ();
	my $titles_len = 0;
	my %sum_titles = ();
	my %valid_sum_titles = map { $_ => 1 } (qw(success referral nxrrset nxdomain failure));

	push (@cmd,
		'-v', 'Requests/s',
		'-t', ($files_num > 1)
		? "Combined Bind stats of $host"
		: "Bind stats of $host");

	for (my $i = 0; $i < $files_num; $i++)
	{
		my $file = "$rrdpath/$customer/$host/" . $files[$i];
		push (@cmd, qq(DEF:v$i=${file}:value:AVERAGE));
	}

	for (my $i = 0; $i < $files_num; $i++)
	{
		my $title = $files[$i];
		if ($title =~ m/bind_stat-([a-zA-Z]+)\.rrd$/)
		{
			$title = $1;
		}
		$titles[$i] = $title;
		if (length ($title) > $titles_len)
		{
			$titles_len = length ($title);
		}

		if (defined ($valid_sum_titles{$title}))
		{
			$sum_titles{$title} = $i;
		}
	}

	if (keys (%sum_titles) > 1)
	{
		my @rest = sort (values %sum_titles);
		my $first = shift (@rest);
		my $cdef = "CDEF:v$files_num=v$first" . join ('', map { ",v$_,+" } (@rest));
		push (@cmd, $cdef);
		push (@cmd, "AREA:v$files_num#e0e0e0");
		$colors[$files_num] = '404040';
		$titles[$files_num] = 'total';
		$files_num++;
	}

	for (my $i = 0; $i < $files_num; $i++)
	{
		my $color = $colors[$i];
		my $title = sprintf ('%-*s', $titles_len, $titles[$i]);

		push (@cmd,
			"LINE1:v${i}#${color}:${title}",
			"GPRINT:v${i}:AVERAGE:%6.1lf Average,",
			"GPRINT:v${i}:MAX:%6.1lf Max,",
			"GPRINT:v${i}:LAST:%6.1lf Last\\l");
	}

	RRDs::graph (@cmd);
	die (RRDs::error ()) if (RRDs::error ());
}

=head1 SEE ALSO

L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
