package CapMan::Plugins::Storage;

use strict;
use warnings;

=head1 NAME

CapMan::Plugins::Storage - CapMan plugin to graph used storage.

=head1 DESCRIPTION

The used storage is a usefull thing to know for fileservers, mailserver-spool
partitions, caching proxy servers etc.

=cut

use RRDs;
use CapMan::Config qw(:rrdcreate get_rrdpath);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd create_rrdfile get_n_colors);

our $GraphDefStorage = ['-v', 'Bytes',
	'DEF:s={filename}:used:AVERAGE',
	'AREA:s#00FF00:Used  ',
	'GPRINT:s:LAST:%5.1lf%sBytes Last'];

our $GraphDefMemory = ['-b', '1024', '-l', '0', '-v', 'Bytes',
	'DEF:used={filename}:used:AVERAGE',
	'DEF:free={filename}:free:AVERAGE',
	'AREA:used#FF0000:Used',
	'GPRINT:used:AVERAGE:%5.1lf%sByte Average,',
	'GPRINT:used:MAX:%5.1lf%sByte Max,',
	'GPRINT:used:LAST:%5.1lf%sByte Last\n',
	'STACK:free#00FF00:Free',
	'GPRINT:free:AVERAGE:%5.1lf%sByte Average,',
	'GPRINT:free:MAX:%5.1lf%sByte Max,',
	'GPRINT:free:LAST:%5.1lf%sByte Last'];

=head1 SERVICES PROVIDED

=over 4

=item storage

Queries the used space on each partition.

=back

=cut

register_input_handler ('storage', '.1.3.6.1.2.1.25.2.3.1', \&storage_handler);
register_create_handler ('storage', \&storage_create_handler);
register_create_handler ('memory',  \&storage_create_mem_handler);
register_graph_handler ('storage', $GraphDefStorage, \&print_meta_graph_storage);
register_graph_handler ('memory',  $GraphDefMemory);

return (1);

sub storage_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $data = shift;

	my %ids = ();
	my $hash = {};

	my %tt =
	(
		'.1.3.6.1.2.1.25.2.3.1.3'	=> 'hrStorageDescr',
		'.1.3.6.1.2.1.25.2.3.1.5'	=> 'hrStorageSize',
		'.1.3.6.1.2.1.25.2.3.1.6'	=> 'hrStorageUsed',
		'.1.3.6.1.2.1.25.2.3.1.2'	=> 'hrStorageType',
		'.1.3.6.1.2.1.25.2.3.1.4'	=> 'hrStorageAllocationUnits'
	);

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;

		$name = $tt{$name} if (defined ($tt{$name}));
		$val =~ s/^(\d+) Bytes$/$1/;

		$hash->{$name}{$seq} = $val;
		$ids{$seq} = 1;
	}

	for (keys (%ids))
	{
		my $id = $_;
		my $descr = $hash->{'hrStorageDescr'}{$id};
		my $size  = $hash->{'hrStorageSize'}{$id};
		my $used  = $hash->{'hrStorageUsed'}{$id};
		my $type  = $hash->{'hrStorageType'}{$id};
		my $block = $hash->{'hrStorageAllocationUnits'}{$id};

		if (!defined ($descr))
		{
			print STDERR "$host: Storage $id has no description.\n";
			next;
		}

		if ($descr eq 'Memory Buffers')
		{
			next;
		}

		if (!defined ($size) or !defined ($used)
				or !defined ($type) or !defined ($block))
		{
			print STDERR "$host: No storage information for ``$descr''.\n";
			next;
		}

		$size *= $block;
		$used *= $block;

		if ($descr eq 'Real Memory'
				or $descr eq 'Physical Memory') # *surprise*: Windows does this different..
		{
			my $free = $size - $used;
			
			write_to_rrd ($customer, $host, 'memory', '', $used, $free);

			next;
		}

		# HOST-RESOURCES-TYPES::hrStorageOther
		next if ($type eq '.1.3.6.1.2.1.25.2.1.1');

		if ($descr =~ m#^/(.*)#)
		{
			if ($descr eq '/')
			{
				$descr = 'root';
			}
			else
			{
				$descr = $1;
				$descr =~ s#/#_#g;
			}
		}
		elsif ($descr =~ m#^([A-Z]):\\(?: Label:(\S*))?#)
		{
			$descr = $1;
			$descr .= " $2" if ($2);
			$descr =~ s#/#_#g;
		}
		elsif ($descr =~ m/swap/i
				or $descr eq 'Virtual Memory') # *surprise*: Windows does this different..
		{
			$descr = 'swap';
		}
		else
		{
			print STDERR "Warning: $host: Ignoring storage item ``$descr''.\n";
			next;
		}

		write_to_rrd ($customer, $host, 'storage', $descr, $used);
	}
}

sub storage_create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:used:GAUGE:$HeartBeat:0:U");
}

sub storage_create_mem_handler ($)
{
	my $file = shift;

	create_rrdfile ($file,
		"DS:used:GAUGE:$HeartBeat:0:U",
		"DS:free:GAUGE:$HeartBeat:0:U"
	);
}

sub print_meta_graph_storage ($$$@)
{
	confess ("Wrong number of arguments") if (@_ < 5);

	my $customer = shift;
	my $host = shift;
	my $begin = shift;
	my $end = shift;
	my @files = grep { $_ !~ m/swap/ } (@_);
	
	my @colors = get_n_colors (scalar (@files));
	
	my @cmd = ('-', '-s', $begin, '-e', $end, '-a', 'PNG', '-t', "Storage on $host", '-v', 'Bytes');
	my $rrdpath = get_rrdpath ();

	my $num = 0;

	@files = sort
	{
		$a eq 'storage-root.rrd' ? -1 : $b eq 'storage-root.rrd' ? 1 : $a cmp $b
	} (@files);

	for (@files)
	{
		my $file = "$rrdpath/$customer/$host/$_";

		push (@cmd, qq(DEF:val$num=$file:used:AVERAGE));
		$num++;
	}

	for (my $i = 0; $i < $num; $i++)
	{
		my $dtype = 'STACK';
		my $color;
		my $desc;

		$dtype = 'AREA' if ($i == 0);

		if ($files[$i] eq 'storage-root.rrd')
		{
			$color = 'FF0000';
			$desc  = 'Root Directory';
		}
		elsif ($files[$i] =~ m/^storage-([A-Z])(?:_(.+))?\.rrd$/)
		{
			my $ind = $i % scalar (@colors);
			$color = $colors[$ind];

			$desc = $1 . '\\:';
			$desc .= " ($2)" if ($2);
		}
		else
		{
			my $ind = $i % scalar (@colors);
			$color = $colors[$ind];

			$desc = $files[$i];
			$desc =~ s#storage-|_#/#g;
			$desc =~ s#\.rrd$##;
		}

		if (length ($desc) > 20)
		{
			substr ($desc, 18) = '..';
		}

		push (@cmd, $dtype . ':val' . $i . '#' . $color . ':' . sprintf ('%-20s', $desc));
		push (@cmd, "GPRINT:val$i:AVERAGE:\%5.1lf\%sByte Average,",
			"GPRINT:val$i:MAX:\%5.1lf\%sByte Max,",
			"GPRINT:val$i:LAST:\%5.1lf\%sByte Last\\n");
	}

	RRDs::graph (@cmd);

	die (RRDs::error ()) if (RRDs::error ());
}


=head1 SEE ALSO

L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
