package CapMan::Plugins::Solaris;

use strict;
use warnings;

=head1 NAME

CapMan::Plugins::Solaris - CapMan plugin for Sun Solaris specific stuff

=head1 DESCRIPTION

Uses the B<suncpu> field in the SysStat return values and (tries to) read the
I<SUN MIB>.

=cut

use RRDs;

use CapMan::Config qw(:rrdcreate get_rrdpath);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd create_rrdfile);

our $GraphDefSunCPU = ['-v', 'Percent', '-u', '100',
	'DEF:u={filename}:user:AVERAGE',
	'DEF:w={filename}:wait:AVERAGE',
	'DEF:s={filename}:syst:AVERAGE',
	'DEF:i={filename}:idle:AVERAGE',
	'HRULE:100#FF0000',
	'AREA:s#FF0000:System',
	'GPRINT:s:AVERAGE:%4.1lf%% Average,',
	'GPRINT:s:MAX:%4.1lf%% Max,',
	'GPRINT:s:LAST:%4.1lf%% Last\n',
	'STACK:u#0000FF:User  ',
	'GPRINT:u:AVERAGE:%4.1lf%% Average,',
	'GPRINT:u:MAX:%4.1lf%% Max,',
	'GPRINT:u:LAST:%4.1lf%% Last\n',
	'STACK:n#FFFF00:Nice  ',
	'GPRINT:w:AVERAGE:%4.1lf%% Average,',
	'GPRINT:w:MAX:%4.1lf%% Max,',
	'GPRINT:w:LAST:%4.1lf%% Last'];

=head1 SERVICES PROVIDED

=over 4

=item solaris (input)

The B<solaris> input handler requests the I<sunHostPerf>-group from the I<SUN
MIB>. This is served by the SUN SNMPd and appears to be useless unless some
additional software is bought and installed.

If this plugin receives any values it will provide B<cpu>, B<interrupts>,
B<io-page>, and  B<io-swap>. The create- and graph-handlers are defined
elsewhere.

=item suncpu (create, graph)

Handler for CPU-usage. You need to install a client-side script to use this. It
is collected (input handler) using L<SysStat|CapMan::Plugins::SysStat>.

=back

=cut

# .1.3.6.1.4.1.42.3.13 == enterprises.sun.sunMib.sunHostPerf
register_input_handler  ('solaris', '.1.3.6.1.4.1.42.3.13', \&input_handler);
register_create_handler ('suncpu', \&create_sun_cpu_handler);
register_graph_handler  ('suncpu', $GraphDefSunCPU, \&print_meta_graph_suncpu);

return (1);

sub create_sun_cpu_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, 
		"DS:user:COUNTER:$HeartBeat:0:110",
		"DS:syst:COUNTER:$HeartBeat:0:110",
		"DS:wait:COUNTER:$HeartBeat:0:110",
		"DS:idle:COUNTER:$HeartBeat:0:110"
	);
}

#octo@entwicklung2:~/capman$ snmpwalk -v 1 -Oqs -c 'public' '213.95.23.101' .1.3.6.1.4.1.42.3.13 
#enterprises.42.3.13.1.0 0
#enterprises.42.3.13.2.0 0
#enterprises.42.3.13.3.0 0
#enterprises.42.3.13.4.0 0
#enterprises.42.3.13.5.0 0
#enterprises.42.3.13.6.0 0
#enterprises.42.3.13.7.0 0
#enterprises.42.3.13.8.0 0
#enterprises.42.3.13.9.0 0
#enterprises.42.3.13.10.0 0
#enterprises.42.3.13.11.0 0
#enterprises.42.3.13.12.0 0
#enterprises.42.3.13.13.0 0
#enterprises.42.3.13.14.0 0
#enterprises.42.3.13.15.0 0
#enterprises.42.3.13.16.0 0
#enterprises.42.3.13.17.0 0
#enterprises.42.3.13.18.0 0

sub input_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host   = shift;
	my $server = shift;
	my $data   = shift;

	my %values =
	(
		rsUserProcessTime   => 0, #  .1
		rsNiceModeTime      => 0, #  .2
		rsSystemProcessTime => 0, #  .3
		rsIdleModeTime      => 0, #  .4
		rsDiskXfer1         => 0, #  .5
		rsDiskXfer2         => 0, #  .6
		rsDiskXfer3         => 0, #  .7
		rsDiskXfer4         => 0, #  .8
		rsVPagesIn          => 0, #  .9
		rsVPagesOut         => 0, # .10
		rsVSwapIn           => 0, # .11
		rsVSwapOut          => 0, # .12
		rsVIntr             => 0, # .13
		rsIfInPackets       => 0, # .14
		rsIfOutPackets      => 0, # .15
		rsIfInErrors        => 0, # .16
		rsIfOutErrors       => 0, # .17
		rsIfCollisions      => 0  # .18
	);

	for (@$data)
	{
		my $d = $_;

		if    ($d->[0] eq '.1.3.6.1.4.1.42.3.13.1') { $values{'rsUserProcessTime'}   = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.2') { $values{'rsNiceModeTime'}      = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.3') { $values{'rsSystemProcessTime'} = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.4') { $values{'rsIdleModeTime'}      = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.5') { $values{'rsDiskXfer1'}         = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.6') { $values{'rsDiskXfer2'}         = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.7') { $values{'rsDiskXfer3'}         = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.8') { $values{'rsDiskXfer4'}         = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.9') { $values{'rsVPagesIn'}          = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.10') { $values{'rsVPagesOut'}        = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.11') { $values{'rsVSwapIn'}          = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.12') { $values{'rsVSwapOut'}         = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.13') { $values{'rsVIntr'}            = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.14') { $values{'rsIfInPackets'}      = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.15') { $values{'rsIfOutPackets'}     = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.16') { $values{'rsIfInErrors'}       = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.17') { $values{'rsIfOutErrors'}      = $d->[2]; }
		elsif ($d->[0] eq '.1.3.6.1.4.1.42.3.13.18') { $values{'rsIfCollisions'}     = $d->[2]; }
		elsif ($::DEBUG || 1) { print STDERR "Unknown type: `" . $d->[0] . "'\n"; }
	}

	if ($::DEBUG)
	{
		require Data::Dumper;
		print Data::Dumper->Dump ([\%values], ['values']);
	}

	if (($values{'rsUserProcessTime'} != 0)
		|| ($values{'rsNiceModeTime'} != 0)
		|| ($values{'rsSystemProcessTime'} != 0)
		|| ($values{'rsIdleModeTime'} != 0))
	{
		write_to_rrd ($customer, $host, 'cpu', 0,
			$values{'rsUserProcessTime'}, $values{'rsNiceModeTime'},
			$values{'rsSystemProcessTime'}, $values{'rsIdleModeTime'})
	}

	if ($values{'rsVIntr'} != 0)
	{
		write_to_rrd ($customer, $host, 'interrupts', '', $values{'rsVIntr'});
	}

	if (($values{'rsVPagesIn'} != 0) || ($values{'rsVPagesOut'} != 0))
	{
		write_to_rrd ($customer, $host, 'io', 'page', $values{'rsVPagesIn'}, $values{'rsVPagesOut'});
	}

	if (($values{'rsVSwapIn'} != 0) || ($values{'rsVSwapOut'} != 0))
	{
		write_to_rrd ($customer, $host, 'io', 'swap', $values{'rsVSwapIn'}, $values{'rsVSwapOut'});
	}

	return;
}

sub print_meta_graph_sun_cpu
{
	confess ("Wrong number of arguments") if (@_ < 5);

	my $customer = shift;
	my $host = shift;
	my $begin = shift;
	my $end = shift;
	my @files = grep { $_ ne 'suncpu.rrd' } (@_);

	my @cmd = ('-', '-s', $begin, '-e', $end, '-a', 'PNG');

	my $num = 0;
	my $rrdpath = get_rrdpath ();

	my $cdef_user = 'CDEF:user=user0';
	my $cdef_nice = 'CDEF:wait=wait0';
	my $cdef_syst = 'CDEF:syst=syst0';
	my $cdef_idle = 'CDEF:idle=idle0';

	for (@files)
	{
		my $file = "$rrdpath/$customer/$host/$_";

		push (@cmd, qq(DEF:user$num=$file:user:AVERAGE));
		push (@cmd, qq(DEF:wait$num=$file:nice:AVERAGE));
		push (@cmd, qq(DEF:syst$num=$file:syst:AVERAGE));
		push (@cmd, qq(DEF:idle$num=$file:idle:AVERAGE));

		$num++;
	}

	for (my $i = 1; $i < $num; $i++)
	{
		$cdef_user .= ",user$i,+";
		$cdef_nice .= ",wait$i,+";
		$cdef_syst .= ",syst$i,+";
		$cdef_idle .= ",idle$i,+";
	}

	if ($num > 1)
	{
		$cdef_user .= ",$num,/";
		$cdef_nice .= ",$num,/";
		$cdef_syst .= ",$num,/";
		$cdef_idle .= ",$num,/";
	}

	push (@cmd,
		'-v', 'Percent', '-u', '100',
		'-t', $num > 1 ? "Combined CPU Usage on $host ($num CPUs)" : "CPU Usage on $host",
		$cdef_user, $cdef_nice, $cdef_syst, $cdef_idle,
		'HRULE:100#FF0000',
		'AREA:syst#FF0000:System ',
		'GPRINT:syst:AVERAGE:%4.1lf%% Average,',
		'GPRINT:syst:MAX:%4.1lf%% Max,',
		'GPRINT:syst:LAST:%4.1lf%% Last\n',
		'STACK:user#0000FF:User   ',
		'GPRINT:user:AVERAGE:%4.1lf%% Average,',
		'GPRINT:user:MAX:%4.1lf%% Max,',
		'GPRINT:user:LAST:%4.1lf%% Last\n',
		'STACK:wait#00FF00:IO-Wait',
		'GPRINT:wait:AVERAGE:%4.1lf%% Average,',
		'GPRINT:wait:MAX:%4.1lf%% Max,',
		'GPRINT:wait:LAST:%4.1lf%% Last');

	RRDs::graph (@cmd);

	die (RRDs::error ()) if (RRDs::error ());
}


=head1 SEE ALSO

L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
