package CapMan::Plugins::Processes;

use strict;
use warnings;

=head1 NAME

CapMan::Plugins::Processes - CapMan plugin to query nuber of processes.

=head1 DESCRIPTION

This plugin queries the number of processes, matched by the config option
I<processmatch>. For example, to see how many Apache-processes are running on
your webserver you might add the following lines to your config:

    [host webserver]
    address: 192.168.0.2
    services: processes
    processmatch: httpd

Of course you can give more then one match to graph more then one process. For
example, for a web-proxy that does also virus scans you might do something
along these lines:

    [host proxy]
    address: 192.168.0.3
    services: processes
    processmatch: squid, clamav

=cut

use RRDs;

use CapMan::Config qw(:rrdcreate get_rrdpath get_host_config);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd create_rrdfile get_n_colors);

our $GraphDef = ['-v', 'Processes',
	'DEF:c={filename}:count:AVERAGE',
	'LINE2:c#000000:Processes',
	'GPRINT:c:AVERAGE:%5.1lf Average,',
	'GPRINT:c:MAX:%5.1lf Max,',
	'GPRINT:c:LAST:%5.1lf Last'];

=head1 SERVICES PROVIDED

=over 4

=item processes

Number of processes matching one of the I<processmatch> option.

=back

=cut

register_input_handler ('processes', '.1.3.6.1.2.1.25.4.2.1.2', \&processes_handler);
register_create_handler ('processes', \&create_processes_handler);
register_graph_handler ('processes', $GraphDef, \&print_meta_graph_processes);

return (1);

sub processes_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $data = shift;

	my @matches = get_host_config ($customer, $host, 'processmatch');
	my %data;
	my %re;

	if (!@matches)
	{
		print STDERR "No matches defined for service ``processes''.\n";
		return;
	}

	%data = map { $_ => 0 } (@matches);
	%re = map { my $n = $_; my $re = quotemeta ($_); $n => qr($re); } (@matches);

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;

		for (@matches)
		{
			$data{$_}++ if ($val =~ $re{$_});
		}
	}
	
	for (@matches)
	{
		write_to_rrd ($customer, $host, 'processes', $_, $data{$_});
	}
}

sub create_processes_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:count:GAUGE:$HeartBeat:0:U");
}

sub print_meta_graph_processes ($$$@)
{
	confess ("Wrong number of arguments") if (@_ < 5);

	my $customer = shift;
	my $host = shift;
	my $begin = shift;
	my $end = shift;
	my @files = @_;
	
	my @colors = get_n_colors (scalar (@files));
	
	my @cmd = ('-', '-s', $begin, '-e', $end, '-a', 'PNG', '-t', "Processes on $host", '-v', 'Processes');
	my $rrdpath = get_rrdpath ();

	my $num = 0;
	my $maxlen = 0;
	my $format;

	for (@files)
	{
		my $file = "$rrdpath/$customer/$host/$_";

		if (m/-([^\.]+)\./ and length ($1) > $maxlen)
		{
			$maxlen = length ($1);
		}

		push (@cmd, qq(DEF:val$num=$file:count:AVERAGE));
		$num++;
	}

	return unless ($maxlen);
	$format = 'LINE2:val%i#%s:%-' . $maxlen . 's';
	
	for (my $i = 0; $i < $num; $i++)
	{
		my ($name) = $files[$i] =~ m/-([^\.]+)\./;
		next unless ($name);

		my $color = $colors[$i % scalar (@colors)];
		
		push (@cmd, sprintf ($format, $i, $color, $name));
		push (@cmd, "GPRINT:val$i:AVERAGE:\%5.1lf Average,",
			"GPRINT:val$i:MIN:\%5.1lf Min,",
			"GPRINT:val$i:MAX:\%5.1lf Max,",
			"GPRINT:val$i:LAST:\%5.1lf Last\\n");
	}

	print STDERR join (' ', @cmd), "\n";

	RRDs::graph (@cmd);

	die (RRDs::error ()) if (RRDs::error ());
}

=head1 SEE ALSO

L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
