package CapMan::Plugins::PowerPlus;

use strict;
use warnings;

=head1 NAME

CapMan::Plugins::PowerPlus - CapMan plugin for the Power+ UPS

=head1 DESCRIPTION

This plugin queries various information from the Power+ UPS from Gamatronix
(L<http://www.gamatronic.com/>), distributed by AdPoS
(L<http://www.adpos-ups.de/>) as "Maxi-J G".

=cut

use RRDs;

use CapMan::Config qw(:rrdcreate get_rrdpath);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd create_rrdfile get_n_colors);

#our $GraphDefCharge = ['-v', 'Percent',
#	'DEF:p={filename}:percentage:AVERAGE',
#	'LINE1:p#000000:Charged',
#	'GPRINT:p:AVERAGE:%5.1lf%% Average,',
#	'GPRINT:p:MIN:%5.1lf%% Min,',
#	'GPRINT:p:MAX:%5.1lf%% Max,',
#	'GPRINT:p:LAST:%5.1lf%% Last'];
#
#our $GraphDefDuration = ['-v', 'Minutes',
#	'DEF:d={filename}:duration:AVERAGE',
#	'LINE1:d#000000:Duration [min]',
#	'GPRINT:d:AVERAGE:%4.1lf Average,',
#	'GPRINT:d:MIN:%4.1lf Min,',
#	'GPRINT:d:MAX:%4.1lf Max,',
#	'GPRINT:d:LAST:%4.1lf Last'];
#
#our $GraphDefStatus = ['-v', 'State',
#	'DEF:d={filename}:status:AVERAGE',
#	'LINE1:d#000000:Status  (2 = good, 1 = bad, 0 = nightmare)'];
#
#our $GraphDefUtilisation = ['-v', 'Percent',
#	'DEF:p={filename}:percentage:AVERAGE',
#	'LINE1:p#000000:Utilisation',
#	'GPRINT:p:AVERAGE:%5.1lf%% Average,',
#	'GPRINT:p:MIN:%5.1lf%% Min,',
#	'GPRINT:p:MAX:%5.1lf%% Max,',
#	'GPRINT:p:LAST:%5.1lf%% Last'];

=head1 SERVICES PROVIDED

=over 4

=item powerplus

Provides information about input/output voltage and current as well as battery
duration, system utilisation and system status of the Power+ UPS.

=back

=cut

register_input_handler ('powerplus-input',   '.1.3.6.1.4.1.6050.5.4', \&pp_input_handler);
register_input_handler ('powerplus-output',  '.1.3.6.1.4.1.6050.5.5', \&pp_output_handler);
register_input_handler ('powerplus-dc',      '.1.3.6.1.4.1.6050.5.6', \&pp_dc_handler);
register_input_handler ('powerplus-modules', '.1.3.6.1.4.1.6050.5.7', \&pp_modules_handler);
register_input_handler ('powerplus-switch',  '.1.3.6.1.4.1.6050.5.8', \&pp_switch_handler);

return (1);

sub submit_array
{
	my $customer = shift;
	my $host = shift;
	my $type = shift;
	my $instance = shift;
	my $array = shift;
	my $factor = @_ ? shift : 1.0;

	if (!defined ($array) || (ref ($array) ne 'ARRAY'))
	{
		return;
	}

	for (my $i = 0; $i < @$array; $i++)
	{
		if (defined ($array->[$i]))
		{
			my $tmp_inst = $instance;
			$tmp_inst =~ s/{suboid}/$i/gi;

			write_to_rrd ($customer, $host, $type, $tmp_inst,
				$array->[$i] * $factor);
		}
	}
} # submit_array

sub pp_input_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $orig_data = shift;
	my $data = {};

	for (@$orig_data)
	{
		my ($name, $seq, $val, $type) = @$_;

		if (!exists ($data->{$name}))
		{
			$data->{$name} = [];
		}
		$data->{$name}[$seq] = $val;
	}

	# Global Input
	submit_array ($customer, $host, 'voltage', 'global_input{suboid}',
		$data->{'.1.3.6.1.4.1.6050.5.4.1.1.2'});
	submit_array ($customer, $host, 'current', 'global_input{suboid}',
		$data->{'.1.3.6.1.4.1.6050.5.4.1.1.3'});
	submit_array ($customer, $host, 'power', 'global_apparent_input{suboid}',
		$data->{'.1.3.6.1.4.1.6050.5.4.1.1.4'}, 100.0);
	submit_array ($customer, $host, 'power', 'global_active_input{suboid}',
		$data->{'.1.3.6.1.4.1.6050.5.4.1.1.5'}, 100.0);
	submit_array ($customer, $host, 'percent', 'performance_factor_input{suboid}',
		$data->{'.1.3.6.1.4.1.6050.5.4.1.1.6'});
}

sub pp_output_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $orig_data = shift;
	my $data = {};

	for (@$orig_data)
	{
		my ($name, $seq, $val, $type) = @$_;

		if (!exists ($data->{$name}))
		{
			$data->{$name} = [];
		}
		$data->{$name}[$seq] = $val;
	}

	submit_array ($customer, $host, 'voltage', 'global_output{suboid}',
		$data->{'.1.3.6.1.4.1.6050.5.5.1.1.2'});
	submit_array ($customer, $host, 'current', 'global_output{suboid}',
		$data->{'.1.3.6.1.4.1.6050.5.5.1.1.3'});
	submit_array ($customer, $host, 'power', 'global_apparent_output{suboid}',
		$data->{'.1.3.6.1.4.1.6050.5.5.1.1.4'}, 100.0);
	submit_array ($customer, $host, 'power', 'global_active_output{suboid}',
		$data->{'.1.3.6.1.4.1.6050.5.5.1.1.5'}, 100.0);
	submit_array ($customer, $host, 'percent', 'load_level_output{suboid}',
		$data->{'.1.3.6.1.4.1.6050.5.5.1.1.6'});
	submit_array ($customer, $host, 'percent', 'active_load_level_output{suboid}',
		$data->{'.1.3.6.1.4.1.6050.5.5.1.1.7'});
	submit_array ($customer, $host, 'percent', 'performance_factor_output{suboid}',
		$data->{'.1.3.6.1.4.1.6050.5.5.1.1.8'});
} # pp_output_handler

sub pp_dc_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $orig_data = shift;
	my $data = {};

	for (@$orig_data)
	{
		my ($name, $seq, $val, $type) = @$_;

		if (!exists ($data->{$name}))
		{
			$data->{$name} = [];
		}
		$data->{$name}[$seq] = $val;
	}

	submit_array ($customer, $host, 'voltage', 'global_dc_positive',
		$data->{'.1.3.6.1.4.1.6050.5.6.1'});
	submit_array ($customer, $host, 'voltage', 'global_dc_negative',
		$data->{'.1.3.6.1.4.1.6050.5.6.2'});
	submit_array ($customer, $host, 'voltage', 'global_dc_total',
		$data->{'.1.3.6.1.4.1.6050.5.6.3'});
} # pp_dc_handler

sub pp_modules_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $orig_data = shift;
	my $data = {};

	for (@$orig_data)
	{
		my ($name, $seq, $val, $type) = @$_;

		if (!exists ($data->{$name}))
		{
			$data->{$name} = [];
		}
		$data->{$name}[$seq] = $val;
	}

	submit_array ($customer, $host, 'voltage', 'module{suboid}_input1',
		$data->{'.1.3.6.1.4.1.6050.5.7.2.1.9'});
	submit_array ($customer, $host, 'voltage', 'module{suboid}_input2',
		$data->{'.1.3.6.1.4.1.6050.5.7.2.1.10'});
	submit_array ($customer, $host, 'voltage', 'module{suboid}_input3',
		$data->{'.1.3.6.1.4.1.6050.5.7.2.1.11'});

	submit_array ($customer, $host, 'voltage', 'module{suboid}_output1',
		$data->{'.1.3.6.1.4.1.6050.5.7.2.1.12'});
	submit_array ($customer, $host, 'voltage', 'module{suboid}_output2',
		$data->{'.1.3.6.1.4.1.6050.5.7.2.1.13'});
	submit_array ($customer, $host, 'voltage', 'module{suboid}_output3',
		$data->{'.1.3.6.1.4.1.6050.5.7.2.1.14'});


	submit_array ($customer, $host, 'current', 'module{suboid}_input1',
		$data->{'.1.3.6.1.4.1.6050.5.7.2.1.15'});
	submit_array ($customer, $host, 'current', 'module{suboid}_input2',
		$data->{'.1.3.6.1.4.1.6050.5.7.2.1.16'});
	submit_array ($customer, $host, 'current', 'module{suboid}_input3',
		$data->{'.1.3.6.1.4.1.6050.5.7.2.1.17'});

	submit_array ($customer, $host, 'current', 'module{suboid}_output1',
		$data->{'.1.3.6.1.4.1.6050.5.7.2.1.18'});
	submit_array ($customer, $host, 'current', 'module{suboid}_output2',
		$data->{'.1.3.6.1.4.1.6050.5.7.2.1.19'});
	submit_array ($customer, $host, 'current', 'module{suboid}_output3',
		$data->{'.1.3.6.1.4.1.6050.5.7.2.1.20'});

	submit_array ($customer, $host, 'voltage', 'module{suboid}_dc_positive',
		$data->{'.1.3.6.1.4.1.6050.5.7.2.1.21'});
	submit_array ($customer, $host, 'voltage', 'module{suboid}_dc_negative',
		$data->{'.1.3.6.1.4.1.6050.5.7.2.1.22'});
} # pp_modules_handler

sub pp_switch_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $orig_data = shift;
	my $data = {};

	for (@$orig_data)
	{
		my ($name, $seq, $val, $type) = @$_;

		if (!exists ($data->{$name}))
		{
			$data->{$name} = [];
		}
		$data->{$name}[$seq] = $val;
	}

	submit_array ($customer, $host, 'voltage', 'inverter_line1',
		$data->{'.1.3.6.1.4.1.6050.5.8.5'});
	submit_array ($customer, $host, 'voltage', 'inverter_line2',
		$data->{'.1.3.6.1.4.1.6050.5.8.6'});
	submit_array ($customer, $host, 'voltage', 'inverter_line3',
		$data->{'.1.3.6.1.4.1.6050.5.8.7'});

	submit_array ($customer, $host, 'voltage', 'bypass_line1',
		$data->{'.1.3.6.1.4.1.6050.5.8.8'});
	submit_array ($customer, $host, 'voltage', 'bypass_line2',
		$data->{'.1.3.6.1.4.1.6050.5.8.9'});
	submit_array ($customer, $host, 'voltage', 'bypass_line3',
		$data->{'.1.3.6.1.4.1.6050.5.8.10'});

	submit_array ($customer, $host, 'frequency', 'inverter',
		$data->{'.1.3.6.1.4.1.6050.5.8.11'}, 1.0/256.0);
	submit_array ($customer, $host, 'frequency', 'bypass',
		$data->{'.1.3.6.1.4.1.6050.5.8.12'}, 1.0/256.0);

}

=head1 SEE ALSO

L<CapMan::Services>, L<http://www.aps-ups.de/>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
