package CapMan::Plugins::Power;

use strict;
use warnings;

=head1 NAME

CapMan::Plugins::Power - Power related graph handlers

=head1

Provides graph handlers for power-graphs, namely voltage and current.

=cut

use Math::Trig;

use CapMan::Services qw(:plugin);
use CapMan::Tools qw(create_rrdfile);
use CapMan::Config qw(:rrdcreate get_rrdpath);

our $GraphDefCurrent = ['-v', 'Amperes',
	'DEF:c={filename}:current:AVERAGE',
	'LINE1:c#000000:Current',
	'GPRINT:c:AVERAGE:%5.1lf%sA Average,',
	'GPRINT:c:MIN:%5.1lf%sA Min,',
	'GPRINT:c:MAX:%5.1lf%sA Max,',
	'GPRINT:c:LAST:%5.1lf%sA Last'];

our $GraphDefVoltage = ['-v', 'Volts',
	'DEF:v={filename}:voltage:AVERAGE',
	'LINE1:v#000000:Voltage',
	'GPRINT:v:AVERAGE:%5.1lf%sV Average,',
	'GPRINT:v:MIN:%5.1lf%sV Min,',
	'GPRINT:v:MAX:%5.1lf%sV Max,',
	'GPRINT:v:LAST:%5.1lf%sV Last'];

our $GraphDefPower = ['-v', 'Watts',
	'DEF:v={filename}:power:AVERAGE',
	'LINE1:v#000000:Power',
	'GPRINT:v:AVERAGE:%5.1lf%sW Average,',
	'GPRINT:v:MIN:%5.1lf%sW Min,',
	'GPRINT:v:MAX:%5.1lf%sW Max,',
	'GPRINT:v:LAST:%5.1lf%sW Last'];

our $GraphDefFrequency = ['-v', 'Hertz',
	'DEF:f={filename}:frequency:AVERAGE',
	'LINE1:f#000000:Frequency',
	'GPRINT:f:AVERAGE:%5.1lf%sHz Average,',
	'GPRINT:f:MIN:%5.1lf%sHz Min,',
	'GPRINT:f:MAX:%5.1lf%sHz Max,',
	'GPRINT:f:LAST:%5.1lf%sHz Last'];

=head1 SERVICES PROVIDED

The following B<graph handlers> are provided:

=over 4

=item voltage I<(meta)>

Graphing voltage.

=item current I<(meta)>

Graphing current.

=item power I<(meta)>

Graphing power.

=item frequency I<(meta)>

Graphing frequencies.

=back

=cut

register_create_handler ('voltage',   \&voltage_create_handler);
register_create_handler ('current',   \&current_create_handler);
register_create_handler ('power',     \&power_create_handler);
register_create_handler ('frequency', \&frequency_create_handler);
register_graph_handler ('voltage',   $GraphDefVoltage,   \&print_meta_graph_voltage);
register_graph_handler ('current',   $GraphDefCurrent,   \&print_meta_graph_current);
register_graph_handler ('power',     $GraphDefPower,     \&print_meta_graph_power);
register_graph_handler ('frequency', $GraphDefFrequency, \&print_meta_graph_frequency);

return (1);

sub frequency_create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:frequency:GAUGE:$HeartBeat:0:511");
}

sub voltage_create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:voltage:GAUGE:$HeartBeat:-10000:10000");
}

sub current_create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:current:GAUGE:$HeartBeat:0:255");
}

sub power_create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:power:GAUGE:$HeartBeat:0:U");
}

sub print_meta_graph_voltage
{
	confess ("Wrong number of arguments") if (@_ < 5);

	my $customer = shift;
	my $host = shift;
	my $begin = shift;
	my $end = shift;
	my @files = @_;
	my @names = ();
	my $name_maxlen;
	
	my @cmd = ('-', '-s', $begin, '-e', $end, '-a', 'PNG', '-t', "Voltage on $host", '-v', 'Volts');
	my $rrdpath = get_rrdpath ();

	my $num = 0;

	$name_maxlen = 0;
	for (@files)
	{
		my $file = "$rrdpath/$customer/$host/$_";
		my $name = (split (m/-/, $_))[1];

		$name =~ s/\.rrd$//;
		$name =~ s/_+/ /g;

		push (@cmd, qq(DEF:val$num=$file:voltage:AVERAGE));
		push (@names, $name);
		if (length ($name) > $name_maxlen)
		{
			$name_maxlen = length ($name);
		}
		$num++;
	}

	for (my $i = 0; $i < $num; $i++)
	{
		my $line = sprintf ('%-*s', $name_maxlen, $names[$i]);
		my $color = get_color ($i, $num);
		
		push (@cmd, 'LINE1:val' . $i . '#' . $color . ":$line");
		push (@cmd, "GPRINT:val$i:AVERAGE:\%5.1lf\%sV Average,",
			"GPRINT:val$i:MIN:\%5.1lf\%sV Min,",
			"GPRINT:val$i:MAX:\%5.1lf\%sV Max,",
			"GPRINT:val$i:LAST:\%5.1lf\%sV Last\\n");
	}

	RRDs::graph (@cmd);

	die (RRDs::error ()) if (RRDs::error ());
}

sub print_meta_graph_current
{
	confess ("Wrong number of arguments") if (@_ < 5);

	my $customer = shift;
	my $host = shift;
	my $begin = shift;
	my $end = shift;
	my @files = @_;
	my @names = ();
	my $name_maxlen;
	
	my @cmd = ('-', '-s', $begin, '-e', $end, '-a', 'PNG', '-t', "Current on $host", '-v', 'Amperes');
	my $rrdpath = get_rrdpath ();

	my $num = 0;

	$name_maxlen = 0;
	for (@files)
	{
		my $file = "$rrdpath/$customer/$host/$_";
		my $name = (split (m/-/, $_))[1];

		$name =~ s/\.rrd$//;
		$name =~ s/_+/ /g;

		push (@cmd, qq(DEF:val$num=$file:current:AVERAGE));
		push (@names, $name);
		if (length ($name) > $name_maxlen)
		{
			$name_maxlen = length ($name);
		}
		$num++
	}

	for (my $i = 0; $i < $num; $i++)
	{
		my $line = sprintf ('%-*s', $name_maxlen, $names[$i]);

		my $color = get_color ($i, $num);
		
		push (@cmd, 'LINE1:val' . $i . '#' . $color . ":$line");
		push (@cmd, "GPRINT:val$i:AVERAGE:\%5.1lf\%sA Average,",
			"GPRINT:val$i:MIN:\%5.1lf\%sA Min,",
			"GPRINT:val$i:MAX:\%5.1lf\%sA Max,",
			"GPRINT:val$i:LAST:\%5.1lf\%sA Last\\n");
	}

	RRDs::graph (@cmd);

	die (RRDs::error ()) if (RRDs::error ());
}

sub print_meta_graph_power ($$$@)
{
	confess ("Wrong number of arguments") if (@_ < 5);

	my $customer = shift;
	my $host = shift;
	my $begin = shift;
	my $end = shift;
	my @files = @_;
	my @names = ();
	my $name_maxlen;
	
	my @cmd = ('-', '-s', $begin, '-e', $end, '-a', 'PNG', '-t', "Power on $host", '-v', 'Watts');
	my $rrdpath = get_rrdpath ();

	my $num = 0;

	$name_maxlen = 0;
	for (@files)
	{
		my $file = "$rrdpath/$customer/$host/$_";
		my $name = (split (m/-/, $_))[1];

		$name =~ s/\.rrd$//;
		$name =~ s/_+/ /g;

		push (@cmd, qq(DEF:val$num=$file:power:AVERAGE));

		if ($name !~ m/\w/)
		{
			$name = "Line $name";
		}

		push (@names, $name);
		if (length ($name) > $name_maxlen)
		{
			$name_maxlen = length ($name);
		}
		$num++;
	}

	for (my $i = 0; $i < $num; $i++)
	{
		my $line = sprintf ('%-*s', $name_maxlen, $names[$i]);
		my $color = get_color ($i, $num);
		
		push (@cmd, 'LINE1:val' . $i . '#' . $color . ":$line");
		push (@cmd, "GPRINT:val$i:AVERAGE:\%5.1lf\%sW Average,",
			"GPRINT:val$i:MIN:\%5.1lf\%sW Min,",
			"GPRINT:val$i:MAX:\%5.1lf\%sW Max,",
			"GPRINT:val$i:LAST:\%5.1lf\%sW Last\\n");
	}

	RRDs::graph (@cmd);

	die (RRDs::error ()) if (RRDs::error ());
}

sub print_meta_graph_frequency ($$$@)
{
	confess ("Wrong number of arguments") if (@_ < 5);

	my $customer = shift;
	my $host = shift;
	my $begin = shift;
	my $end = shift;
	my @files = @_;
	my @names = ();
	
	my @cmd = ('-', '-s', $begin, '-e', $end, '-a', 'PNG', '-t', "Frequency on $host", '-v', 'Hertz');
	my $rrdpath = get_rrdpath ();

	my $num = 0;

	for (@files)
	{
		my $file = "$rrdpath/$customer/$host/$_";
		my $name = (split (m/-/, $_))[1];

		$name =~ s/\.rrd$//;
		$name =~ s/_+/ /g;

		push (@cmd, qq(DEF:val$num=$file:frequency:AVERAGE));
		push (@names, $name);
		$num++
	}

	for (my $i = 0; $i < $num; $i++)
	{
		my $line = $names[$i];
		if ($line !~ m/\w/)
		{
			$line = "Line $line";
		}

		my $color = get_color ($i, $num);
		
		push (@cmd, 'LINE1:val' . $i . '#' . $color . ":$line");
		push (@cmd, "GPRINT:val$i:AVERAGE:\%5.1lf\%sHz Average,",
			"GPRINT:val$i:MIN:\%5.1lf\%sHz Min,",
			"GPRINT:val$i:MAX:\%5.1lf\%sHz Max,",
			"GPRINT:val$i:LAST:\%5.1lf\%sHz Last\\n");
	}

	RRDs::graph (@cmd);

	die (RRDs::error ()) if (RRDs::error ());
}

sub get_color ($$)
{
	my $num = shift;
	my $max = shift;

	die unless ($num < $max);

	my $total_pos = 6 * $num / $max;
	my $rel_pos = pi * ($total_pos - int ($total_pos));
	$total_pos = int ($total_pos);

	my $up   = 1.5 * pi;
	my $down = 0.5 * pi;

	my $r_value = 0;
	my $g_value = 0;
	my $b_value = 0;
	
	if ($total_pos == 0)
	{
		$r_value = 255;
		$g_value = int (128 + (128 * sin ($rel_pos + $up)));
	}
	elsif ($total_pos == 1)
	{
		$r_value = int (128 + (128 * sin ($rel_pos + $down)));
		$g_value = 255;
	}
	elsif ($total_pos == 2)
	{
		$g_value = 255;
		$b_value = int (128 + (128 * sin ($rel_pos + $up)));
	}
	elsif ($total_pos == 3)
	{
		$g_value = int (128 + (128 * sin ($rel_pos + $down)));
		$b_value = 255;
	}
	elsif ($total_pos == 4)
	{
		$b_value = 255;
		$r_value = int (128 + (128 * sin ($rel_pos + $up)));
	}
	elsif ($total_pos == 5)
	{
		$b_value = int (128 + (128 * sin ($rel_pos + $down)));
		$r_value = 255;
	}

	$r_value = 255 if ($r_value > 255); $r_value = 0 if ($r_value < 0);
	$g_value = 255 if ($g_value > 255); $g_value = 0 if ($g_value < 0);
	$b_value = 255 if ($b_value > 255); $b_value = 0 if ($b_value < 0);

	my $color = sprintf ("%02X%02X%02X", $r_value, $g_value, $b_value);

	#print STDERR "Color: $num/$max: $color\n";
	return ($color);
}

=head1 SEE ALSO

L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
