package CapMan::Plugins::NetScreen;

use strict;
use warnings;

=head1 NAME

CapMan::Plugins::NetScreen - CapMan plugin to collect statistics of NetScreen firewalls.

=head1 DESCRIPTION

This plugin collects CPU and memory utilisation, as well as sessioncount of
NetScreen firewalls.

=cut

use CapMan::Config qw(:rrdcreate);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd create_rrdfile);

our $CPUGraphDef = ['-v', 'Percent',
	'DEF:v={filename}:value:AVERAGE',
	'AREA:v#F7B7B7',
	'LINE1:v#FF0000:CPU',
	'GPRINT:v:MIN:%3.0lf%% Min,',
	'GPRINT:v:AVERAGE:%3.0lf%% Average,',
	'GPRINT:v:MAX:%3.0lf%% Max,',
	'GPRINT:v:LAST:%3.0lf%% Last'];

our $MemoryGraphDef = ['-v', 'Bytes', '-l', '0',
	'DEF:used={filename}:used:AVERAGE',
	'DEF:free={filename}:free:AVERAGE',
	'DEF:frag={filename}:frag:AVERAGE',
	'CDEF:total=used,free,+',
	'AREA:total#B7EFB7',
	'AREA:used#F7B7B7',
	'LINE1:total#00E000:Free',
	'GPRINT:free:MIN:%6.1lf%s Min,',
	'GPRINT:free:AVERAGE:%6.1lf%s Average,',
	'GPRINT:free:MAX:%6.1lf%s Max,',
	'GPRINT:free:LAST:%6.1lf%s Last\l',
	'LINE1:used#FF0000:Used',
	'GPRINT:used:MIN:%6.1lf%s Min,',
	'GPRINT:used:AVERAGE:%6.1lf%s Average,',
	'GPRINT:used:MAX:%6.1lf%s Max,',
	'GPRINT:used:LAST:%6.1lf%s Last\l',
	'LINE1:frag#0000FF:Frag',
	'GPRINT:frag:MIN:%6.1lf%s Min,',
	'GPRINT:frag:AVERAGE:%6.1lf%s Average,',
	'GPRINT:frag:MAX:%6.1lf%s Max,',
	'GPRINT:frag:LAST:%6.1lf%s Last\l'];

our $SessionsGraphDef = ['-v', 'Sessions', '-l', '0',
	'DEF:num={filename}:num:AVERAGE',
	'DEF:max={filename}:max:AVERAGE',
	'AREA:max#B7EFB7',
	'AREA:num#F7B7B7',
	'LINE1:max#00E000',
	'LINE1:num#FF0000:Sessions',
	'GPRINT:num:MIN:%5.0lf Min,',
	'GPRINT:num:AVERAGE:%5.0lf Average,',
	'GPRINT:num:MAX:%5.0lf Max,',
	'GPRINT:num:LAST:%5.0lf Last',
	'GPRINT:max:LAST:(Limit %5.0lf)\l'];

=head1 SERVICES PROVIDED

=over 4

=item netscreen

Handler for all things NetScreen.

=back

=cut

register_input_handler ('netscreen', '.1.3.6.1.4.1.3224.16', \&input_handler);

register_create_handler ('netscreen_cpu', \&cpu_create_handler);
register_create_handler ('netscreen_memory', \&memory_create_handler);
register_create_handler ('netscreen_sessions', \&sessions_create_handler);

register_graph_handler ('netscreen_cpu', $CPUGraphDef);
register_graph_handler ('netscreen_memory', $MemoryGraphDef);
register_graph_handler ('netscreen_sessions', $SessionsGraphDef);

return (1);

sub input_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $data = shift;
	my $hash = {};

	my $nsResCpuAvg       = '.1.3.6.1.4.1.3224.16.1.1';
	my $nsResMemAllocate  = '.1.3.6.1.4.1.3224.16.2.1';
	my $nsResMemLeft      = '.1.3.6.1.4.1.3224.16.2.2';
	my $nsResMemFrag      = '.1.3.6.1.4.1.3224.16.2.3';
	my $nsResSessAllocate = '.1.3.6.1.4.1.3224.16.3.2';
	my $nsResSessMaxium   = '.1.3.6.1.4.1.3224.16.3.3';
	my $nsResSessFailed   = '.1.3.6.1.4.1.3224.16.3.4';

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;
		$name =~ s/^enterprises\./.1.3.6.1.4.1./;
		if ((!defined ($seq) || ("$seq" eq '')) && ($name =~ m/(.*)\.0$/))
		{
			$name = $1;
			$seq = 0;
		}
		$hash->{$name}[$seq] = $val;
	}

	if (defined ($hash->{$nsResCpuAvg}))
	{
		write_to_rrd ($customer, $host, 'netscreen_cpu', '',
			$hash->{$nsResCpuAvg}[0]);
	}

	if (defined ($hash->{$nsResMemAllocate})
		&& defined ($hash->{$nsResMemLeft})
		&& defined ($hash->{$nsResMemFrag}))
	{
		write_to_rrd ($customer, $host, 'netscreen_memory', '',
			$hash->{$nsResMemAllocate}[0],
			$hash->{$nsResMemLeft}[0],
			$hash->{$nsResMemFrag}[0]);
	}

	if (defined ($hash->{$nsResSessAllocate})
		&& defined ($hash->{$nsResSessMaxium}))
	{
		write_to_rrd ($customer, $host, 'netscreen_sessions', '',
			$hash->{$nsResSessAllocate}[0],
			$hash->{$nsResSessMaxium}[0]);
	}
}

sub cpu_create_handler
{
	my $file = shift;
	create_rrdfile ($file,
		"DS:value:GAUGE:$HeartBeat:0:100"
	);
}

sub memory_create_handler
{
	my $file = shift;
	create_rrdfile ($file,
		"DS:used:GAUGE:$HeartBeat:0:U",
		"DS:free:GAUGE:$HeartBeat:0:U",
		"DS:frag:GAUGE:$HeartBeat:0:U"
	);
}

sub sessions_create_handler
{
	my $file = shift;
	create_rrdfile ($file,
		"DS:num:GAUGE:$HeartBeat:0:U",
		"DS:max:GAUGE:$HeartBeat:0:U"
	);
}

=head1 SEE ALSO

L<CapMan::Services>, RT#264250

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
