package CapMan::Plugins::MySQL;

use strict;
use warnings;

=head1 NAME

CapMan::Plugins::MySQL - CapMan plugin which doesn't do anything actually

=head1 DESCRIPTION

This is a plugin which shows you how plugins work.

=cut

use CapMan::Config qw(:rrdcreate get_rrdpath);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd create_rrdfile get_n_colors);

our $GraphDefIssues = ['-v', 'Issues/s',
	'DEF:value={filename}:value:AVERAGE',
	'LINE1:value#0000FF:Issues/s',
	'GPRINT:value:AVERAGE:%4.2lf Average,',
	'GPRINT:value:MAX:%4.2lf Max,',
	'GPRINT:value:LAST:%4.2lf Last'];

our $GraphDefThreads = ['-v', 'Threads',
	'DEF:running={filename}:running:AVERAGE',
	'DEF:connected={filename}:connected:AVERAGE',
	'DEF:cached={filename}:cached:AVERAGE',
	'DEF:created={filename}:created:AVERAGE',
	"CDEF:unknown=created,UNKN,+",
	"CDEF:cached_agg=connected,cached,+",
	"AREA:cached_agg#B8F0B8",
	"AREA:connected#B8B8F8",
	"AREA:running#F8B8B8",
	"LINE1:cached_agg#00E000:Cached   ",
	'GPRINT:cached:AVERAGE:%5.1lf Average,',
	'GPRINT:cached:MAX:%5.1lf Max,',
	'GPRINT:cached:LAST:%5.1lf Last\l',
	"LINE1:connected#0000FF:Connected",
	'GPRINT:connected:AVERAGE:%5.1lf Average,',
	'GPRINT:connected:MAX:%5.1lf Max,',
	'GPRINT:connected:LAST:%5.1lf Last\l',
	"LINE1:running#FF0000:Running  ",
	'GPRINT:running:AVERAGE:%5.1lf Average,',
	'GPRINT:running:MAX:%5.1lf Max,',
	'GPRINT:running:LAST:%5.1lf Last\l',
	"LINE1:unknown#F5F5F5:Created  ",
	'GPRINT:created:AVERAGE:%5.0lf Average,',
	'GPRINT:created:MAX:%5.0lf Max,',
	'GPRINT:created:LAST:%5.0lf Last\l'];

our $GraphDefTraffic = ['-v', 'Bits/s',
	'DEF:out_avg_raw={filename}:tx:AVERAGE',
	'DEF:inc_avg_raw={filename}:rx:AVERAGE',
	'CDEF:out_avg=out_avg_raw,8,*',
	'CDEF:inc_avg=inc_avg_raw,8,*',
	'CDEF:overlap=out_avg,inc_avg,GT,inc_avg,out_avg,IF',
	'CDEF:mytime=out_avg_raw,TIME,TIME,IF',
	'CDEF:sample_len_raw=mytime,PREV(mytime),-',
	'CDEF:sample_len=sample_len_raw,UN,0,sample_len_raw,IF',
	'CDEF:out_avg_sample=out_avg_raw,UN,0,out_avg_raw,IF,sample_len,*',
	'CDEF:out_avg_sum=PREV,UN,0,PREV,IF,out_avg_sample,+',
	'CDEF:inc_avg_sample=inc_avg_raw,UN,0,inc_avg_raw,IF,sample_len,*',
	'CDEF:inc_avg_sum=PREV,UN,0,PREV,IF,inc_avg_sample,+',
	"AREA:out_avg#B7EFB7",
	"AREA:inc_avg#B7B7F7",
	"AREA:overlap#89B3C9",
	"LINE1:out_avg#00E000:Outgoing",
	'GPRINT:out_avg:AVERAGE:%5.1lf%s Average,',
	'GPRINT:out_avg:MAX:%5.1lf%s Max,',
	'GPRINT:out_avg:LAST:%5.1lf%s Last',
	'GPRINT:out_avg_sum:LAST:(ca. %5.1lf%sB Total)\l',
	"LINE1:inc_avg#0000FF:Incoming",
	'GPRINT:inc_avg:AVERAGE:%5.1lf%s Average,',
	'GPRINT:inc_avg:MAX:%5.1lf%s Max,',
	'GPRINT:inc_avg:LAST:%5.1lf%s Last',
	'GPRINT:inc_avg_sum:LAST:(ca. %5.1lf%sB Total)\l'];

our $GraphDefQcache = ['-v', 'Queries/s',
	"DEF:hits_avg={filename}:hits:AVERAGE",
	"DEF:inserts_avg={filename}:inserts:AVERAGE",
	"DEF:not_cached_avg={filename}:not_cached:AVERAGE",
	"DEF:lowmem_prunes_avg={filename}:lowmem_prunes:AVERAGE",
	"DEF:queries_avg={filename}:queries_in_cache:AVERAGE",
	"CDEF:unknown=queries_avg,UNKN,+",
	"CDEF:not_cached_agg=hits_avg,inserts_avg,+,not_cached_avg,+",
	"CDEF:inserts_agg=hits_avg,inserts_avg,+",
	"CDEF:hits_agg=hits_avg",
	"AREA:not_cached_agg#F3DFB7",
	"AREA:inserts_agg#B7B7F7",
	"AREA:hits_agg#B7EFB7",
	"LINE1:not_cached_agg#F0A000:Not Cached      ",
	'GPRINT:not_cached_avg:AVERAGE:%5.2lf Avg,',
	'GPRINT:not_cached_avg:MAX:%5.2lf Max,',
	'GPRINT:not_cached_avg:LAST:%5.2lf Last\l',
	"LINE1:inserts_agg#0000FF:Inserts         ",
	'GPRINT:inserts_avg:AVERAGE:%5.2lf Avg,',
	'GPRINT:inserts_avg:MAX:%5.2lf Max,',
	'GPRINT:inserts_avg:LAST:%5.2lf Last\l',
	"LINE1:hits_agg#00E000:Hits            ",
	'GPRINT:hits_avg:AVERAGE:%5.2lf Avg,',
	'GPRINT:hits_avg:MAX:%5.2lf Max,',
	'GPRINT:hits_avg:LAST:%5.2lf Last\l',
	"LINE1:lowmem_prunes_avg#FF0000:Lowmem Prunes   ",
	'GPRINT:lowmem_prunes_avg:AVERAGE:%5.2lf Avg,',
	'GPRINT:lowmem_prunes_avg:MAX:%5.2lf Max,',
	'GPRINT:lowmem_prunes_avg:LAST:%5.2lf Last\l',
	"LINE1:unknown#F5F5F5:Queries in cache",
	'GPRINT:queries_avg:AVERAGE:%5.0lf Avg,',
	'GPRINT:queries_avg:MAX:%5.0lf Max,',
	'GPRINT:queries_avg:LAST:%5.0lf Last\l'];

our $GraphDefKeyblockAccess = ['-v', 'Accesses/s',
	"DEF:req={filename}:requests:AVERAGE",
	"DEF:acc={filename}:accesses:AVERAGE",
	"AREA:req#B7EFB7",
	"AREA:acc#F8B8B8",
	"LINE1:req#00E000:Requests",
	'GPRINT:req:AVERAGE:%5.2lf Avg,',
	'GPRINT:req:MAX:%5.2lf Max,',
	'GPRINT:req:LAST:%5.2lf Last\l',
	"LINE1:acc#FF0000:Accesses",
	'GPRINT:acc:AVERAGE:%5.2lf Avg,',
	'GPRINT:acc:MAX:%5.2lf Max,',
	'GPRINT:acc:LAST:%5.2lf Last\l'];

=head1 SERVICES PROVIDED

=over 4

=item name

This is the service which would be provided if this was a real plugin.

=back

=cut

register_input_handler ('mysql', '.1.3.6.1.4.1.12114.42.1', \&input_handler);
register_graph_handler ('mysql_command', $GraphDefIssues, \&graph_handler_commands);
register_graph_handler ('mysql_handler', $GraphDefIssues, \&graph_handler_handlers);
register_graph_handler ('mysql_threads', $GraphDefThreads);
register_graph_handler ('mysql_traffic', $GraphDefTraffic);
register_graph_handler ('mysql_qcache',  $GraphDefQcache);
register_graph_handler ('mysql_keyblock_access', $GraphDefKeyblockAccess);
register_create_handler ('mysql_command', \&create_handler_issues);
register_create_handler ('mysql_handler', \&create_handler_issues);
register_create_handler ('mysql_threads', \&create_handler_threads);
register_create_handler ('mysql_traffic', \&create_handler_traffic);
register_create_handler ('mysql_qcache',  \&create_handler_qcache);
register_create_handler ('mysql_keyblock_access',  \&create_handler_keyblock_access);

return (1);

sub input_handler
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $data = shift;

	my @values;
	my @keys;

	my %threads = (running => -1, connected => -1, cached => -1, created => -1);
	my %traffic = (sent => -1, received => -1);
	my %qcache  = (hits => -1, inserts => -1, not_cached => -1,
		lowmem_prunes => -1, queries_in_cache => -1);
	my %keyblock_read = (requests => -1, accesses => -1);
	my %keyblock_write = (requests => -1, accesses => -1);

	for (@$data)
	{
		my $data_set = $_;
		my $oid = $data_set->[0];
		my $num = $data_set->[1];
		my $val = $data_set->[2];

		if (!$num)
		{
			if ($oid =~ m/(.*)\.(\d+)$/)
			{
				$oid = $1;
				$num = $2;
			}
			else
			{
				print STDERR "Unknown OID and no NUM: OID = $oid\n";
				next;
			}
		}

		if ($oid =~ m/\.12114\.42\.1\.1\.1$/)
		{
			$keys[$num] = $val;
		}
		elsif ($oid =~ m/\.12114\.42\.1\.1\.2$/)
		{
			$values[$num] = $val;
		}
		else
		{
			print STDERR "Unknown OID: $oid\n";
			next;
		}
	}
			
	for (my $i = 0; ($i < @values) && ($i < @keys); $i++)
	{
		my $key = $keys[$i];
		my $value = $values[$i];

		next if (!$key or !defined $value);

		if ($key =~ m/^Com_(\w+)/)
		{
			my $command = $1;
			next if (!$value);
			write_to_rrd ($customer, $host, 'mysql_command', $command, $value);
			next;
		}
		elsif ($key =~ m/^Handler_(\w+)/)
		{
			my $handler = $1;
			next if (!$value);
			write_to_rrd ($customer, $host, 'mysql_handler', $handler, $value);
			next;
		}
		elsif ($key =~ m/^Threads_(running|connected|cached|created)$/)
		{
			$threads{$1} = $value;
			next;
		}
		elsif ($key =~ m/^Bytes_(sent|received)$/)
		{
			$traffic{$1} = $value;
			next;
		}
		elsif ($key =~ m/^Qcache_(hits|inserts|not_cached|lowmem_prunes|queries_in_cache)$/)
		{
			$qcache{$1} = $value;
			next;
		}
		elsif ($key =~ m/^Key_(read|write)((?:_request)?s)$/)
		{
			my $rw = $1;
			my $type = $2 eq 's' ? 'accesses' : 'requests';
			if ($rw eq 'read')
			{
				$keyblock_read{$type} = $value;
			}
			else
			{
				$keyblock_write{$type} = $value;
			}
		}

		print "$key = $value\n" if ($::DEBUG);
	}

	if ($threads{'created'} > 0)
	{
		write_to_rrd ($customer, $host, 'mysql_threads', '', @threads{qw(running connected cached created)});
	}

	if (($traffic{'sent'} > 0) && ($traffic{'received'} > 0))
	{
		write_to_rrd ($customer, $host, 'mysql_traffic', '',
			$traffic{'received'}, $traffic{'sent'});
	}

	if ($qcache{'queries_in_cache'} > 0)
	{
		write_to_rrd ($customer, $host, 'mysql_qcache', '',
			@qcache{qw(hits inserts not_cached lowmem_prunes queries_in_cache)});
	}
	else
	{
		print STDERR "queries_in_cache <= 0\n";
	}

	if ($keyblock_read{'requests'} > 0)
	{
		write_to_rrd ($customer, $host, 'mysql_keyblock_access', 'read',
			$keyblock_read{'requests'}, $keyblock_read{'accesses'});
	}
	if ($keyblock_write{'requests'} > 0)
	{
		write_to_rrd ($customer, $host, 'mysql_keyblock_access', 'write',
			$keyblock_write{'requests'}, $keyblock_write{'accesses'});
	}

	return;
}

sub create_handler_issues ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:value:COUNTER:$HeartBeat:0:U");
}

sub create_handler_threads
{
	my $file = shift;

	create_rrdfile ($file, "DS:running:GAUGE:$HeartBeat:0:U",
		"DS:connected:GAUGE:$HeartBeat:0:U",
		"DS:cached:GAUGE:$HeartBeat:0:U",
		"DS:created:COUNTER:$HeartBeat:0:U");
}

sub create_handler_traffic ($)
{
	my $file = shift;

	create_rrdfile ($file,
		"DS:rx:COUNTER:$HeartBeat:0:U",
		"DS:tx:COUNTER:$HeartBeat:0:U");
}

sub create_handler_qcache ($)
{
	my $file = shift;

	create_rrdfile ($file,
		"DS:hits:COUNTER:$HeartBeat:0:U",
		"DS:inserts:COUNTER:$HeartBeat:0:U",
		"DS:not_cached:COUNTER:$HeartBeat:0:U",
		"DS:lowmem_prunes:COUNTER:$HeartBeat:0:U",
		"DS:queries_in_cache:GAUGE:$HeartBeat:0:U");
}

sub create_handler_keyblock_access ($)
{
	my $file = shift;

	create_rrdfile ($file,
		"DS:requests:COUNTER:$HeartBeat:0:U",
		"DS:accesses:COUNTER:$HeartBeat:0:U");
}

sub graph_handler_issues
{
	confess ("Wrong number of arguments") if (@_ < 6);

	my $customer = shift;
	my $host  = shift;
	my $begin = shift;
	my $end   = shift;
	my $title = shift;
	my @files = @_;

	my @colors = get_n_colors (scalar (@files));

	my @names;
	my $names_max = 0;
	my $num = 0;

	my @cmd = ('-', '-s', $begin, '-e', $end, '-a', 'PNG', '-t', $title, '-v', 'Issues/s', '--height', '300');
	my $path = get_rrdpath ();

	for (@files)
	{
		my $file = "$path/$customer/$host/$_";
		my $name = (split (m/-/, $_))[1];

		$name =~ s/\.rrd$//;
		$name =~ s/_+/ /g;

		push (@cmd, qq(DEF:val$num=$file:value:AVERAGE));
		$num++;

		push (@names, $name);
		if (length ($name) > $names_max)
		{
			$names_max = length ($name);
		}
	}

	my $format = "%-${names_max}s";
	for (my $i = 0; $i < $num; $i++)
	{
		my $line = sprintf ($format, $names[$i]);
		my $color = $colors[$i % scalar (@colors)];
		
		push (@cmd, 'LINE1:val' . $i . '#' . $color . ":$line");
		push (@cmd, "GPRINT:val$i:AVERAGE:\%5.1lf Average,",
			"GPRINT:val$i:MAX:\%5.1lf Max,",
			"GPRINT:val$i:LAST:\%5.1lf Last\\n");
	}

	RRDs::graph (@cmd);

	die (RRDs::error ()) if (RRDs::error ());
}

sub graph_handler_commands
{
	my $cust  = shift;
	my $host  = shift;
	my $begin = shift;
	my $end   = shift;
	my @files = @_;
	graph_handler_issues ($cust, $host, $begin, $end, "MySQL commands on $host", @files);
}

sub graph_handler_handlers
{
	my $cust  = shift;
	my $host  = shift;
	my $begin = shift;
	my $end   = shift;
	my @files = @_;
	graph_handler_issues ($cust, $host, $begin, $end, "MySQL handlers on $host", @files);
}

=head1 SEE ALSO

L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
