package CapMan::Plugins::Load;

use strict;
use warnings;

=head1 NAME

CapMan::Plugins::Load - CapMan plugin which queries the system load of unix
installations.

=head1

All Unix platform offer values called "System Load" that give a general
impression of the systems usage. Though not very accurate these values do say
something about a system. This plugin graphs these nubers.

=cut

use CapMan::Config qw(:rrdcreate);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd create_rrdfile);

our $GraphDef = ['-v', 'Load',
	'DEF:a={filename}:one:AVERAGE',
	'DEF:b={filename}:five:AVERAGE',
	'DEF:c={filename}:ten:AVERAGE',
	'AREA:a#00FF00: 1 Minute Average',
	'GPRINT:a:AVERAGE:%4.1lf Average,',
	'GPRINT:a:MAX:%4.1lf Max,',
	'GPRINT:a:LAST:%4.1lf Last\n',
	'LINE1:b#0000FF: 5 Minute Average',
	'GPRINT:b:AVERAGE:%4.1lf Average,',
	'GPRINT:b:MAX:%4.1lf Max,',
	'GPRINT:b:LAST:%4.1lf Last\n',
	'LINE1:c#FF0000:15 Minute Average',
	'GPRINT:c:AVERAGE:%4.1lf Average,',
	'GPRINT:c:MAX:%4.1lf Max,',
	'GPRINT:c:LAST:%4.1lf Last'];

=head1 SERVICES PROVIDED

=over 4

=item load

Handler for the system-load

=back

=cut

register_input_handler ('load', '.1.3.6.1.4.1.2021.10.1.5', \&load_handler);
register_create_handler ('load', \&load_create_handler);
register_graph_handler ('load', $GraphDef);

return (1);

sub load_handler ($$$)
{
	confess ("Wrong number of arguments") if (@_ != 4);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $data = shift;

	my @values = ();

	$data->[0][2] = -100 unless (defined ($data->[0][2]));
	$data->[1][2] = -100 unless (defined ($data->[1][2]));
	$data->[2][2] = -100 unless (defined ($data->[2][2]));

	$values[0] = $data->[0][2] / 100;
	$values[1] = $data->[1][2] / 100;
	$values[2] = $data->[2][2] / 100;

	write_to_rrd ($customer, $host, $srv, '', @values);
}

sub load_create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file,
		"DS:one:GAUGE:$HeartBeat:0:256",
		"DS:five:GAUGE:$HeartBeat:0:256",
		"DS:ten:GAUGE:$HeartBeat:0:256"
	);
}

=head1 SEE ALSO

L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
