package CapMan::Plugins::Infratec;

use strict;
use warnings;

use RRDs;

=head1 NAME

CapMan::Plugins::Infratec - CapMan plugin for Infratec "Rack Monitoring
Systems" (RMS)

=head1

This plugin queries the temperature, humidity and voltage from the RMS. Other
data may be queried in the future. The supported models are H2-17 and H2-30.
Other models should be easy to adopt.

=cut

use CapMan::Config qw(:rrdcreate);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(write_to_rrd);

=head1 SERVICES PROVIDED

=over 4

=item infratec_h2_17_temp, infratec_h2_30_temp

Handler for the thermometers.

=item infratec_h2_17_hum, infratec_h2_30_hum

Handler for the humidity sensors.

=item infratec_h2_17_voltage, infratec_h2_30_voltage

Handler for the voltage sensors.

=back

=cut

register_input_handler ('infratec_h2_17_input',   '.1.3.6.1.4.1.4519.10.2.1.1', \&infratec_h2_17_input_handler);
register_input_handler ('infratec_h2_17_alarm',   '.1.3.6.1.4.1.4519.10.2.2.1', \&infratec_h2_17_alarm_handler);
register_input_handler ('infratec_h2_17_ups',     '.1.3.6.1.4.1.4519.10.2.3.1', \&infratec_h2_17_ups_handler);
register_input_handler ('infratec_h2_17_output',  '.1.3.6.1.4.1.4519.10.3.1.1', \&infratec_h2_17_output_handler);
register_input_handler ('infratec_h2_17_temp',    '.1.3.6.1.4.1.4519.10.4.1.1', \&infratec_h2_17_temp_handler);
register_input_handler ('infratec_h2_17_hum',     '.1.3.6.1.4.1.4519.10.5.1.1', \&infratec_h2_17_hum_handler);
register_input_handler ('infratec_h2_17_voltage', '.1.3.6.1.4.1.4519.10.6.1.1', \&infratec_h2_17_voltage_handler);

register_input_handler ('infratec_h2_30_input',   '.1.3.6.1.4.1.1909.10.2.1.1', \&infratec_h2_30_input_handler);
register_input_handler ('infratec_h2_30_alarm',   '.1.3.6.1.4.1.1909.10.2.2.1', \&infratec_h2_30_alarm_handler);
register_input_handler ('infratec_h2_30_ups',     '.1.3.6.1.4.1.1909.10.2.3.1', \&infratec_h2_30_ups_handler);
register_input_handler ('infratec_h2_30_output',  '.1.3.6.1.4.1.1909.10.3.1.1', \&infratec_h2_30_output_handler);
register_input_handler ('infratec_h2_30_temp',    '.1.3.6.1.4.1.1909.10.4.1.1', \&infratec_h2_30_temp_handler);
register_input_handler ('infratec_h2_30_hum',     '.1.3.6.1.4.1.1909.10.5.1.1', \&infratec_h2_30_hum_handler);
register_input_handler ('infratec_h2_30_voltage', '.1.3.6.1.4.1.1909.10.6.1.1', \&infratec_h2_30_voltage_handler);

return (1);

sub infratec_h2_30_input_handler
{
	infratec_binary_handler (@_, 'Input', '.1.3.6.1.4.1.1909.10.2.1.1');
}

sub infratec_h2_17_input_handler
{
	infratec_binary_handler (@_, 'Input', '.1.3.6.1.4.1.4519.10.2.1.1');
}

sub infratec_h2_30_alarm_handler
{
	infratec_binary_handler (@_, 'Alarm', '.1.3.6.1.4.1.1909.10.2.2.1');
}

sub infratec_h2_17_alarm_handler
{
	infratec_binary_handler (@_, 'Alarm', '.1.3.6.1.4.1.4519.10.2.2.1');
}

sub infratec_h2_30_ups_handler
{
	infratec_binary_handler (@_, 'UPS', '.1.3.6.1.4.1.1909.10.2.3.1');
}

sub infratec_h2_17_ups_handler
{
	infratec_binary_handler (@_, 'UPS', '.1.3.6.1.4.1.4519.10.2.3.1');
}

sub infratec_h2_30_output_handler
{
	infratec_binary_handler (@_, 'Output', '.1.3.6.1.4.1.1909.10.3.1.1');
}

sub infratec_h2_17_output_handler
{
	infratec_binary_handler (@_, 'Output', '.1.3.6.1.4.1.4519.10.3.1.1');
}

sub infratec_binary_handler
{
	confess ("Wrong number of arguments") if (@_ != 6);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $data = shift;
	my $type = shift;
	my $prefix = shift;

	my $i;
	my $hash = {};

	my $name_prefix = "$prefix.2";
	my $value_prefix = ($type =~ m/Output/i) ? "$prefix.7" : "$prefix.3";

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;
		$hash->{$name}[$seq] = $val;
	}

	if (!$hash->{$name_prefix})
	{
		print STDERR "$host/$srv/${type}: No names are defined..\n";
		return;
	}

	return unless ($hash->{$name_prefix} and $hash->{$value_prefix});
	
	my @names  = @{$hash->{$name_prefix}};
	my @values = map { ($_ || 0) == 2 ? 1 : 0 } (@{$hash->{$value_prefix}});

	for ($i = 0; $i < @names; $i++)
	{
		my $name  = $names[$i];
		my $value = $values[$i];

		# Ignore ports with an empty description..
		next unless ($name);
		
		write_to_rrd ($customer, $host, 'binary', "$type $name", $value);
	}
}

sub infratec_h2_17_temp_handler
{
	infratec_temp_handler (@_, '.1.3.6.1.4.1.4519.10.4.1.1');
}

sub infratec_h2_30_temp_handler
{
	infratec_temp_handler (@_, '.1.3.6.1.4.1.1909.10.4.1.1');
}

sub infratec_temp_handler
{
	confess ("Wrong number of arguments") if (@_ != 5);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $data = shift;
	my $prefix = shift;

	my $i;
	my $hash = {};

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;
		$hash->{$name}[$seq] = $val;
	}
	
	for ($i = 1; $i <= 4; $i++)
	{
		if (defined ($hash->{"$prefix.2"}[$i])
				and defined ($hash->{"$prefix.3"}[$i]))
		{
			my $name = $hash->{"$prefix.2"}[$i];
			my $temp = $hash->{"$prefix.3"}[$i];

			next if (($temp < -50) or ($temp > 100));

			write_to_rrd ($customer, $host, 'temperature', $name, $temp);
		}
	}
}

# Humidity
sub infratec_h2_17_hum_handler
{
	infratec_hum_handler (@_, '.1.3.6.1.4.1.4519.10.5.1.1');
}

sub infratec_h2_30_hum_handler
{
	infratec_hum_handler (@_, '.1.3.6.1.4.1.1909.10.5.1.1');
}

sub infratec_hum_handler
{
	confess ("Wrong number of arguments") if (@_ != 5);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $data = shift;
	my $prefix = shift;

	my $i;
	my $hash = {};

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;
		$hash->{$name}[$seq] = $val;
	}
	
	for ($i = 1; $i <= 4; $i++)
	{
		if (defined ($hash->{"$prefix.2"}[$i])
				and defined ($hash->{"$prefix.3"}[$i]))
		{
			my $name = $hash->{"$prefix.2"}[$i];
			my $humi = $hash->{"$prefix.3"}[$i];

			if ($humi != 255)
			{
				write_to_rrd ($customer, $host, 'humidity', $name, $humi);
			}
		}
	}
}

sub infratec_h2_17_voltage_handler
{
	infratec_voltage_handler (@_, '.1.3.6.1.4.1.4519.10.6.1.1');
}

sub infratec_h2_30_voltage_handler
{
	infratec_voltage_handler (@_, '.1.3.6.1.4.1.1909.10.6.1.1');
}

sub infratec_voltage_handler
{
	confess ("Wrong number of arguments") if (@_ != 5);

	my $customer = shift;
	my $host = shift;
	my $srv = shift;
	my $data = shift;
	my $prefix = shift;

	my $i;
	my $hash = {};

	for (@$data)
	{
		my ($name, $seq, $val, $type) = @$_;
		$hash->{$name}[$seq] = $val;
	}

	for ($i = 1; $i <= 3; $i++)
	{
		if (defined ($hash->{"$prefix.3"}[$i]))
		{
			my $val = $hash->{"$prefix.3"}[$i];

			write_to_rrd ($customer, $host, 'voltage', $i, $val);
		}
	}
}

=head1 SEE ALSO

L<http://www.infratec-ag.de/>, L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
