package CapMan::Plugins::Environment;

use strict;
use warnings;

=head1 NAME

CapMan::Plugins::Environment - CapMan plugin for temperature and humidity.

=head1 DESCRIPTION

Output plugins for temperature and humidity

=cut

use CapMan::Config qw(:rrdcreate);
use CapMan::Services qw(:plugin);
use CapMan::Tools qw(create_rrdfile);

our $GraphDefTemp = ['-v', 'Celsius',
	'DEF:t={filename}:temp:AVERAGE',
	'CDEF:neg=t,0,GE,UNKN,t,IF',
	'CDEF:pos=t,0,GE,t,UNKN,IF',
	'AREA:pos#FFDDDD',
	'AREA:neg#DDDDFF',
	'LINE1:t#000000:Temperature  ',
	'GPRINT:t:AVERAGE:%5.1lfC Average,',
	'GPRINT:t:MIN:%5.1lfC Min,',
	'GPRINT:t:MAX:%5.1lfC Max,',
	'GPRINT:t:LAST:%5.1lfC Last'];

our $GraphDefHum = ['-v', 'Percent',
	'DEF:h={filename}:humidity:AVERAGE',
	'LINE1:h#000000:Humidity  ',
	'GPRINT:h:AVERAGE:%5.1lf%% Average,',
	'GPRINT:h:MIN:%5.1lf%% Min,',
	'GPRINT:h:MAX:%5.1lf%% Max,',
	'GPRINT:h:LAST:%5.1lf%% Last'];

=head1 CREATE HANDLER

=over 4

=item temperature

=item humidity

=back

=cut

register_create_handler ('temperature', \&temperature_create_handler);
register_create_handler ('humidity',    \&humidity_create_handler);

=head1 GRAPH HANDLER

=over 4

=item B<temperature> (I<temperature in degrees celsius>)

=item B<humidity> (I<humidity in percent>)

=back

=cut

register_graph_handler ('temperature', $GraphDefTemp);
register_graph_handler ('humidity',    $GraphDefHum);

return (1);

sub temperature_create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:temp:GAUGE:$HeartBeat:-273:U");
}

sub humidity_create_handler ($)
{
	my $file = shift;

	create_rrdfile ($file, "DS:humidity:GAUGE:$HeartBeat:0:U");
}

=head1 SEE ALSO

L<CapMan::Services>

=head1 AUTHOR

Florian octo Forster E<lt>octo@noris.netE<gt> for the noris network AG
L<http://noris.net/>
