#!/usr/bin/perl

use strict;
use warnings;

use File::ReadBackwards ();

our $data = {};

=head1 WHAT IS THIS?

This is a very small perlscript, that collects some basic information about the
computer it runs on and displays them on STDOUT. This script is supposed to be
called by snmpd.

The entry in the F<snmpd.conf> looks like this:

  exec .1.3.6.1.4.1.2021.50 sysstat /var/snmp/bin/sysstat.pl

=head1 COLLECTED DATA

=over 4

=item From F</proc/stat>

CPU usage, page and swap usage, interrupts, context switches, forkrate, disk-IO

=cut

my $fh;
open ($fh, '< /proc/stat') or die $!;
while (<$fh>)
{
	chomp;

	my @d = split (' ', $_);
	my $key = shift (@d);

	if ($key =~ m/^cpu(\d+)$/)
	{
		if (@d == 4)
		{
			$data->{"cpu$1"} = join (',', @d);
		}
		elsif (@d == 7)
		{
			# Steal time is missing
			$data->{"cpu_linux26-$1"} = join (',', @d);
		}
		elsif (@d >= 8)
		{
			# $d[7] == steal time
			$data->{"cpu_linux26-$1"} = join (',', @d[0..7]);
		}
	}
	elsif ($key =~ m/^(page|swap)$/)
	{
		$data->{$key} = join (',', @d[0,1]);
	}
	elsif ($key =~ m/^(intr|ctxt|btime|processes)$/)
	{
		$data->{$key} = $d[0];
	}
	elsif ($key =~ m/^disk_io/)
	{
		for (@d)
		{
			next unless (m/^\((\d+,\d+)\):\((\d+,\d+,\d+,\d+,\d+)\)$/);
			$data->{"disk_io_$1"} = $2;
		}
	}
	else
	{
		#print STDERR "Unknown key: $key\n";
	}
}
close ($fh);

if ((!$data->{'swap'} || !$data->{'page'}) && (-e '/proc/vmstat'))
{
	if (open ($fh, '/proc/vmstat'))
	{
		my $pgpgin;
		my $pgpgout;
		my $pswpin;
		my $pswpout;

		while (<$fh>)
		{
			my ($key, $value) = split (' ', $_);
			if ($key eq 'pgpgin') { $pgpgin = $value; }
			elsif ($key eq 'pgpgout') { $pgpgout = $value; }
			elsif ($key eq 'pswpin') { $pswpin = $value; }
			elsif ($key eq 'pswpout') { $pswpout = $value; }
		}
		close ($fh);

		if (defined ($pgpgin) && defined ($pgpgout))
		{
			$data->{'page'} = $pgpgin . ',' . $pgpgout;
		}
		if (defined ($pswpin) && defined ($pswpout))
		{
			$data->{'swap'} = $pswpin . ',' . $pswpout;
		}
	}
}

=item From F</proc/meminfo>

Used memory, buffer-cache, page-cache, free memory

=cut

open ($fh, '< /proc/meminfo') or die $!;
{
	my $total  = 0;
	my $buffer = 0;
	my $pagec  = 0;
	my $free   = 0;
	my $used   = 0;

	my @swap = (undef, undef, undef);
	
	while (<$fh>)
	{
		my $key;
		my $val;

		chomp;

		if (m/^(\w+):\s+(\d+) kB$/)
		{
			$key = lc ($1);
			$val = $2 * 1024;
		}
		else
		{
			next;
		}

		if ($key eq 'memtotal')
		{
			$total = $val;
		}
		elsif ($key eq 'memfree')
		{
			$free = $val;
		}
		elsif ($key eq 'buffers')
		{
			$buffer = $val;
		}
		elsif ($key eq 'cached')
		{
			$pagec = $val;
		}
		elsif ($key =~ m/^Swap(Cached|Total|Free)$/i)
		{
			my $type = $1;
			if ($type =~ m/Cached/i)   { $swap[2] = $val; }
			elsif ($type =~ m/Total/i) { $swap[0] = $val; }
			elsif ($type =~ m/Free/i)  { $swap[1] = $val; }
		}
	}

	$used = $total - ($buffer + $pagec + $free);

	$data->{'meminfo'} = join (',', $used, $buffer, $pagec, $free);

	if (defined ($swap[0]) && defined ($swap[1]) && defined ($swap[2]))
	{
		$data->{'swapinfo'} = join (',', @swap);
	}
}
close ($fh);

if (open ($fh, '< /proc/diskstats'))
{
	# Linux 2.6
	while (<$fh>)
	{
		my @fields = split (' ', $_);
		if (@fields == 7)
		{
			if (grep { $_ != 0 } (@fields[3..6]))
			{
				$data->{'partition_' . $fields[2]} = join (',', @fields[3..6]);
			}
		}
		elsif (@fields == 14)
		{
			if (grep { $_ != 0 } (@fields[3..13]))
			{
				$data->{'disk_' . $fields[2]} = join (',', @fields[3..13]);
			}
		}
	}
	close ($fh);
}
elsif (open ($fh, '< /proc/partitions'))
{
	# Linux 2.4
	while (<$fh>)
	{
		my @fields = split (' ', $_);
		next if (@fields != 15);
		next if (($fields[0] =~ m/\D/) || ($fields[1] =~ m/\D/));

		next if (!grep { $_ != 0 } (@fields[4..14]));

		if ($fields[1] == 0) # disk
		{
			$data->{'disk_' . $fields[3]} = join (',', @fields[4..14]);
		}
		else
		{
			$data->{'partition_' . $fields[3]} = join (',', @fields[4,6,8,10]);
		}
	}
	close ($fh);
}

=item From I<mysqladmin status>

MySQL queries

=cut

if (-e '/var/lib/mysql/mysql.sock' and -e '/usr/bin/mysqladmin')
{
	$_ = qx(mysqladmin status 2>/dev/null);

	if (m/Questions: (\d+)/)
	{
		$data->{'mysql_queries'} = $1;
	}
}

if (-e '/var/log/named/named.stats' && -x '/usr/sbin/rndc')
{
	while (42)
	{
		my $frb;

		if (system ('/usr/sbin/rndc stats') != 0)
		{
			last;
		}

		$frb = File::ReadBackwards->new ('/var/log/named/named.stats') or last;
		while (my $line = $frb->readline ())
		{
			chomp ($line);
			if ($line =~ m/^--- Statistics Dump --- \((\d+)\)$/)
			{
				last if ((time () - $1) > 240);
			}
			elsif ($line =~ m/^\+\+\+ Statistics Dump \+\+\+ \((\d+)\)$/)
			{
				last;
			}
			elsif ($line =~ m/^(\w+)\s+(\d+)$/)
			{
				$data->{'bind_stat-' . lc ($1)} = $2;
			}
		} # while (ReadBackwards)

		$frb->close ();
		last;
	} # while (42)
} # if (named stats)

if (-d '/owfs')
{
	my $dh;
	my $sensors = [];

	opendir ($dh, "/owfs") or die ("opendir (/owfs): $!");
	while (my $file = readdir ($dh))
	{
		if ((-e "/owfs/$file/temperature")
			&& ($file =~ m/^([0-9a-fA-F][0-9a-fA-F])\.([0-9a-fA-F]{12})(?:[0-9a-fA-F][0-9a-fA-F])?$/))
		{
			push (@$sensors, { family => $1, id => $2, file => $file });
		}
	}
	closedir ($dh);

	for (@$sensors)
	{
		my $sensor = $_;
		my $filename = "/owfs/" . $sensor->{'file'} . '/temperature';
		my $fh;
		my $temperature;

		open ($fh, "< $filename");
		if (!$fh)
		{
			warn ("open ($filename): $!");
			next;
		}
		$temperature = <$fh>;
		close ($fh);

		$temperature =~ s/^\s+//;
		$temperature =~ s/\s+$//;

		$data->{"temperature-" . $sensor->{'id'}} = $temperature;
	} # for (@$sensors)
}

print STDOUT $_ . '=' . $data->{$_} . "\n" for (sort (keys (%$data)));

exit (0);

=back

=head1 AUTHOR

Florian Forster E<lt>octo at noris.netE<gt> for the noris network AG. 

=cut
