# --
# Ticket/Number/DBAutoIncrement.pm - a ticket number auto increment generator
# using the database to store the counter state. Also adds a 1 digit checksum.
#
# Note: This is used like a method of a Kernel::System::Ticket object.
# available objects are: ConfigObject, LogObject and DBObject

package Kernel::System::Ticket::Number::DBAutoIncrementChecksum;

use strict;
use warnings;

use vars qw($VERSION);
$VERSION = qw($Revision$) [1];

sub TicketCreateNumber {
    my ( $Self, $JumpCounter ) = @_;
    if ( !$JumpCounter ) {
        $JumpCounter = 0;
    }

    # get needed config options
    my $SystemID   = $Self->{ConfigObject}->Get('SystemID');
    my $MinSize
        = $Self->{ConfigObject}->Get('Ticket::NumberGenerator::AutoIncrement::MinCounterSize')
        || $Self->{ConfigObject}->Get('Ticket::NumberGenerator::MinCounterSize')
        || 5;

    while (1) {
        # read count
        if (! $Self->{DBObject}->Do( SQL  => 'INSERT into  next_ticket_number values ()' ) ) {
            $Self->{LogObject}->Log(
                Priority => 'error',
                Message  => "INSERT into next_ticket_number failed.",
            );
            return;
        }
        $Self->{DBObject}->Prepare( SQL => 'Select last_insert_id()' );
        my $Count = ($Self->{DBObject}->FetchrowArray())[0];
        $Self->{LogObject}->Log(
            Priority => 'debug',
            Message  => "Read counter from next_ticket_number: $Count",
        );

        # pad ticket number with leading '0' to length $MinSize (config option)
        while ( length($Count) < $MinSize ) {
            $Count = '0' . $Count;
        }
    
        # create new ticket number
        my $Tn = $SystemID . $Count;
        
        # crc
        my $crc = 0;
        my $faktor = 3;
        for my $i (0 .. length($Tn)-1) {
            $crc += $faktor * substr($Tn,$i,1);
            $faktor = 4 - $faktor;
        }
        $Tn .= $crc % 10;
    
        # Already in use?
        if ( ! $Self->TicketCheckNumber( Tn => $Tn ) ) {
            # No. Go ahead.
            return $Tn;
        }
        
        # Oops, $Tn is already in use. Should not happen.
        # Emergency workaround: Get a new one in next loop run,
        # unless we have tried too often. 
        $Self->{LoopProtectionCounter}++;
        if ( $Self->{LoopProtectionCounter} >= 16000 ) {  
            # loop protection
            $Self->{LogObject}->Log(
                Priority => 'error',
                Message  => "CounterLoopProtection is now $Self->{LoopProtectionCounter}!"
                    . " Stopped TicketCreateNumber()!",
            );
            return;
        }
        
        $Self->{LogObject}->Log(
            Priority => 'notice',
            Message  => "Tn ($Tn) exists! Creating a new one.",
        );
    }
    # never reached
}

sub GetTNByString {
    my ( $Self, $String ) = @_;
    if ( !$String ) {
        return;
    }

    # get needed config options
    my $CheckSystemID = $Self->{ConfigObject}->Get('Ticket::NumberGenerator::CheckSystemID');
    my $SystemID      = '';
    if ($CheckSystemID) {
        $SystemID = $Self->{ConfigObject}->Get('SystemID');
    }
    my $TicketHook        = $Self->{ConfigObject}->Get('Ticket::Hook::Subject');
    my $TicketHookDivider = $Self->{ConfigObject}->Get('Ticket::HookDivider');

    # check ticket number
    if ( $String =~ /\Q$TicketHook\E\s*\Q$TicketHookDivider\E($SystemID\d+)/i ) {
        return $1;
    }

    if ( $String =~ /\Q$TicketHook\E:\s{0,2}($SystemID\d+)/i ) {
        return $1;
    }

    return;
}

1;
