# --
# Kernel/System/Ticket/Event/Test.pm - test event module
# Copyright (C) 2001-2008 OTRS AG, http://otrs.org/
# --
# $Id$
# --
# This software comes with ABSOLUTELY NO WARRANTY. For details, see
# the enclosed file COPYING for license information (GPL). If you
# did not receive this file, see http://www.gnu.org/licenses/gpl-2.0.txt.
# --

package Kernel::System::Ticket::Event::NorisFollowupWorkflow;

use strict;
use warnings;

use vars qw($VERSION);
$VERSION = qw($Revision$) [1];

sub new {
    my ( $Type, %Param ) = @_;

    # allocate new hash for object
    my $Self = {};
    bless( $Self, $Type );

    # get needed objects
    for (qw(ConfigObject DBObject EncodeObject TicketObject LogObject TimeObject UserObject CustomerUserObject MainObject SendmailObject)) {
        $Self->{$_} = $Param{$_} || die "Got no $_!";
    }

    return $Self;
}

sub Run {
    my ( $Self, %Param ) = @_;

    # check needed stuff
    for (qw(TicketID Event Config)) {
        if ( !$Param{$_} ) {
            $Self->{LogObject}->Log( Priority => 'error', Message => "Need $_!" );
            return;
        }
    }
    
    # sanity check
    return 1 if !$Param{TicketID} || !$Param{UserID};

    # get ticket
    my %Ticket = $Self->{TicketObject}->TicketGet(
        TicketID => $Param{TicketID},
    );

    my $wakeup_pattern;
    my $message;
    
    # if ticket is closed, it's OK.
    if (defined $Ticket{StateType}
        && $Ticket{StateType} eq 'closed' 
        && defined $Param{PreviousStateType}
        && $Param{PreviousStateType} ne 'closed') {
        # this ticket has been closed. 
        # Action: - leave a note on all parents
        #         - wake up pending parents
        $wakeup_pattern = qr '^pending reminder|parking|needinfo|retry|remind';
        $message = "closed";
    }
    elsif (defined $Ticket{StateType}
           && $Ticket{StateType} !~ m{ \A (close|merge|remove) }
           && defined $Param{PreviousStateType}
           && $Param{PreviousStateType} eq 'closed' ) {
           # this ticket has been re-opened. 
           # Action: - leave a note on all parents
           #         - re-open all parents that are not open.
           $wakeup_pattern = qr 'close|pending|parking|needinfo|retry|remind';
           $message = 're-opened';
    }
    elsif (defined $Ticket{StateType}
           && $Ticket{StateType} !~ m{ \A (close|merge|remove) }
           && ! defined $Param{PreviousStateType}) {
           # this is a new subticket, wake only parents that are closed
           $wakeup_pattern = qr 'close|pending auto close';
           $message = 'created';
    }
    else {
        # nothing interesting happened.
        return 1;
    }
    
    # create new instance of the link object
    if ( !$Self->{LinkObject} ) {
        $Self->{LinkObject} = Kernel::System::LinkObject->new(
            %{$Self},
            %Param,
        );
    }

    # link tickets
    my $Links = $Self->{LinkObject}->LinkList(
        Object => 'Ticket',
        Key    => $Param{TicketID},
        State  => 'Valid',
        Type   => 'MainSub',
        UserID => $Param{UserID},
    );

    return 1 if !$Links;
    return 1 if ref $Links ne 'HASH';
    return 1 if !$Links->{Ticket};
    return 1 if ref $Links->{Ticket} ne 'HASH';
    return 1 if !$Links->{Ticket}->{MainSub};
    return 1 if ref $Links->{Ticket}->{MainSub} ne 'HASH';
    return 1 if !$Links->{Ticket}->{MainSub}->{Source};
    return 1 if ref $Links->{Ticket}->{MainSub}->{Source} ne 'HASH';
    
    my %Article = $Self->{TicketObject}->ArticleLastArticle( TicketID => $Ticket{TicketID} );
    my $TN = $Ticket{TicketNumber};
    my $Subject = "#$TN $message: $Article{Subject}";
    my $ChildIsLocked = $Self->{TicketObject}->LockIsTicketLocked(TicketID => $Param{TicketID});
    my $ChildOwnerID = $Ticket{OwnerID};

    for my $TicketID ( sort keys %{$Links->{Ticket}->{MainSub}->{Source} } ) {

        # get ticket
        my %ParentTicket = $Self->{TicketObject}->TicketGet(
            TicketID => $TicketID,
        );
        if ( $Ticket{StateType} !~ m{ \A (merge|remove) }xms ) {
            if ($ParentTicket{State} =~ $wakeup_pattern) {
                $Self->{TicketObject}->StateSet(
                    State => 'open',
                    TicketID => $TicketID,
                    UserID => $Param{UserID},
                );
            }
            if ( $Self->{TicketObject}->LockIsTicketLocked(TicketID => $TicketID)
                 && ($ParentTicket{OwnerID} == $Param{UserID}
                     ||  $ChildIsLocked && $ParentTicket{OwnerID} == $ChildOwnerID )  )
            {
                # parent ticket belongs to current agent or owner of child ticket, 
                # a history entry is sufficient
                $Self->{TicketObject}->HistoryAdd(
                    Name => $Subject,
                    HistoryType => 'Misc',
                    TicketID => $TicketID,
                    UserID => $Param{UserID},
                    CreateUserID => $Param{UserID},
                );
            }
            else {
                my %User = $Self->{UserObject}->GetUserData( UserID => $Param{UserID} );
                $Self->{TicketObject}->ArticleCreate(
                    TicketID => $TicketID,
                    ArticleType => 'note-internal',
                    SenderType => 'system',
                    Subject => $Subject,
                    Body => '',
                    ContentType => 'text/plain',
                    From => "$User{UserFirstname} $User{UserLastname} <$User{UserEmail}>",
                    ReplyTo => $Self->{TicketObject}->TicketEnvelopeSender(TicketID => $Param{TicketID}),
                    HistoryType => 'Misc',
                    HistoryComment => $Subject,
                    UserID => $Param{UserID},
                    CreateUserID => $Param{UserID},
                );
            }
        }
    }

    return 1;
}

1;
