# --
# Kernel/System/Ticket/Acl/RestrictStatesIfNotOwner.pm - acl module
# - limit accessible states if agent is not the owner of a ticket -

package Kernel::System::Ticket::Acl::RestrictStatesIfNotOwner;

use strict;
use warnings;

use vars qw($VERSION);
$VERSION = qw($Revision$) [1];

sub new {
    my ( $Type, %Param ) = @_;

    # allocate new hash for object
    my $Self = {};
    bless( $Self, $Type );

    # get needed objects
    for (
        qw(ConfigObject DBObject TicketObject LogObject UserObject CustomerUserObject MainObject TimeObject)
        )
    {
        $Self->{$_} = $Param{$_} || die "Got no $_!";
    }

    return $Self;
}

sub Run {
    my ( $Self, %Param ) = @_;
    
    if (! $Param{TicketID}) {
        # we're only interested in existing tickets.
        return 1;
    }

    # check needed stuff
    for (qw(Config Acl)) {
        if ( !$Param{$_} ) {
            $Self->{LogObject}->Log( Priority => 'error', Message => "Need $_!" );
            return;
        }
    }

    # Wenn Owner = Agent, 
    # oder das Ticket nicht gesperrt ist, der Benutzer aber Prio-Rechte hat
    # "when back" verbieten
    if ($Self->{TicketObject}->OwnerCheck(
            TicketID => $Param{TicketID},
            OwnerID  => $Param{UserID}
        ) || ! $Self->{TicketObject}->LockIsTicketLocked( TicketID => $Param{TicketID} )
             && $Self->{TicketObject}->Permission(
                    Type     => 'priority',
                    TicketID => $Param{TicketID},
                    UserID   => $Param{UserID}
                )
    ) {
        $Param{Acl}->{RestrictStatesIfNotOwner} = {

                # match properties
                Properties => {
                    Ticket => {
                        TicketID => [ $Param{TicketID} ],
                    },
                },

                # return possible options (black list)
                PossibleNot => {
                    Ticket => {
                        State => ['when back'],
                    },
                },
        };
        return 1;
    }
    
    # Wenn der Benutzer Prio-Rechte hat, das Ticket aber für jemand anderen gesperrt ist:
    # Alles außer 'parked' erlauben
    if ($Self->{TicketObject}->Permission(
                    Type     => 'priority',
                    TicketID => $Param{TicketID},
                    UserID   => $Param{UserID}
    ))
    {
        $Param{Acl}->{RestrictStatesIfNotOwner} = {

                # match properties
                Properties => {
                    Ticket => {
                        TicketID => [ $Param{TicketID} ],
                    },
                },

                # return possible options (black list)
                PossibleNot => {
                    Ticket => {
                        State => ['parking'],
                    },
                },
        };
        return 1;
    }
    
    # Wenn der Benutzer der Ansprechpartner ist, darf er öffnen oder schließen
    my $AgentIsCustomerUser = $Param{Checks}->{User}->{UserLogin}
            eq $Param{Checks}->{Ticket}->{CustomerUserID};
    if (! $AgentIsCustomerUser) {
        my %UserCustomerData = $Self->{CustomerUserObject}->CustomerUserDataGet(
            User => $Param{Checks}->{User}->{UserLogin});
        $AgentIsCustomerUser = $UserCustomerData{UserID}
            == $Param{Checks}->{Ticket}->{CustomerUserID};
    }

    if ($AgentIsCustomerUser) {
        $Param{Acl}->{RestrictStatesIfNotOwner} = {
            # match properties
            Properties => {
                Ticket => {
                    TicketID => [ $Param{TicketID} ],
                },
            },
            # return possible options (black list)
            Possible => {
                Ticket => {
                    State => ['open', 'closed'],
                },
            },
        };
        return 1;
    }
    
    # Ansonsten darf der Benutzer gar nichts ändern
    $Param{Acl}->{RestrictStatesIfNotOwner} = {
        # match properties
        Properties => {
            Ticket => {
                TicketID => [ $Param{TicketID} ],
            },
        },
        # return possible options (black list)
        Possible => {
            Ticket => {
                State => ['-'],
            },
        },
    };
    return 1;
}

 1;
