# --
# Kernel/System/Ticket/Acl/NorisFollowupWorkflow.pm - acl module
#
# for parent tickets:
#  - forbid closing as long as there's an open or pending child (acl)
#  - forbid pending to close as long as there's an open or pending/reminder child (acl)
# for child tickets:
#  - if child is re-opened and there's a closed parent, open the parent (with notice) (event)
#  - if child is closed and there's a pending parent, open the parent (with notice) (event)
# pending for auto-close is treated as closed.

package Kernel::System::Ticket::Acl::NorisFollowupWorkflow;

use strict;
use warnings;

use Kernel::System::LinkObject;

use vars qw($VERSION);
$VERSION = qw($Revision$) [1];

use Data::Dump qw(pp);

sub new {
    my ( $Type, %Param ) = @_;

    # allocate new hash for object
    my $Self = {};
    bless( $Self, $Type );

    # get needed objects
    for (
        qw(ConfigObject DBObject EncodeObject TicketObject LogObject UserObject CustomerUserObject MainObject TimeObject)
        )
    {
        $Self->{$_} = $Param{$_} || die "Got no $_!";
    }

    return $Self;
}

sub Run {
    my ( $Self, %Param ) = @_;

    # check needed stuff
    for (qw(Config Acl)) {
        if ( !$Param{$_} ) {
            $Self->{LogObject}->Log( Priority => 'error', Message => "Need $_!" );
            return;
        }
    }
    # check if child tickets are not closed
    return 1 if !$Param{TicketID} || !$Param{UserID};

    # get ticket
    my %Ticket = $Self->{TicketObject}->TicketGet(
        TicketID => $Param{TicketID},
    );

    # create new instance of the link object
    if ( !$Self->{LinkObject} ) {
        $Self->{LinkObject} = Kernel::System::LinkObject->new(
            %{$Self},
            %Param,
        );
    }

    # link tickets
    my $Links = $Self->{LinkObject}->LinkList(
        Object => 'Ticket',
        Key    => $Param{TicketID},
        State  => 'Valid',
        Type   => 'MainSub',
        UserID => $Param{UserID},
    );

    return 1 if !$Links;
    return 1 if ref $Links ne 'HASH';
    return 1 if !$Links->{Ticket};
    return 1 if ref $Links->{Ticket} ne 'HASH';
    return 1 if !$Links->{Ticket}->{MainSub};
    return 1 if ref $Links->{Ticket}->{MainSub} ne 'HASH';
    return 1 if !$Links->{Ticket}->{MainSub}->{Target};
# die pp( $Links->{Ticket}->{MainSub}->{Target} );
   return 1 if ref $Links->{Ticket}->{MainSub}->{Target} ne 'HASH';

    my $OpenSubTickets = 0;
    my $PendingCloseSubTickets = 0;
    TICKETID:
    for my $TicketID ( sort keys %{$Links->{Ticket}->{MainSub}->{Target} } ) {

        # get ticket
        my %Ticket = $Self->{TicketObject}->TicketGet(
            TicketID => $TicketID,
        );
        
        if ( $Ticket{StateType} !~ m{ \A (close|merge|remove) }xms ) {
            if ( $Ticket{State} =~ m{ close }xms ) {
               $PendingCloseSubTickets = 1;
            }
            else {
                $OpenSubTickets = 1;
                last TICKETID;
            }
        }
    }

    # generate acl
    if ($OpenSubTickets || $PendingCloseSubTickets) {
        $Param{Acl}->{NorisFollowupWorkflow} = {

            # match properties
            Properties => {

                # current ticket match properties
                Ticket => {
                    TicketID => [ $Param{TicketID} ],
                },
            },

            # return not possible options (black list)
            PossibleNot => {
                Ticket => {
                    State => $OpenSubTickets ? 
                               [ 'closed',
                                 'closed at first call',
                                 'pending auto close'
                               ]
                             : [ 'closed',
                                 'closed at first call',
                               ]
                },
            },

            # return possible options (white list)
            Possible => {

                # possible action options
                Action => {
                    AgentTicketLock        => 1,
                    AgentTicketZoom        => 1,
                    AgentTicketClose       => 0,
                    AgentTicketPending     => 1,
                    AgentTicketNote        => 1,
                    AgentTicketHistory     => 1,
                    AgentTicketPriority    => 1,
                    AgentTicketFreeText    => 1,
                    AgentTicketHistory     => 1,
                    AgentTicketCompose     => 1,
                    AgentTicketBounce      => 1,
                    AgentTicketForward     => 1,
                    AgentLinkObject        => 1,
                    AgentTicketPrint       => 1,
                    AgentTicketPhone       => 1,
                    AgentTicketCustomer    => 1,
                    AgentTicketOwner       => 1,
                    AgentTicketResponsible => 1,
                },
            },
        };
    }

    return 1;
}

1;
