# --
# Kernel/System/CustomerUser/HTTP.pm - some customer user functions
# implements an http customer backend
# (c) 2008 noris network AG

package Kernel::System::CustomerUser::HTTP;

use strict;
# use warnings;
use Kernel::System::CheckItem;
use Kernel::System::Valid;
use Kernel::System::Cache;
use Kernel::Noris::kundebunt ();

use Crypt::PasswdMD5 qw(unix_md5_crypt);

use vars qw(@ISA $VERSION);

sub new {
    my $Type = shift;
    my %Param = @_;

    # allocate new hash for object
    my $Self = {};
    bless ($Self, $Type);
    # check needed objects
    foreach (qw(DBObject ConfigObject LogObject PreferencesObject CustomerUserMap)) {
        $Self->{$_} = $Param{$_} || die "Got no $_!";
    }
    # create valid object
    $Self->{ValidObject} = Kernel::System::Valid->new(%Param);
    # create check item object
    $Self->{CheckItemObject} = Kernel::System::CheckItem->new(%Param);
    # create cache object
    if ($Self->{CustomerUserMap}->{'CacheTTL'}) {
        $Self->{CacheObject} = Kernel::System::Cache->new(%Param);
        $Self->{CacheTTL} =  $Self->{CustomerUserMap}->{'CacheTTL'};
    }

    # max shown user a search list
    $Self->{UserSearchListLimit} = $Self->{CustomerUserMap}->{'CustomerUserSearchListLimit'} || 250;
    # config options
    $Self->{CustomerTable} = $Self->{CustomerUserMap}->{Params}->{Table}
        || die "Need CustomerUser->Params->Table in Kernel/Config.pm!";
    $Self->{CustomerKey} = $Self->{CustomerUserMap}->{CustomerKey}
        || $Self->{CustomerUserMap}->{Key}
        || die "Need CustomerUser->CustomerKey in Kernel/Config.pm!";
    $Self->{CustomerID} = $Self->{CustomerUserMap}->{CustomerID}
        || die "Need CustomerUser->CustomerID in Kernel/Config.pm!";
    $Self->{ReadOnly} = $Self->{CustomerUserMap}->{ReadOnly};
    $Self->{ExcludePrimaryCustomerID} = $Self->{CustomerUserMap}->{CustomerUserExcludePrimaryCustomerID} || 0;
    $Self->{SearchPrefix} = $Self->{CustomerUserMap}->{'CustomerUserSearchPrefix'};
    if (!defined($Self->{SearchPrefix})) {
        $Self->{SearchPrefix} = '';
    }
    $Self->{SearchSuffix} = $Self->{CustomerUserMap}->{'CustomerUserSearchSuffix'};
    if (!defined($Self->{SearchSuffix})) {
        $Self->{SearchSuffix} = '*';
    }
    $Self->{WebService} = $Self->{CustomerUserMap}->{Params}->{WebService};

    # cache key prefix
    if (!defined($Param{Count})) {
        $Self->{LogObject}->Log(
            Priority => 'error',
            Message => "Need Count param, update Kernel/System/CustomerUser.pm to v1.32 or higher!",
        );
        $Param{Count} = '';
    }
    $Self->{CacheKey} = "CustomerUser".$Param{Count};

    # create new db connect if DSN is given
    if ($Self->{CustomerUserMap}->{Params}->{DSN}) {
        $Self->{DBObject} = Kernel::System::DB->new(
            LogObject => $Param{LogObject},
            ConfigObject => $Param{ConfigObject},
            MainObject => $Param{MainObject},
            DatabaseDSN => $Self->{CustomerUserMap}->{Params}->{DSN},
            DatabaseUser => $Self->{CustomerUserMap}->{Params}->{User},
            DatabasePw => $Self->{CustomerUserMap}->{Params}->{Password},
            Type => $Self->{CustomerUserMap}->{Params}->{Type} || '',
        ) || die ('Can\'t connect to database!');
        # remember that we have the DBObject not from parent call
        $Self->{NotParentDBObject} = 1;
    }

    return $Self;
}

sub CustomerIDList {
    my ( $Self ) = @_;
    my $Rows = $Self->http_client('customer-id', search => '*');
    return map $_->{name}, @$Rows;
}

sub CustomerIDTranslate {
    my ( $Self, %Param ) = @_;
    my $retVal;
    
    # check needed stuff
    for (qw(CustomerID)) {
        if ( !$Param{$_} ) {
            $Self->{LogObject}->Log( Priority => 'error', Message => "Need $_!" );
            return;
        }
    }
    
    if ($Self->{CacheObject}) {
        $retVal = $Self->{CacheObject}->Get(
            Type => 'KuBuCustomerID',
            Key  => $Self->{CacheKey}."::$Param{CustomerID}",
        );
        return $retVal if defined $retVal;
    }
    my $Rows =
        $Self->http_client( 'customer-id', search => $Param{CustomerID} );
    return undef unless @$Rows;
    $retVal = @$Rows > 1 ? [map $_->{name}, @$Rows] : $Rows->[0]->{name};
    if ($Self->{CacheObject}) {
        $Self->{CacheObject}->Set(
            Type  => 'KuBuCustomerID',
            Key   => $Self->{CacheKey}."::$Param{CustomerID}",
            Value => $retVal,
            TTL   => $Self->{CacheTTL},
        );
    }
    return $retVal;
}

sub CustomerName {
    # /otrsdata/customer-user/?id=$ID
    my $Self = shift;
    my %Param = @_;
    my $Name = '';
    my $retVal;
    # check needed stuff
    if (!$Param{UserLogin}) {
        $Self->{LogObject}->Log(Priority => 'error', Message => "Need UserLogin!");
        return;
    }
    
    if ($Self->{CacheObject}) {
        $retVal = $Self->{CacheObject}->Get(
            Type => 'KuBuCustomerName',
            Key  => $Self->{CacheKey}."::$Param{UserLogin}",
        );
    }
    if ( !defined $retVal ) {
        my $Rows = $Self->http_client('customer-user', id => $Param{UserLogin});
        if ( @$Rows == 1 ) {
            $retVal = $$Rows[0]->{name} || $$Rows[0]->{user} || "# $$Rows[0]->{id}";
            if ($Self->{CacheObject}) {
                $Self->{CacheObject}->Set(
                    Type  => 'KuBuCustomerName',
                    Key   => $Self->{CacheKey}."::$Param{UserLogin}",
                    Value => $retVal,
                    TTL   => $Self->{CacheTTL},
                );
            }
        }

    }
    return $retVal;
}

# Suche nach customer users
# Parameter (nur ein Parameter möglich):
#   Search: Suche nach Suchausdruck (mit * und ? als Jokerzeichen)
#   PostMasterSearch: Suche nach email-Adresse (für ein neue eingegangenes Ticket)
#   UserLogin: Suche per ID des Kundenbemutzers (in kundebunt: person.id)
# Rückgabe:
#   Hash, Key ist jeweils der CustomerUserKey (person.id), 
#         Value ist ein String im Format '"$Name" <$email>'

sub CustomerSearch {
    # /otrsdata/customer-user/?search=$whatever
    # /otrsdata/customer-user/?email=$email
    # /otrsdata/customer-user/?id=$ID
    
    my $Self = shift;
    my %Param = @_;
    my %Users = ();
    my $Valid = defined $Param{Valid} ? $Param{Valid} : 1;
    my $Rows;

    # check needed stuff
    if (!$Param{Search} && !$Param{UserLogin} && !$Param{PostMasterSearch}) {
        $Self->{LogObject}->Log(Priority => 'error', Message => "Need Search, UserLogin or PostMasterSearch!");
        return;
    }
    
    if ($Param{Search}) {
        $Rows = $Self->http_client('customer-user', search => $Param{Search}, valid => $Valid);
    }
    elsif ($Param{PostMasterSearch}) {
        $Rows = $Self->http_client('customer-user', email => $Param{PostMasterSearch}, valid => $Valid);
    }
    elsif ( $Param{UserLogin} ) {
        # Caching makes sense only in UserLogin Param
        if ( $Self->{CacheObject} ) {
            $Rows = $Self->{CacheObject}->Get(
                Type => 'KuBuCustomerSearchUL',
                Key  => $Self->{CacheKey}."::$Param{UserLogin}" . "::$Valid",
            )
        }
        if ( !$Rows ) {
            $Rows = $Self->http_client( 'customer-user', id => $Param{UserLogin}, valid => $Valid );
            if ($Self->{CacheObject}) {
                $Self->{CacheObject}->Set(
                    Type  => 'KuBuCustomerSearchUL',
                    Key   => "$Self->{CacheKey}" . "::$Param{UserLogin}" . "::$Valid",
                    Value => $Rows,
                    TTL   => $Self->{CacheTTL},
                );
            }
        }
    }
    # @Rows ist jetzt eine Liste aus Hashes mit den Keys
    #         id, user, email, name, kunde
    for my $Row ( @{ $Rows || [] } ) {
        if ($Row->{id}) {
            my $u = $Row->{kunde};
            if ($Row->{name}) {
                $u = "$u: $Row->{display_name} ";
            }
            elsif ($Row->{user}) {
                $u = "$u: $Row->{user} ";
            }
            if ($Row->{email}) {
                $u .= "<$Row->{email}>";
            }
            $Users{$Row->{id}} = $u;
        }
        else {
            $Users{$Row->{kunde}} = $Row->{kunde};
        }
    }

    return %Users;
}

sub CustomerUserList {
    # /otrsdata/customer-user/?search=*
    my $Self = shift;
    my %Param = @_;
    my $Valid = defined $Param{Valid} ? $Param{Valid} : 1;
    # check cache
    if ($Self->{CacheObject}) {
        my $Users = $Self->{CacheObject}->Get(
            Type => "KuBuCustomerUserList",
            Key => $Self->{CacheKey}."::$Valid",
        );
        if ($Users) {
            return %{$Users};
        }
    }
    # get data
    my $Rows = $Self->http_client('customer-user', search => '*', valid => $Valid);
    my $Map = $Self->{CustomerUserMap};
    my %Users = ();
    for my $Row ( @$Rows ) {
        $Users{$Row->{$Map->{CustomerKey}}} = $Map->{CustomerUserListFormat}->(%$Row);
    }
    # cache request
    if ($Self->{CacheObject}) {
        $Self->{CacheObject}->Set(
            Type => "KuBuCustomerUserList",
            Key => $Self->{CacheKey}."::$Valid",
            Value => \%Users,
            TTL => $Self->{'CacheTTL'},
        );
    }
    return %Users;
}

sub CustomerIDs {
    my $Self = shift;
    my %Param = @_;
    # check needed stuff
    if (!$Param{User}) {
        $Self->{LogObject}->Log(Priority => 'error', Message => "Need User!");
        return;
    }
    # get customer data
    my %Data = $Self->CustomerUserDataGet(
        User => $Param{User},
    );
    # there are multi customer ids
    my @CustomerIDs = split(';', $Data{UserCustomerIDs});

    # use also the primary customer id
    if ($Data{UserCustomerID} && !$Self->{ExcludePrimaryCustomerID}) {
        push (@CustomerIDs, $Data{UserCustomerID});
    }
    return @CustomerIDs;
}

sub CustomerUserDataGet {
    # /otrsdata/customer-user/?kunde=$KUNDEN_NAME
    my $Self = shift;
    my %Param = @_;
    my %Data;
    # check needed stuff
    if (!$Param{User} && !$Param{CustomerID}) {
        $Self->{LogObject}->Log(Priority => 'error', Message => "Need User or CustomerID!");
        return;
    }
    my $Rows;
    if ($Param{User}) {
        # check cache
        if ($Self->{CacheObject}) {
            my $Data = $Self->{CacheObject}->Get(
                Type => 'KuBuCustomerUserDataGet',
                Key  => $Self->{CacheKey}."::User::".$Param{User},
            );
            if ($Data) {
                return %{$Data};
            }
        }
        $Rows = $Self->http_client('customer-user', id => $Param{User});
    }
    elsif ($Param{CustomerID}) {
        # check cache
        if ($Self->{CacheObject}) {
            my $Data = $Self->{CacheObject}->Get(
                Type => 'KuBuCustomerUserDataGet',
                Key  => $Self->{CacheKey}."::CustomerID::".$Param{CustomerID},
            );
            if ($Data) {
                return %{$Data};
            }
        }
        $Rows = $Self->http_client('customer-user', kunde => $Param{CustomerID});
    }
    if ( ! defined $Rows || ! @$Rows) {
        $Self->{LogObject}->Log(Priority => 'notice', Message => "CustomerUserDataGet: Nothing found. $Param{User}");
        return;
    }
    
    my $Row = $$Rows[0];
    foreach my $Entry (@{$Self->{CustomerUserMap}->{Map}}) {
        $Data{$Entry->[0]} = $Row->{$Entry->[2]};
    }
    foreach my $Entry (@{$Self->{CustomerUserMap}->{TicketMap}}) {
        $Data{$Entry->[0]} = $Row->{$Entry->[2]};
    }
    # check data
    if (! exists $Data{UserLogin} && $Param{User}) {
        # cache request
        if ($Self->{CacheObject}) {
            $Self->{CacheObject}->Set(
                Type => 'KuBuCustomerUserDataGet',
                Key => $Self->{CacheKey}."::User::".$Param{User},
                Value => { },
                TTL => $Self->{CacheTTL},
            );
        }
        return;
    }
    if (! exists $Data{UserLogin} && $Param{CustomerID}) {
        # cache request
        if ($Self->{CacheObject}) {
            $Self->{CacheObject}->Set(
                Type => 'KuBuCustomerUserDataGet',
                Key => $Self->{CacheKey}."::CustomerID::".$Param{CustomerID},
                Value => { },
                TTL => $Self->{CacheTTL},
            );
        }
        return;
    }
    # compat!
    $Data{UserID} = $Data{UserLogin};
    # get preferences
    my %Preferences = $Self->{PreferencesObject}->GetPreferences(UserID => $Data{UserID});
    # cache request
    if ($Self->{CacheObject}) {
        if ($Param{User}) {
            $Self->{CacheObject}->Set(
                Type => 'KuBuCustomerUserDataGet',
                Key => $Self->{CacheKey}."::User::".$Param{User},
                Value => { %Data, %Preferences },
                TTL => $Self->{CacheTTL},
            );
        }
        elsif ($Param{CustomerID}) {
            $Self->{CacheObject}->Set(
                Type => 'KuBuCustomerUserDataGet',
                Key => $Self->{CacheKey}."::CustomerID::".$Param{CustomerID},
                Value => { %Data, %Preferences },
                TTL => $Self->{CacheTTL},
            );
        }
    }
    # return data
    return (%Data, %Preferences);
}

sub CustomerUserAdd {
    my $Self = shift;
    die "Not implemented: CustomerUserAdd";
#     my %Param = @_;
#     # check ro/rw
#     if ($Self->{ReadOnly}) {
#         $Self->{LogObject}->Log(Priority => 'error', Message => "Customer backend is ro!");
#         return;
#     }
#     # check needed stuff
#     foreach my $Entry (@{$Self->{CustomerUserMap}->{Map}}) {
#         if (!$Param{$Entry->[0]} && $Entry->[4]) {
#             $Self->{LogObject}->Log(Priority => 'error', Message => "Need $Entry->[0]!");
#             return;
#         }
#     }
#     if (!$Param{UserID}) {
#         $Self->{LogObject}->Log(Priority => 'error', Message => "Need UserID!");
#         return;
#     }
#     # just if no UserLogin is given
#     if (!$Param{UserLogin} && $Self->{CustomerUserMap}->{AutoLoginCreation}) {
#         my ($Sec, $Min, $Hour, $Day, $Month, $Year, $WDay) = localtime(time());
#         $Year = $Year-100;
#         $Year = "0$Year" if ($Year <10);
#         $Month = $Month+1;
#         $Month = "0$Month" if ($Month <10);
#         $Day = "0$Day" if ($Day <10);
#         $Hour = "0$Hour" if ($Hour <10);
#         $Min = "0$Min" if ($Min <10);
#         my $Prefix = $Self->{CustomerUserMap}->{AutoLoginCreationPrefix} || 'auto';
#         $Param{UserLogin} = "$Prefix-$Year$Month$Day$Hour$Min".int(rand(99));
#     }
#     # check if user login exists
#     if (!$Param{UserLogin}) {
#         $Self->{LogObject}->Log(Priority => 'error', Message => "Need UserLogin!");
#         return;
#     }
#     # check email address if already exists
#     if ($Param{UserEmail} && $Self->{CustomerUserMap}->{CustomerUserEmailUniqCheck}) {
#         my %Result = $Self->CustomerSearch(
#             Valid => 1,
#             PostMasterSearch => $Param{UserEmail},
#         );
#         if (%Result) {
#             $Self->{LogObject}->Log(Priority => 'error', Message => "Email already exists!");
#             return;
#         }
#     }
#     # check email address mx
#     if ($Param{UserEmail} && !$Self->{CheckItemObject}->CheckEmail(Address => $Param{UserEmail})) {
#         $Self->{LogObject}->Log(
#             Priority => 'error',
#             Message => "Email address ($Param{UserEmail}) not valid (".
#                 $Self->{CheckItemObject}->CheckError().")!",
#         );
#         return;
#     }
# 
#     # quote values
#     my %Value;
#     for my $Entry (@{$Self->{CustomerUserMap}->{Map}}) {
#         if ($Entry->[5] =~ /^int$/i) {
#             if ($Param{ $Entry->[0] }) {
#                 $Value{ $Entry->[0] } = $Self->{DBObject}->Quote($Param{ $Entry->[0] }, 'Integer');
#             }
#             else {
#                 $Value{ $Entry->[0] } = 0;
#             }
#         }
#         else {
#             if ($Param{ $Entry->[0] }) {
#                 $Value{ $Entry->[0] } = "'" . $Self->{DBObject}->Quote($Param{$Entry->[0]}) . "'";
#             }
#             else {
#                 $Value{ $Entry->[0] }= "''";
#             }
#         }
#     }
# 
#     # build insert
#     my $SQL = "INSERT INTO $Self->{CustomerTable} (";
#     foreach my $Entry (@{$Self->{CustomerUserMap}->{Map}}) {
#         if ($Entry->[0] !~ /^UserPassword$/i) {
#             $SQL .= " $Entry->[2], ";
#         }
#     }
#     $SQL .= "create_time, create_by, change_time, change_by)";
#     $SQL .= " VALUES (";
#     foreach my $Entry (@{$Self->{CustomerUserMap}->{Map}}) {
#         if ($Entry->[0] !~ /^UserPassword$/i) {
#             $SQL .= " $Value{ $Entry->[0] }, ";
#         }
#     }
#     $SQL .= "current_timestamp, $Param{UserID}, current_timestamp, $Param{UserID})";
#     if ($Self->{DBObject}->Do(SQL => $SQL)) {
#         # log notice
#         $Self->{LogObject}->Log(
#             Priority => 'notice',
#             Message => "CustomerUser: '$Param{UserLogin}' created successfully ($Param{UserID})!",
#         );
#         # set password
#         if ($Param{UserPassword}) {
#             $Self->SetPassword(UserLogin => $Param{UserLogin}, PW => $Param{UserPassword});
#         }
#         # cache resete
#         if ($Self->{CacheObject}) {
#             $Self->{CacheObject}->Delete(
#                 Type => 'KuBuCustomerUserDataGet',
#                 Key => $Self->{CacheKey}."::User::$Param{UserLogin}",
#             );
#         }
#         return $Param{UserLogin};
#     }
#     else {
#         return;
#     }
}

sub CustomerUserUpdate {
    die "Not Implemented: CustomerUserUpdate";
#     my $Self  = shift;
#     my %Param = @_;
# 
#     # check ro/rw
#     if ($Self->{ReadOnly}) {
#         $Self->{LogObject}->Log(Priority => 'error', Message => "Customer backend is ro!");
#         return;
#     }
# 
#     # check needed stuff
#     foreach my $Entry (@{$Self->{CustomerUserMap}->{Map}}) {
#         if (!$Param{$Entry->[0]} && $Entry->[4] && $Entry->[0] ne 'UserPassword') {
#             $Self->{LogObject}->Log(Priority => 'error', Message => "Need $Entry->[0]!");
#             return;
#         }
#     }
# 
#     # check email address
#     if ($Param{UserEmail} && !$Self->{CheckItemObject}->CheckEmail(Address => $Param{UserEmail})) {
#         $Self->{LogObject}->Log(
#             Priority => 'error',
#             Message => "Email address ($Param{UserEmail}) not valid (".
#                 $Self->{CheckItemObject}->CheckError().")!",
#         );
#         return;
#     }
#     # get old user data (pw)
#     my %UserData = $Self->CustomerUserDataGet(User => $Param{ID});
# 
#     # quote values
#     my %Value;
#     for my $Entry (@{$Self->{CustomerUserMap}->{Map}}) {
#         if ($Entry->[5] =~ /^int$/i) {
#             if ($Param{ $Entry->[0] }) {
#                 $Value{ $Entry->[0] } = $Self->{DBObject}->Quote($Param{ $Entry->[0] }, 'Integer');
#             }
#             else {
#                 $Value{ $Entry->[0] } = 0;
#             }
#         }
#         else {
#             if ($Param{ $Entry->[0] }) {
#                 $Value{ $Entry->[0] } = "'" . $Self->{DBObject}->Quote($Param{$Entry->[0]}) . "'";
#             }
#             else {
#                 $Value{ $Entry->[0] }= "''";
#             }
#         }
#     }
# , valid => $Valid
#     # update db
#     my $SQL = "UPDATE $Self->{CustomerTable} SET ";
#     foreach my $Entry (@{$Self->{CustomerUserMap}->{Map}}) {
#         if ($Entry->[0] !~ /^UserPassword$/i) {
#             $SQL .= " $Entry->[2] = $Value{ $Entry->[0] }, ";
#         }
#     }
#     $SQL .= " change_time = current_timestamp, ";
#     $SQL .= " change_by = $Param{UserID} ";
#     $SQL .= " WHERE LOWER($Self->{CustomerKey}) = LOWER('".$Self->{DBObject}->Quote($Param{ID})."')";
# 
#     if ($Self->{DBObject}->Do(SQL => $SQL)) {
#         # log notice
#         $Self->{LogObject}->Log(
#             Priority => 'notice',
#             Message => "CustomerUser: '$Param{UserLogin}' updated successfully ($Param{UserID})!",
#         );
#         # check pw
#         if ($Param{UserPassword}) {
#             $Self->SetPassword(UserLogin => $Param{UserLogin}, PW => $Param{UserPassword});
#         }
#         # cache resete
#         if ($Self->{CacheObject}) {
#             $Self->{CacheObject}->Delete(
#                 Type => 'KuBuCustomerUserDataGet',
#                 Key  => $Self->{CacheKey}."::User::".$Param{UserLogin},
#             );
#         }
#         return 1;
#     }
#     else {
#         return;
#     }
}

sub SetPassword {
    die "Not implemented: SetPassword";
#     my $Self = shift;
#     my %Param = @_;
#     my $Pw = $Param{PW} || '';
#     # check ro/rw
#     if ($Self->{ReadOnly}) {
#         $Self->{LogObject}->Log(Priority => 'error', Message => "Customer backend is ro!");
#         return;
#     }
#     # check needed stuff
#     if (!$Param{UserLogin}) {
#         $Self->{LogObject}->Log(Priority => 'error', Message => "Need UserLogin!");
#         return;
#     }
#     my $CryptedPw = '';
#     # md5 pw
#     if ($Self->{ConfigObject}->Get('Customer::AuthModule::DB::CryptType') &&
#         $Self->{ConfigObject}->Get('Customer::AuthModule::DB::CryptType') eq 'plain') {
#         $CryptedPw = $Pw;
#     }
#     elsif ($Self->{ConfigObject}->Get('Customer::AuthModule::DB::CryptType') &&
#         $Self->{ConfigObject}->Get('Customer::AuthModule::DB::CryptType') eq 'md5') {
#         $CryptedPw = unix_md5_crypt($Pw, $Param{UserLogin});
#     }
#     # crypt pw
#     else {
#         # crypt given pw (unfortunately there is a mod_perl2 bug on RH8 - check if
#         # crypt() is working correctly) :-/
#         if (crypt('root', 'root@localhost') eq 'roK20XGbWEsSM') {
#             $CryptedPw = crypt($Pw, $Param{UserLogin});
#         }
#         else {
#             $Self->{LogObject}->Log(
#                 Priority => 'notice',
#                 Message => "The crypt() of your mod_perl(2) is not working correctly! Update mod_perl!",
#             );
#             my $TempUser = quotemeta($Param{UserLogin});
#             my $TempPw = quotemeta($Pw);
#             my $CMD = "perl -e \"print crypt('$TempPw', '$TempUser');\"";
#             open (IO, " $CMD | ") || print STDERR "Can't open $CMD: $!";
#             while (<IO>) {
#                 $CryptedPw .= $_;
#             }
#             close (IO);
#             chomp $CryptedPw;
#         }
#     }
#     # update db
#     foreach my $Entry (@{$Self->{CustomerUserMap}->{Map}}) {
#         if ($Entry->[0] =~ /^UserPassword$/i) {
#             $Param{PasswordCol} = $Entry->[2];
#         }
#         if ($Entry->[0] =~ /^UserLogin$/i) {
#             $Param{LoginCol} = $Entry->[2];
#         }
#     }
#     # check if needed pw col. exists (else there is no pw col.)
#     if ($Param{PasswordCol} && $Param{LoginCol}) {
#         if ($Self->{DBObject}->Do(
#             SQL => "UPDATE $Self->{CustomerTable} ".
#                 " SET ".
#                 " $Param{PasswordCol} = '".$Self->{DBObject}->Quote($CryptedPw)."' ".
#                 " WHERE ".
#                 " LOWER($Param{LoginCol}) = LOWER('".$Self->{DBObject}->Quote($Param{UserLogin})."')",
#         )) {
#             # log notice
#             $Self->{LogObject}->Log(
#                 Priority => 'notice',
#                 Message => "CustomerUser: '$Param{UserLogin}' changed password successfully!",
#             );
# 
#             # cache reset
#             if ($Self->{CacheObject}) {
#                 $Self->{CacheObject}->Delete(
#                     Type => 'KuBuCustomerUserDataGet',
#                     Key  => $Self->{CacheKey}."::User::".$Param{UserLogin},
#                 );
#             }
#             return 1;
#         }
#         else {
#             return;
#         }
#     }
#     else {
#         # need no pw to set
#         return 1;
#     }
}

# sub GenerateRandomPassword {
#     my $Self = shift;
#     my %Param = @_;
#     # Generated passwords are eight characters long by default.
#     my $Size = $Param{Size} || 8;
# 
#     # The list of characters that can appear in a randomly generated password.
#     # Note that users can put any character into a password they choose themselves.
#     my @PwChars = (0..9, 'A'..'Z', 'a'..'z', '-', '_', '!', '@', '#', '$', '%', '^', '&', '*');
# 
#     # The number of characters in the list.
#     my $PwCharsLen = scalar(@PwChars);
# 
#     # Generate the password.
#     my $Password = '';
#     for ( my $i=0 ; $i<$Size ; $i++ ) {
#         $Password .= $PwChars[rand($PwCharsLen)];
#     }
# 
#     # Return the password.
#     return $Password;
# }

sub DESTROY {
    my $Self = shift;
    # disconnect if it's not a parent DBObject
    if ($Self->{NotParentDBObject}) {
        if ($Self->{DBObject}) {
            $Self->{DBObject}->Disconnect();
        }
    }
}

# Abfrage kundebunt-WebService, vgl. RT#400809
# Beispielaufruf: my $data = $self->ask_kundebunt( kunde => 'sluka' );
# my $first_email = $self->http_client('customer-user', kunde=>'sluka')->{data}[0]{email};
sub http_client {
    my $Self = shift;
    my $ViewName = shift;
    return Kernel::Noris::kundebunt::kundebunt_client(
        ConfigObject => $Self->{ConfigObject},
        Method => $ViewName,
        Params => \@_
    );
}

1;
