use strict;
use warnings;
use utf8;

package Kernel::Noris::TicketServer::Request::FollowTicket;

use Moose;
use Kernel::Noris::TicketServer::FollowResultBuffer;
use Kernel::Noris::TicketServer::Conversions;
use Log::Log4perl qw(get_logger);
use Data::Dump qw(pp);
use List::Util qw(first);

extends 'Kernel::Noris::TicketServer::Request';
has 'result_buffer' => (is => 'ro', isa => 'Object');
has 'src_buffer' => (is => 'ro', isa => 'Object');
has 'follow_by' => (is => 'rw', isa => 'ArrayRef');

sub is_data_required {
    return 1;
}

has 'src_buffer' => (is => 'ro', isa => 'Object');
sub respond {
    my ($self) = @_;
    my $user_id = $self->connection->eff_user_id;
    die "follow_tickets request requires a valid user\n" unless defined $user_id;
    my $buffer_nr = shift @{$self->param};
    my $src_buffer = Kernel::Noris::TicketServer::ResultBuffer::get_buffer($buffer_nr);
    die "No such buffer: $buffer_nr\n" unless defined $src_buffer;
    $self->{src_buffer} = $src_buffer;
    $self->{follow_by} = $self->req_data->{follow_by};
    $self->_check_follow_by();
    my @fields = @Kernel::Noris::TicketServer::Conversions::Fields{@{$self->param}};
    for my $i (0..$#fields) {
        die "No such field: " . $self->param->[$i] . "\n" unless defined $fields[$i];
    }
    $self->{result_buffer} = Kernel::Noris::TicketServer::FollowResultBuffer->new(
        fields => \@fields);
    return $self->{result_buffer}->buffer_id() . "\n";
}


sub background_process {
    my $logger = get_logger(__PACKAGE__);
    my ($self) = @_;
    my $connection = $self->connection;
    die "follow_tickets request requires a valid user\n" unless defined $connection->eff_user_id;

    my @queue = @{$self->{src_buffer}->{results}};
    my $pos = 0;
    my %seen_ids = ();
    for my $id (@queue) {
        $seen_ids{$id} = \[$id];
    }
    while ($pos < @queue) {
        for my $candidate ($self->_find_next_nodes($queue[$pos])) {
            if (exists $seen_ids{$candidate}) {
                if (${$seen_ids{$candidate}} != ${$seen_ids{$queue[$pos]}}) {
                    $logger->debug(
                        "follow_tickets: at $candidate, merging "
                        . pp($seen_ids{$candidate},$seen_ids{$queue[$pos]}))
                        if $logger->is_debug;
                    my $union = [ @${$seen_ids{$queue[$pos]}}, @${$seen_ids{$candidate}} ];
                    ${$seen_ids{$_}} = $union for @$union;
                    $logger->debug("result: " . pp($seen_ids{$queue[$pos]}))
                        if $logger->is_debug;
                }
                # else: sind identisch, alles schon erledigt
            }
            else {
                $logger->debug(
                        "follow_tickets: pushing $candidate, origin: "
                        . pp($seen_ids{$queue[$pos]}))
                        if $logger->is_debug;
                push @queue, $candidate;
                $seen_ids{$candidate} = $seen_ids{$queue[$pos]};
            }
        }
        ++ $pos;
    }
    $logger->debug("follow_tickets: got " . scalar @queue . " items") if $logger->is_debug;
    my $buffer = $self->{result_buffer};
    $buffer->set_results(\@queue, \%seen_ids);
}

{
    my %type_map = (
                    main => 'MainSub',
                    'sub'  => 'MainSub',
                    split_source => 'Split',
                    split_target => 'Split',
                    merge_parent => 'ParentChild',
                    merge_child =>  'ParentChild',
                    normal => 'Normal',
                );
    my %dir_map = (
                    main => ['Source'],
                    'sub' => ['Target'],
                    split_source => ['Source'],
                    split_target => ['Target'],
                    merge_parent => ['Source'],
                    merge_child =>  ['Target'],
                    normal => [qw(Source Target)],
                );

    sub _check_follow_by {
        my ($self) = @_;
        my @wrong = grep( ! exists $type_map{$_},
                          @{$self->follow_by});
        die "wrong follow_by specification: @wrong\n" if @wrong;
    }

    sub _find_next_nodes {
        my ($self, $ticket_id) = @_;
        my @results = ();
        my $LinkList = $self->connection->otrs->{link}->LinkList(
                Object => 'Ticket',
                Key => $ticket_id,
                Object2 => 'Ticket',
                State => 'Valid',
                UserID => $self->connection->eff_user_id
        );
        for my $follow_by (@{$self->follow_by}) {
            for my $dir (@{$dir_map{$follow_by}}) {
                if (exists $LinkList->{Ticket}->{$type_map{$follow_by}}->{$dir}) {
                    push @results, keys %{ $LinkList->{Ticket}->{$type_map{$follow_by}}->{$dir}  };
                }
            }
        }
        return @results;
    }
}

1;
