use utf8;
use strict;
use warnings;


package Kernel::Noris::TicketServer::Conversions::MergeParentField;

use Moose;
use Log::Log4perl qw(get_logger);
use Data::Dump qw(pp);
use List::Util qw(min);

has 'api' => (is => 'ro', isa => 'Str', required => 1);
has 'recursive_search' => (is => 'ro', isa => 'Bool', required => 1);  # for merge_root

sub build_otrs_filter {
    my ($self, $otrs, $data) = @_;
    my ($op, @values);
    my @results = ();
    my $logger = get_logger(__PACKAGE__);
    my @queue = ();
    if (ref $data ne 'ARRAY') {
        $op = 'in';
        @values = ($data);
    }
    else {
        ($op, @values) = @$data;
    }
    die "cannot handle operator \"$op\" for field $self->{api}\n"
        unless $op =~ /^in$/;
    foreach my $tn (@values) {
        my $id = $otrs->{ticket}->TicketIDLookup(TicketNumber => $tn);
        push @queue, $id if defined $id;
    }
    my @result_tns = @values;
    my %seen = ();
    while (@queue) {
        my $id = shift @queue;
        if (! exists $seen{$id}) {
            $seen{$id} = 1;
            my $LinkList = $otrs->{link}->LinkListWithData(
                Object    => 'Ticket',
                Key       => $id,
                Object2   => 'Ticket',
                State     => 'Valid',
                Direction => 'Target',
                UserID    => 1,
            );
            $LinkList = $LinkList->{Ticket}->{ParentChild}->{Target};
            if (defined $LinkList && %$LinkList) {
                $logger->debug("build_otrs_filter got link list from otrs for ticket id $id: "
                            . pp(values %$LinkList))
                    if $logger->is_debug();
                push @results, values %$LinkList;
                unshift @queue, keys %$LinkList if $self->recursive_search;
            }
        }
    }
    foreach my $ticketData (@results) {
        if ($ticketData->{StateType} eq 'merged') {
            push @result_tns, $ticketData->{TicketNumber};
        }
        else {
            $logger->info('build_otrs_filter: ticket id $id, #$ticketData->{TicketNumber} is not in StateType merged, though it is linked as a merged ticket. Skipping this ticket.');
        }
    }
    return (TicketNumber => \@result_tns);
}
        
sub get_from_otrs_data
{
    my ($self, $otrs, $data) = @_;
    my $logger = get_logger(__PACKAGE__);
    while ($data->{StateType} eq 'merged') {
        my $LinkList = $otrs->{link}->LinkListWithData(
            Object    => 'Ticket',
            Key       => $data->{TicketID},
            Object2   => 'Ticket',
            State     => 'Valid',
            Direction => 'Source',
            UserID    => 1,
        );
        $LinkList = $LinkList->{Ticket}->{ParentChild}->{Source};
        if (defined $LinkList && %$LinkList) {
            $logger->debug(
                    "Got link list from otrs for ticket id $data->{TicketID}, #$data->{TicketNumber}: "
                    . pp(map [$_->{TicketID},$_->{TicketNumber}], values %$LinkList))
                if $logger->is_debug();
            $logger->info(
                    "ticket id $data->{TicketID}, #$data->{TicketNumber} has multiple merge parents. Ignoring all but the first one."
                ) if (values %$LinkList > 1);
            $data = (values %$LinkList)[0];
            if ($self->recursive_search) {
                $logger->debug("Recursive MergeParent proceeds to ticket id $data->{TicketID}, #$data->{TicketNumber}");
            }
            else {
                return $data->{TicketNumber};
            }
        }
        else {
            $logger->info(
                "Trotz Status 'merge' ist kein Ticket mit Ticket id $data->{TicketID}, #$data->{TicketNumber} verlinkt. Das ist normal, wenn das uebergeordnete Ticket in einer noch nicht importierten Queue liegt. MergeParentField gibt jetzt undef zurueck."
            );
            return undef;
        }
    }
    return $data->{TicketNumber};
}


1;
