use utf8;
use strict;
use warnings;


package Kernel::Noris::TicketServer::Conversions::LinkedObjectsField;

use Moose;
use Log::Log4perl qw(get_logger);
use Data::Dump qw(pp);
use List::Util qw(min);

has 'api' => (is => 'ro', isa => 'Str', required => 1);
has 'object_type' => (is => 'ro', isa => 'Str', required => 1);
has 'link_type' => (is => 'ro', isa => 'Str', required => 1);
has 'link_direction' => (is => 'ro', isa => 'Str', required => 1);

# my %reverse_direction = (
#     Source => 'Target',
#     Target => 'Source',
#     Both => 'Both',
# );
        
sub build_otrs_filter {
    my ($self, $otrs, $data) = @_;
    my @results = ();
    my $logger = get_logger(__PACKAGE__);
    my @queue = ();
    if (ref $data ne 'ARRAY') {
        $data = [ $data ]
    }
    my %HasAnyTickets = ();
    my %HasNoneTickets = ();
    for my $term (@$data) {
        die "$self->{api} expects a list of list references but got something different\n"
            unless ref $term eq 'ARRAY';
        my ($op, @objectnames) = @$term;
        my $ticket_numbers = $self->_get_linked_tickets_as_hash(\@objectnames, $otrs);
        if ($op eq 'has_any') {
            if (%HasAnyTickets) {
                _intersect(\%HasAnyTickets, $ticket_numbers);
            }
            else {
                %HasAnyTickets = %$ticket_numbers;
            }
            return (TicketNumber => []) unless %HasAnyTickets;   # keine Treffer mehr moeglich
        }
        elsif ($op eq 'has_none') {
            if (%HasNoneTickets) {
                _union(\%HasNoneTickets, $ticket_numbers);
            }
            else {
                %HasNoneTickets = %$ticket_numbers;
            }
        }
        else {
            die "$self->{api} expects a list of list references, each starting with 'has_any' or 'has_none' but got a '$op'.\n";
        }
    }
    if ($logger->is_debug) {
        $logger->debug("has_any: " . pp(\%HasAnyTickets));
        $logger->debug("has_none: " . pp(\%HasNoneTickets));
    }
    if (%HasAnyTickets) {
        _difference(\%HasAnyTickets, \%HasNoneTickets);
        return (TicketNumber => [keys %HasAnyTickets]);
    }
    else {
        return (TicketNumberNot => [keys %HasNoneTickets]);
    }
}

# $left := $left INTERSECTION $right; $left and $right are hash references, only keys are important
sub _intersect {
    my ($left, $right) = @_;
    for my $key (keys %$left) {
        delete $left->{$key} unless exists $right->{$key};
    }
}

# $left := $left UNION $right; $left and $right are hash references, only keys are important
sub _union {
    my ($left, $right) = @_;
    for my $key (keys %$right) {
        $left->{$right} = undef;
    }
}

# $left := $left \ $right; $left and $right are hash references, only keys are important
sub _difference {
    my ($left, $right) = @_;
    for my $key (keys %$right) {
        delete $left->{$key} if exists $left->{$key};
    }
}

# return a hash ref, keys are ticket numbers of the tickets linked to at least one
# of the objects given in @$objectnames
sub _get_linked_tickets_as_hash {
    my ($self, $objectnames, $otrs) = @_;
    my %result = ();
    for my $name (@$objectnames) {
        for my $id ($self->_get_object_ids($name, $otrs)) {
            my $LinkList = $otrs->{link}->LinkListWithData(
                    Object    => $self->object_type,
                    Key       => $id,
                    Object2   => 'Ticket',
                    LinkType  => $self->link_type,
                    State     => 'Valid',
                    Direction => $self->link_direction,
                    UserID    => 1,
                );
            for my $data (
                    values %{$LinkList->{Ticket}
                            ->{$self->link_type}
                            ->{$self->link_direction}}
                ) {
                $result{$data->{TicketNumber}} = undef;
            }
        }
    }
    return \%result
}

sub _get_object_ids {
    my ($self, $name, $otrs) = @_;
    my $objects = $otrs->{link}->ObjectSearch(
        Object => $self->object_type,
        SearchParams => { name => $name },
        UserID => 1,
    );
    if (defined $objects->{$self->object_type}->{NOTLINKED}->{Source}) {
        my @result = keys %{$objects->{$self->object_type}->{NOTLINKED}->{Source}};   # Die API von OTRS *ist* halt so.
        return @result;
    }
    else {
        return ();
    }
}


sub get_from_otrs_data
{
    my ($self, $otrs, $data) = @_;
    my $logger = get_logger(__PACKAGE__);
    my $LinkList = $otrs->{link}->LinkListWithData(
        Object    => 'Ticket',
        Key       => $data->{TicketID},
        Object2   => $self->object_type,
        LinkType  => $self->link_type,
        State     => 'Valid',
        Direction => $self->link_direction,
        UserID    => 1,
    );
    $LinkList = $LinkList->{$self->object_type}->{$self->link_type}->{$self->link_direction};
    if (defined $LinkList && %$LinkList) {
        $data = [map $_->{name}, values %$LinkList];
        return $data;
    }
    else {
        return [];
    }
}
