use strict;
use warnings;
use utf8;

package Kernel::Noris::TicketServer::Connection;

use Encode;
use IO::Socket::INET qw($CRLF);
use JSON qw(decode_json);
use Kernel::Config;
use Kernel::Noris::TicketServer::Request::Quit;
use Kernel::System::CustomerUser;
use Kernel::System::Encode;
use Kernel::System::Time;
use Kernel::System::Log;
use Kernel::System::Main;
use Kernel::System::DB;
use Kernel::System::LinkObject;
use Kernel::System::Ticket;
use Kernel::System::Ticket::Article;
use Kernel::System::Queue;
use Log::Log4perl qw(get_logger);
use Moose;

has 'real_user_name' => (is => 'ro', isa => 'Str', required => 1);
has 'otrs' => (is => 'ro', isa => 'HashRef', default => sub { return _otrs_connect(); }  );
has 'real_user_id' => (is => 'rw', isa => 'Int');
has 'eff_user_id' => (is => 'rw', isa => 'Int');
has 'remote_host' =>  (is => 'ro', isa => 'Str', required => 1);

sub BUILD {
    my ($self) = @_;
    $self->real_user_id($self->get_user_id($self->real_user_name));
    if ($self->is_allowed_to_change_user($self->real_user_name)) {
        $self->eff_user_id($self->real_user_id);
    }
    my $logger = get_logger(__PACKAGE__);
    $logger->debug("initial eff_user_id: " . ($self->eff_user_id || 'undef'));
}

# find user id (int) for a user name (str)
# returns None if there is no otrs user with this name
sub get_user_id {
    my ($self, $user_name) = @_;
    my $logger = get_logger(__PACKAGE__);
    my $user_id = $self->otrs->{user}->UserLookup( UserLogin => $user_name);
    $logger->debug("found no user id for user name \"$user_name\", failed\n") unless defined $user_id;
    return $user_id;
}

# initialisiert otrs-Variablen, Anbindung an Datenbank usw.
# not a method
sub _otrs_connect {
    my $ConfigObject = Kernel::Config->new();
    my $EncodeObject = Kernel::System::Encode->new(
        ConfigObject => $ConfigObject,
    );
    my $LogObject    = Kernel::System::Log->new(
        ConfigObject => $ConfigObject,
        LogPrefix    => 'import',
    );
    my $TimeObject = Kernel::System::Time->new(
        LogObject    => $LogObject,
        ConfigObject => $ConfigObject,
    );
    my $MainObject = Kernel::System::Main->new(
        LogObject    => $LogObject,
        ConfigObject => $ConfigObject,
    );
    my $DBObject = Kernel::System::DB->new(
        ConfigObject => $ConfigObject,
        EncodeObject => $EncodeObject,
        MainObject   => $MainObject,
        LogObject    => $LogObject,
    );
    my $LinkObject = Kernel::System::LinkObject->new(
        ConfigObject => $ConfigObject,
        EncodeObject => $EncodeObject,
        LogObject    => $LogObject,
        DBObject     => $DBObject,
        TimeObject   => $TimeObject,
        MainObject   => $MainObject,
    );
    my $TicketObject = Kernel::System::Ticket->new(
        ConfigObject => $ConfigObject,
        EncodeObject => $EncodeObject,
        LogObject    => $LogObject,
        DBObject     => $DBObject,
        MainObject   => $MainObject,
        TimeObject   => $TimeObject,
    );
    my $UserObject = Kernel::System::User->new(
        ConfigObject => $ConfigObject,
        EncodeObject => $EncodeObject,
        LogObject    => $LogObject,
        MainObject   => $MainObject,
        TimeObject   => $TimeObject,
        DBObject     => $DBObject,
    );
    my $QueueObject = Kernel::System::Queue->new(
        ConfigObject => $ConfigObject,
        LogObject    => $LogObject,
        DBObject     => $DBObject,
        MainObject   => $MainObject,
    );
    my $StateObject = Kernel::System::State->new(
        ConfigObject => $ConfigObject,
        LogObject    => $LogObject,
        DBObject     => $DBObject,
        TimeObject   => $TimeObject,
    );
    my $TypeObject = Kernel::System::Type->new(
        ConfigObject => $ConfigObject,
        LogObject    => $LogObject,
        DBObject     => $DBObject,
        TimeObject   => $TimeObject,
    );
    my $CustomerUserObject = Kernel::System::CustomerUser->new(
        ConfigObject => $ConfigObject,
        EncodeObject => $EncodeObject,
        LogObject    => $LogObject,
        DBObject     => $DBObject,
        MainObject   => $MainObject,
    );
    my $LockObject = Kernel::System::Lock->new(
        ConfigObject => $ConfigObject,
        LogObject    => $LogObject,
        DBObject     => $DBObject,
        TimeObject   => $TimeObject,
    );

    return {
        config => $ConfigObject,
        encode => $EncodeObject,
        log => $LogObject,
        time => $TimeObject,
        main => $MainObject,
        db => $DBObject,
        link => $LinkObject,
        ticket => $TicketObject,
        user => $UserObject,
        queue => $QueueObject,
        state => $StateObject,
        type => $TypeObject,
        customer => $CustomerUserObject,
        lock => $LockObject,
    };
}

# liest von Socket bis eine Anfrage komplett ist und dekodiert die Anfrage.
sub handle_request {
    my ($self, $cl) = @_;
    my $logger = get_logger(__PACKAGE__);
    my $iologger = get_logger(__PACKAGE__ . ".io");
    my $line_raw = $cl->getline();
    if (! defined $line_raw) {
        $iologger->debug("--> (EOF)");
        return '';
    }
    my $line1 = decode_utf8($line_raw);
    $line1 =~ s/\s+\z//;
    $iologger->debug("--> $line1");
    my @words = split ' ', $line1;
    my $req = Kernel::Noris::TicketServer::Request::create($self, @words);
    die "Unbekannter Befehl: @words\n" unless defined $req;

    if ($req->is_data_required()) {
        my @lines;
        my $line;
        while ($line = $cl->getline()) {
            $line =~ s/\s+\z//;
            last if $line eq '';
            push @lines, decode_utf8($line);
        }
        my $data_text = join ' ',@lines;
        $iologger->debug("+-> $data_text");
        Log::Log4perl::NDC->push("input='$line1' data='$data_text'");
        $req->req_data(_decode_data($data_text));
    }
    else {
        Log::Log4perl::NDC->push("input='$line1'");
    }
    die "Not allowed\n"
        unless $req->check_allowed();
    my $response = $req->respond();
    my $DBError = $self->otrs->{db}->Error();
    die "Database error: $DBError\n" if $DBError;
    $response =~ s/\n/$CRLF/g;
    print $cl encode_utf8($response);
    $cl->flush();
    $req->background_process();  # Das kann später in einen Thread ausgelagert werden
    Log::Log4perl::NDC->pop();
    if ($iologger->is_debug) {
        $response =~ s/\s+\z//;
        $iologger->debug("<-- $response");
    }
    return ! $req->is_quit();
}

sub _decode_data {
    my ($data) = @_;
    my $result = eval { return decode_json($data); };
    die "Data not in JSON format: $@" if $@;
    return $result;
}

# return whether the real user is allowed to change to the given user.
# dies if the remote host is not allowed to connect at all.
# returns 0 if the new_user_name is undefined (for connects by www-run etc.)
sub is_allowed_to_change_user {
    my ($self, $new_user_name) = @_;
    my $logger     = get_logger(__PACKAGE__);
    my $real_user_name = $self->real_user_name;
    my $hostname = $self->remote_host;
    $logger->debug( "remote host:" . $hostname );
    my $user_map = 
        $self->otrs->{config}->{'TicketServer::HostUserMap'}->{$hostname};
    die "Your Host ($hostname) is not allowed to connect! (See " 
      . ' $Config->{"TicketServer::HostUserMap"}->{\''
      . $hostname
      . "'})\n"
      unless defined $user_map;
    return undef unless defined $new_user_name;

    my $permitted_users = $user_map->{$real_user_name};
    $permitted_users = $user_map->{'*'}
      unless defined $permitted_users;
    if (! defined $permitted_users) {
        $logger->debug("No users permitted.");
        return 0;
    }
    $logger->debug( "permitted users: @$permitted_users" );
    
    foreach my $user_cand (@$permitted_users) {
        if ( ( $new_user_name eq $user_cand ) || ( $user_cand eq '*' ) ) {
            return 1;
        }
        if ( ( $new_user_name eq $real_user_name ) && ( $user_cand eq '' ) ) {
            return 1;
        }
    }
    $logger->debug("Forbidden");
    return 0;
}

1;
