use strict;
use warnings;
use utf8;

package Kernel::Noris::TicketServer;
   
use FindBin qw($Bin);
use lib "$Bin/../cpan-lib";
unshift (@INC, "$Bin/../..", "$Bin/../cpan-lib");

use Encode;
use IO::Socket::SSL;
use Net::Ident qw(ident_lookup);
use JSON ();
use Data::Dump qw(pp);

use Authen::Libwrap qw(hosts_ctl);
use Getopt::Long qw(GetOptions :config no_ignore_case);
use Kernel::Noris::TicketServer::Request;

use Kernel::Config;
use Kernel::Noris::TicketServer::Connection;
use Log::Log4perl qw(get_logger);
use Log::Log4perl::NDC;
use Pod::Usage qw(pod2usage);
use Socket;

my $IDENT_TIMEOUT = 30; # 30 Sekunden
my $DEFAULT_PORT = 54444;

my $LogCategory = __PACKAGE__;

my $fake_user;

# kümmert sich um den Socket und ruft den Rest auf
sub server {
    my ($port) = @_;
    my $logger = get_logger($LogCategory);
    local $SIG{CHLD} = 'IGNORE';
    # plain (non-ssl) server
    my $srv = IO::Socket::INET->new( 
        Listen => 1, 
        LocalPort => $port, 
        Proto => 'tcp', 
        ReuseAddr => 1
    ) or $logger->error_die("Kann keinen Server-Socket für Port $port öffnen: $!\n");
    $logger->info("Ticket-Server auf Port $port gestartet");
    while (1) {
        my $cl = $srv->accept() or next; # non-ssl accept
        
        if ( ! hosts_ctl( 'ticketserver', $cl )) {
            
            $logger->info('Zugriff von der Client-IP ' . $cl->peerhost() 
                 . ' nicht zugelassen (hosts.allow ...)');
            next;
        }
        defined( my $pid = fork()) or logger->error_die($!);
        if ( $pid ) {
            # parent
            close($cl);
            next;
        }
        # child
#         IO::Socket::SSL->start_SSL( $cl, 
#                 SSL_server => 1,
#                 SSL_ca_file => .. , SSL_cert_file => .., ...
#             ) or die $SSL_ERROR;
        # $cl ist nun IO::Socket::SSL objekt, verbunden mit
        # client
        {
            close($srv);
            my $logger = get_logger($LogCategory);
            Log::Log4perl::NDC->remove();
            eval {
                Log::Log4perl::NDC->push("remote=" . $cl->peerhost() . ":" . $cl->peerport());
                
                my $username = get_user($cl);
                $logger->debug("connected");
                my $connection = Kernel::Noris::TicketServer::Connection->new(
                    real_user_name => $username,
                    remote_host => _get_hostname($cl->peeraddr()),
                );
                while ($connection->handle_request($cl)) {
                    ;
                }
            };
            if ($@) {
                my $errmsg = $@;
                chomp $errmsg;
                my $logger = get_logger($LogCategory);
                $logger->info("Error: $errmsg");
                print $cl 'ERROR: ' . encode("utf8",$errmsg) . "\n\n";
                $cl->flush();
            }
            $logger->debug("closing");
            close($cl);
        }
        return;
    }
}

# ermittelt den Hostname eines Rechners anhand seiner IP-Addresse
# und überprüft ob Vorwärts und Rückwärtsauflösung übereinstimmen.
sub _get_hostname {
    my ($packed_ipaddr) = @_;
    my $hostname = gethostbyaddr( $packed_ipaddr, AF_INET );

    my ( undef, undef, undef, undef, @packed_ips ) = gethostbyname($hostname);

    die "No Reverse Lookup found for: " . inet_ntoa($packed_ipaddr) . "\n"
      unless (@packed_ips);

    foreach my $ip (@packed_ips) {
        return $hostname
          if ( $ip eq $packed_ipaddr );
    }

    die "Reverse Lookup check failed for:"
      . inet_ntoa($packed_ipaddr)
      . "<-> $hostname\n";
}

# ermittelt den Benutzernamen (als String) über ident.
sub get_user {
    my ($cl) = @_;
    my $username = ident_lookup($cl, $IDENT_TIMEOUT);
    chomp $username;
    die "ident lookup failed\n" unless defined $username;
    my $logger = get_logger();
    Log::Log4perl::NDC->push("real-user=$username");
    
    return $username;
}

sub setup_logging {
    my ($log_config, $debug_categories) = @_;
    if ( defined $log_config ) {
        die "Log-Konfigurationsdatei $log_config existiert nicht"  unless -e $log_config;
        Log::Log4perl->init_and_watch($log_config, "HUP");
    }
    else {
        # Konfiguration für Testserver auf Kommandozeile:
        # Nur Probleme ausgeben
        Log::Log4perl->init(\<<EOT);
            log4perl.logger = INFO, debug
            log4perl.appender.debug = Log::Log4perl::Appender::Screen
            log4perl.appender.debug.stderr  = 1
            log4perl.appender.debug.utf8 = 1
            log4perl.appender.debug.layout = Log::Log4perl::Layout::PatternLayout
            log4perl.appender.debug.layout.ConversionPattern = %c pid=%P %x: %m%n
EOT
    }
    if (defined @$debug_categories) {
        for my $cat (@$debug_categories) {
            my $logger = get_logger($cat);
            $logger->level("DEBUG");
        }
    }
}

sub parse_options {
    my ($port, $log_config, $help, @debug_categories) 
        = ($DEFAULT_PORT);
    GetOptions(
        'port|p=i'        => \$port,
        'debug|d:s'       => \@debug_categories,
        'logconf|l=s'     => \$log_config,
        'help|usage|h'    => \$help,
    ) || pod2usage( -verbose => 0, -exitval => 2 );
    if (@debug_categories == 1 && $debug_categories[0] eq '') {
        $debug_categories[0] = __PACKAGE__;
    }
    return {
        port              => $port,
        debug_categories  => \@debug_categories,
        log_config        => $log_config,
        help              => $help,
    };
}

##############################
# main
##############################

{
    my $opt = parse_options();
    pod2usage( -verbose => 2, -exitval => 0 ) if $opt->{help};
    setup_logging($opt->{log_config}, $opt->{debug_categories});
    server($opt->{port});
}

=head1 NAME

TicketServer - Ein Server, der OTRS-Tickets über tcp zur Verfügung stellt.

=head1 SYNOPSIS

TicketServer [options]

 Options:
    --port, -p      Der zu öffnende tcp-Port. Default ist $DEFAULT_PORT.
    --logconf, -l   Dateiname der Log-Konfiguration. 
                    Ohne diese Option wird eine Standardkonfiguration benutzt,
                    bei der nur Info-Meldungen an stderr ausgegeben werden.
                    Siehe auch Log::Log4Perl
    --debug, -d     Debug-Meldungen für otrs.Noris.TicketServer an stderr ausgeben.
    --debug=<Kategorie>, -d=<Kategorie>
                    Für die angegebene Kategorie Debug-Meldungen an stderr ausgeben.
                    Die Option kann mehrfach verwendet werden.

=cut

1;
