# --
# Kernel/Modules/AgentTicketNorisSearch.pm - Utilities for tickets
# Copyright (C) 2009 - noris network AG, http://www.noris.net/
# --
# $Id$
# --
# This software comes with ABSOLUTELY NO WARRANTY. For details, see
# the enclosed file COPYING for license information (GPL). If you
# did not receive this file, see http://www.gnu.org/licenses/gpl-2.0.txt.
# --

package Kernel::Modules::AgentTicketNorisSearch;

use strict;
use warnings;

use Kernel::System::CustomerUser;
use Kernel::System::Priority;
use Kernel::System::SearchProfile;
use Kernel::System::Service;
use Kernel::System::SLA;
use Kernel::System::State;
use Kernel::System::Type;

use vars qw($VERSION);
$VERSION = qw($Revision: 2.3.3 $) [1];

sub new {
    my ( $Type, %Param ) = @_;

    # allocate new hash for object
    my $Self = {%Param};
    bless( $Self, $Type );

    # check needed objects
    for (qw(ParamObject DBObject TicketObject LayoutObject LogObject ConfigObject MainObject)) {
        if ( !$Self->{$_} ) {
            $Self->{LayoutObject}->FatalError( Message => "Got no $_!" );
        }
    }
    $Self->{CustomerUserObject}  = Kernel::System::CustomerUser->new(%Param);
    $Self->{PriorityObject}      = Kernel::System::Priority->new(%Param);
    $Self->{StateObject}         = Kernel::System::State->new(%Param);
    $Self->{SearchProfileObject} = Kernel::System::SearchProfile->new(%Param);
    $Self->{ServiceObject}       = Kernel::System::Service->new(%Param);
    $Self->{SLAObject}           = Kernel::System::SLA->new(%Param);
    $Self->{TypeObject}          = Kernel::System::Type->new(%Param);

    # if we need to do a fulltext search on an external mirror database
    if ( $Self->{ConfigObject}->Get('Core::MirrorDB::DSN') ) {
        my $ExtraDatabaseObject = Kernel::System::DB->new(
            LogObject    => $Param{LogObject},
            ConfigObject => $Param{ConfigObject},
            MainObject   => $Param{MainObject},
            DatabaseDSN  => $Self->{ConfigObject}->Get('Core::MirrorDB::DSN'),
            DatabaseUser => $Self->{ConfigObject}->Get('Core::MirrorDB::User'),
            DatabasePw   => $Self->{ConfigObject}->Get('Core::MirrorDB::Password'),
        );
        if ( !$ExtraDatabaseObject ) {
            $Self->{LayoutObject}->FatalError();
        }
        $Self->{TicketObjectSearch} = Kernel::System::Ticket->new(
            %Param,
            DBObject => $ExtraDatabaseObject,
        );
    }
    else {
        $Self->{TicketObjectSearch} = $Self->{TicketObject};
    }

    $Self->{Config} = $Self->{ConfigObject}->Get("Ticket::Frontend::$Self->{Action}");

    return $Self;
}

sub Run {
    my ( $Self, %Param ) = @_;

    my $Output;

    # get confid data
    $Self->{StartHit}        = $Self->{ParamObject}->GetParam( Param => 'StartHit' ) || 1;
    $Self->{SearchLimit}     = $Self->{Config}->{SearchLimit}                        || 200;
    $Self->{SearchPageShown} = $Self->{Config}->{SearchPageShown}                    || 40;
    $Self->{SortBy} = $Self->{Config}->{'SortBy::Default'} || ['Age'];
    $Self->{Order}  = $Self->{Config}->{'Order::Default'}  || ['Down'];
    $Self->{Profile}        = $Self->{ParamObject}->GetParam( Param => 'Profile' )        || '';
    $Self->{SaveProfile}    = $Self->{ParamObject}->GetParam( Param => 'SaveProfile' )    || '';
    $Self->{TakeLastSearch} = $Self->{ParamObject}->GetParam( Param => 'TakeLastSearch' ) || '';
    $Self->{SelectTemplate} = $Self->{ParamObject}->GetParam( Param => 'SelectTemplate' ) || '';
    $Self->{EraseTemplate}  = $Self->{ParamObject}->GetParam( Param => 'EraseTemplate' )  || '';

    # check request
    if ( $Self->{Subaction} eq 'OpenSearchDescription' ) {
        my $Output = $Self->{LayoutObject}->Output(
            TemplateFile => 'AgentTicketNorisSearchOpenSearchDescription',
            Data         => {%Param},
        );
        return $Self->{LayoutObject}->Attachment(
            Filename    => 'OpenSearchDescription.xml',
            ContentType => "text/xml",
            Content     => $Output,
            Type        => 'inline',
        );
    }

    # check request
    if ( $Self->{ParamObject}->GetParam( Param => 'SearchTemplate' ) && $Self->{Profile} ) {
        return $Self->{LayoutObject}->Redirect(
            OP =>
                "Action=AgentTicketNorisSearch&Subaction=Search&TakeLastSearch=1&SaveProfile=1&Profile=$Self->{Profile}"
        );
    }

    my $ShowColumnsCount = 2;

    # get signle params
    my %GetParam = ();

    # load profiles string params (press load profile)
    if ( $Self->{Subaction} eq 'LoadProfile' && $Self->{Profile} || $Self->{TakeLastSearch} ) {
        %GetParam = $Self->{SearchProfileObject}->SearchProfileGet(
            Base      => 'TicketSearch',
            Name      => $Self->{Profile},
            UserLogin => $Self->{UserLogin},
        );
        if ($Self->{Subaction} eq 'LoadProfile') {
            delete $GetParam{Sort};
            delete $GetParam{Order};
        }
        else {
            for (qw(Sort Order)) {
                if ( $GetParam{$_} ) {
                    $GetParam{$_} = $Self->{ParamObject}->GetParam( Param => $_ );
                }
            }
        }
    }

    # get search string params (get submitted params)
    else {
        for (
            qw(TicketNumber Title From To Cc Subject Body CustomerID CustomerUserLogin StateType
            Agent ResultForm CreateTimeSearchType CloseTimeSearchType UseSubQueues
            TicketEscalationTime
            TicketEscalationTimeStart TicketEscalationTimeStartDay TicketEscalationTimeStartMonth
            TicketEscalationTimeStartYear
            TicketEscalationTimeStop TicketEscalationTimeStopDay TicketEscalationTimeStopMonth
            TicketEscalationTimeStopYear           
            TicketEscalationTimePoint TicketEscalationTimePointFormat TicketEscalationTimePointStart
            EscalationTimeSearchType
            TicketFreeTime1
            TicketFreeTime1Start TicketFreeTime1StartDay TicketFreeTime1StartMonth
            TicketFreeTime1StartYear
            TicketFreeTime1Stop TicketFreeTime1StopDay TicketFreeTime1StopMonth
            TicketFreeTime1StopYear
            TicketFreeTime2
            TicketFreeTime2Start TicketFreeTime2StartDay TicketFreeTime2StartMonth
            TicketFreeTime2StartYear
            TicketFreeTime2Stop TicketFreeTime2StopDay TicketFreeTime2StopMonth
            TicketFreeTime2StopYear
            TicketFreeTime3
            TicketFreeTime3Start TicketFreeTime3StartDay TicketFreeTime3StartMonth
            TicketFreeTime3StartYear
            TicketFreeTime3Stop TicketFreeTime3StopDay TicketFreeTime3StopMonth
            TicketFreeTime3StopYear
            TicketFreeTime4
            TicketFreeTime4Start TicketFreeTime4StartDay TicketFreeTime4StartMonth
            TicketFreeTime4StartYear
            TicketFreeTime4Stop TicketFreeTime4StopDay TicketFreeTime4StopMonth
            TicketFreeTime4StopYear
            TicketFreeTime5
            TicketFreeTime5Start TicketFreeTime5StartDay TicketFreeTime5StartMonth
            TicketFreeTime5StartYear
            TicketFreeTime5Stop TicketFreeTime5StopDay TicketFreeTime5StopMonth
            TicketFreeTime5StopYear
            TicketFreeTime6
            TicketFreeTime6Start TicketFreeTime6StartDay TicketFreeTime6StartMonth
            TicketFreeTime6StartYear
            TicketFreeTime6Stop TicketFreeTime6StopDay TicketFreeTime6StopMonth
            TicketFreeTime6StopYear
            TicketFreeTime7
            TicketFreeTime7Start TicketFreeTime7StartDay TicketFreeTime7StartMonth
            TicketFreeTime7StartYear
            TicketFreeTime7Stop TicketFreeTime7StopDay TicketFreeTime7StopMonth
            TicketFreeTime7StopYear
            TicketFreeTime8
            TicketFreeTime8Start TicketFreeTime8StartDay TicketFreeTime8StartMonth
            TicketFreeTime8StartYear
            TicketFreeTime8Stop TicketFreeTime8StopDay TicketFreeTime8StopMonth
            TicketFreeTime8StopYear
            TicketCreateTimePointFormat TicketCreateTimePoint
            TicketCreateTimePointStart
            TicketCreateTimeStart TicketCreateTimeStartDay TicketCreateTimeStartMonth
            TicketCreateTimeStartYear
            TicketCreateTimeStop TicketCreateTimeStopDay TicketCreateTimeStopMonth
            TicketCreateTimeStopYear
            TicketCloseTimePointFormat TicketCloseTimePoint
            TicketCloseTimePointStart
            TicketCloseTimeStart TicketCloseTimeStartDay TicketCloseTimeStartMonth
            TicketCloseTimeStartYear
            TicketCloseTimeStop TicketCloseTimeStopDay TicketCloseTimeStopMonth
            TicketCloseTimeStopYear
            LinkedObjectID LinkedObjectKey
            )
            )
        {

            # get search string params (get submitted params)
            $GetParam{$_} = $Self->{ParamObject}->GetParam( Param => $_ );

            # remove white space on the start and end
            if ( $GetParam{$_} ) {
                $GetParam{$_} =~ s/\s+$//g;
                $GetParam{$_} =~ s/^\s+//g;
            }
        }
        if ( $Param{FastSearch} ) {
            $GetParam{TicketNumber} = $Param{TicketNumber} unless $GetParam{TicketNumber};
            $GetParam{Title} = $Param{Title} unless $GetParam{Title};
        }
    }
    if ($Self->{Subaction} eq 'LoadProfile') {
        delete $GetParam{Sort};
        delete $GetParam{Order};
    }
    else {
        for (qw(Sort Order)) {
            $GetParam{$_} = $Self->{ParamObject}->GetParam( Param => $_ );
        }
    }

    # get creation, close, and escalation time option
    for my $Key ('Create', 'Close', 'Escalation') {
        if ( !$GetParam{"${Key}TimeSearchType"} ) {
            $GetParam{"${Key}TimeSearchType::None"} = 'checked="checked"';
        }
        elsif ( $GetParam{"${Key}TimeSearchType"} eq 'TimePoint' ) {
            $GetParam{"${Key}TimeSearchType::TimePoint"} = 'checked="checked"';
        }
        elsif ( $GetParam{"${Key}TimeSearchType"} eq 'TimeSlot' ) {
            $GetParam{"${Key}TimeSearchType::TimeSlot"} = 'checked="checked"';
        }
    }

    # set result form env
    if ( !$GetParam{ResultForm} ) {
        $GetParam{ResultForm} = '';
    }
    if ( $GetParam{ResultForm} eq 'Print' || $GetParam{ResultForm} eq 'CSV' ) {
        $Self->{SearchPageShown} = $Self->{SearchLimit};
    }
    my %ShowTicketData = (
        Profile => $Self->{Profile} || 'last-search'
    );
    for ( qw( Age From Subject Title State Queue Owner CustomerID CustomerName Priority ) ) {

        # get Show string params (get submitted params)
        $GetParam{ 'ShowTicket' . $_ } =
            $Self->{ParamObject}->GetParam( Param => 'ShowTicket' . $_ )
                || $GetParam{ 'ShowTicket' . $_ };
#                || ! $Self->{Subaction};
        if ( $GetParam{ 'ShowTicket' . $_ } ) {
            $Self->{LayoutObject}->Block(
                Name => 'ShowHeadTicket' . $_,
                Data => \%ShowTicketData,
            );
            $ShowColumnsCount++;
        }
    }
    for ( 1 .. 16 ) {

        # get Show string params (get submitted params)
        if ( $GetParam{ 'ShowFreeText' . $_ } =
            $Self->{ParamObject}->GetParam( Param => 'ShowFreeText' . $_ )
            || $GetParam{ 'ShowFreeText' . $_ }
           )
        {
            if ( my $tmpFreeKeyH = $Self->{ConfigObject}->Get( 'TicketFreeKey' . $_ ) ) {
                $ShowColumnsCount++;

                # generate free key
                my @tmpFreeKeyA = keys %{$tmpFreeKeyH};

                $Self->{LayoutObject}->Block(
                    Name => 'ShowHeadFreeText',
                    Data => {
                        TicketFreeTextKey => $tmpFreeKeyH->{ $tmpFreeKeyA[0] },
                        Count => $_,
                        %ShowTicketData,
                    }
                );
            } else {
                $Self->{LayoutObject}->Block( Name => 'ShowHeadFreeText', );
            }
        }
    }
    for ( 1 .. 6 ) {

        # get Show string params (get submitted params)
        if ( $GetParam{ 'ShowFreeTime' . $_ } =
            $Self->{ParamObject}->GetParam( Param => 'ShowFreeTime' . $_ )
            || $GetParam{ 'ShowFreeTime' . $_ }
           )
        {
            if ( my $tmpFreeTimeKey = $Self->{ConfigObject}->Get( 'TicketFreeTimeKey' . $_ ) ) {
                $ShowColumnsCount++;
                $Self->{LayoutObject}->Block(
                    Name => 'ShowHeadFreeTime',
                    Data => {
                        TicketFreeTimeKey => $tmpFreeTimeKey,
                        Count => $_,
                        %ShowTicketData
                    }
                );
            }
        }
    }

    # get array params
    for (
        qw(StateIDs StateTypeIDs QueueIDs PriorityIDs OwnerIDs
        CreatedQueueIDs CreatedUserIDs WatchUserIDs ResponsibleIDs
        TypeIDs ServiceIDs SLAIDs
        TicketFreeKey1 TicketFreeText1 TicketFreeKey2 TicketFreeText2
        TicketFreeKey3 TicketFreeText3 TicketFreeKey4 TicketFreeText4
        TicketFreeKey5 TicketFreeText5 TicketFreeKey6 TicketFreeText6
        TicketFreeKey7 TicketFreeText7 TicketFreeKey8 TicketFreeText8
        TicketFreeKey9 TicketFreeText9 TicketFreeKey10 TicketFreeText10
        TicketFreeKey11 TicketFreeText11 TicketFreeKey12 TicketFreeText12
        TicketFreeKey13 TicketFreeText13 TicketFreeKey14 TicketFreeText14
        TicketFreeKey15 TicketFreeText15 TicketFreeKey16 TicketFreeText16)
        )
    {

        # get search array params (get submitted params)
        my @Array = $Self->{ParamObject}->GetArray( Param => $_ );
        if (@Array) {
            $GetParam{$_} = \@Array;
        }
    }

    # get Sorting params
    my (@SortArray , @OrderArray);
    if ( $GetParam{'Sort'} && $GetParam{'Order'} ) {
        push @SortArray,  $GetParam{'Sort'};
        push @OrderArray, $GetParam{'Order'};
    }
    for ( 1 .. 3 ) {
        $GetParam{"SortBy$_"} = $Self->{ParamObject}->GetParam( Param => 'SortBy' . $_ )
          || $GetParam{"SortBy$_"}
          || next;

        $GetParam{"Order$_"} = $Self->{ParamObject}->GetParam( Param => 'Order' . $_ )
          || $GetParam{"Order$_"}
          || next;

        # Needed for TicketSearch method
        push @SortArray,  $GetParam{"SortBy$_"};
        push @OrderArray, $GetParam{"Order$_"};
    }

    # Save 
    if ( @SortArray && @OrderArray ) {
        $Self->{SortBy} = \@SortArray;
        $Self->{Order}  = \@OrderArray;
    }
    unless ( ref( $Self->{SortBy} ) ) {
        $Self->{SortBy} = [ $Self->{SortBy} ];
        $Self->{Order} = [ $Self->{Order} ];
    }

    # show result site
    if ( $Self->{Subaction} eq 'Search' && !$Self->{EraseTemplate} ) {

        # fill up profile name (e.g. with last-search)
        if ( !$Self->{Profile} || !$Self->{SaveProfile} ) {
            $Self->{Profile} = 'last-search';
        }

        # store last queue screen
        my $URL
            = "Action=AgentTicketNorisSearch&Subaction=Search&Profile=$Self->{Profile}"
            . "&TakeLastSearch=1&StartHit=$Self->{StartHit}";
        $Self->{SessionObject}->UpdateSessionID(
            SessionID => $Self->{SessionID},
            Key       => 'LastScreenOverview',
            Value     => $URL,
        );
        $Self->{SessionObject}->UpdateSessionID(
            SessionID => $Self->{SessionID},
            Key       => 'LastScreenView',
            Value     => $URL,
        );

        # save search profile (under last-search or real profile name)
        $Self->{SaveProfile} = 1;

        # remember last search values
        if ( $Self->{SaveProfile} && $Self->{Profile} ) {

            # remove old profile stuff
            $Self->{SearchProfileObject}->SearchProfileDelete(
                Base      => 'TicketSearch',
                Name      => $Self->{Profile},
                UserLogin => $Self->{UserLogin},
            );

            # insert new profile params
            for my $Key ( keys %GetParam ) {
                if ( $GetParam{$Key} ) {
                    $Self->{SearchProfileObject}->SearchProfileAdd(
                        Base      => 'TicketSearch',
                        Name      => $Self->{Profile},
                        Key       => $Key,
                        Value     => $GetParam{$Key},
                        UserLogin => $Self->{UserLogin},
                    );
                } else {
                    if ($Key =~ m/^Show/) {
                        $Self->{SearchProfileObject}->SearchProfileAdd(
                            Base      => 'TicketSearch',
                            Name      => $Self->{Profile},
                            Key       => $Key,
                            Value     => $GetParam{$Key} || 0,
                            UserLogin => $Self->{UserLogin},
                        );
                    }
                }
            }
        }

        # get create, close, and escalation time settings
        for my $Key ('Create', 'Close', 'Escalation') {
            if ( !$GetParam{"${Key}TimeSearchType"} ) {
    
                # do noting ont time stuff
            }
            elsif ( $GetParam{"${Key}TimeSearchType"} eq 'TimeSlot' ) {
                for (qw(Month Day)) {
                    $GetParam{"Ticket${Key}TimeStart$_"}
                        = sprintf( "%02d", $GetParam{"Ticket${Key}TimeStart$_"} );
                }
                for (qw(Month Day)) {
                    $GetParam{"Ticket${Key}TimeStop$_"}
                        = sprintf( "%02d", $GetParam{"Ticket${Key}TimeStop$_"} );
                }
                if (
                    $GetParam{"Ticket${Key}TimeStartDay"}
                    && $GetParam{"Ticket${Key}TimeStartMonth"}
                    && $GetParam{"Ticket${Key}TimeStartYear"}
                    )
                {
                    $GetParam{"Ticket${Key}TimeNewerDate"}
                        = $GetParam{"Ticket${Key}TimeStartYear"} . '-'
                        . $GetParam{"Ticket${Key}TimeStartMonth"} . '-'
                        . $GetParam{"Ticket${Key}TimeStartDay"}
                        . ' 00:00:01';
                }
                if (
                    $GetParam{"Ticket${Key}TimeStopDay"}
                    && $GetParam{"Ticket${Key}TimeStopMonth"}
                    && $GetParam{"Ticket${Key}TimeStopYear"}
                    )
                {
                    $GetParam{"Ticket${Key}TimeOlderDate"}
                        = $GetParam{"Ticket${Key}TimeStopYear"} . '-'
                        . $GetParam{"Ticket${Key}TimeStopMonth"} . '-'
                        . $GetParam{"Ticket${Key}TimeStopDay"}
                        . ' 23:59:59';
                }
            }
            elsif ( $GetParam{"${Key}TimeSearchType"} eq 'TimePoint' ) {
                if (
                    $GetParam{"Ticket${Key}TimePoint"}
                    && $GetParam{"Ticket${Key}TimePointStart"}
                    && $GetParam{"Ticket${Key}TimePointFormat"}
                    )
                {
                    my $Time = 0;
                    if ( $GetParam{"Ticket${Key}TimePointFormat"} eq 'minute' ) {
                        $Time = $GetParam{"Ticket${Key}TimePoint"};
                    }
                    elsif ( $GetParam{"Ticket${Key}TimePointFormat"} eq 'hour' ) {
                        $Time = $GetParam{"Ticket${Key}TimePoint"} * 60;
                    }
                    elsif ( $GetParam{"Ticket${Key}TimePointFormat"} eq 'day' ) {
                        $Time = $GetParam{"Ticket${Key}TimePoint"} * 60 * 24;
                    }
                    elsif ( $GetParam{"Ticket${Key}TimePointFormat"} eq 'week' ) {
                        $Time = $GetParam{"Ticket${Key}TimePoint"} * 60 * 24 * 7;
                    }
                    elsif ( $GetParam{"Ticket${Key}TimePointFormat"} eq 'month' ) {
                        $Time = $GetParam{"Ticket${Key}TimePoint"} * 60 * 24 * 30;
                    }
                    elsif ( $GetParam{"Ticket${Key}TimePointFormat"} eq 'year' ) {
                        $Time = $GetParam{"Ticket${Key}TimePoint"} * 60 * 24 * 365;
                    }
                    
                    if ( $GetParam{"Ticket${Key}TimePointStart"} eq 'Before' ) {
                        $GetParam{"Ticket${Key}TimeOlderMinutes"} = $Time;
                    }
                    elsif ( $GetParam{"Ticket${Key}TimePointStart"} eq 'After' ) {
                        $GetParam{"Ticket${Key}TimeNewerMinutes"} = -$Time;
                    }
                    elsif ( $GetParam{"Ticket${Key}TimePointStart"} eq 'Next' ) {
                        $GetParam{"Ticket${Key}TimeOlderMinutes"} = -$Time;
                        $GetParam{"Ticket${Key}TimeNewerMinutes"} = 0;
                    }
                    else {
                        $GetParam{"Ticket${Key}TimeNewerMinutes"} = $Time;
                        $GetParam{"Ticket${Key}TimeOlderMinutes"} = 0;
                    }
                }
            }
        }
        
        # free time
        for ( qw(TicketFreeTime1 TicketFreeTime2 TicketFreeTime3
                 TicketFreeTime4 TicketFreeTime5 TicketFreeTime6
                ) ) {
            if ( !$GetParam{ $_ } ) {
                for my $Type (qw(Year Month Day)) {
                    $GetParam{ $_ . 'Start' . $Type } = undef;
                    $GetParam{ $_ . 'Stop' . $Type }  = undef;
                }
                $GetParam{ $_ . 'NewerDate' } = undef;
                $GetParam{ $_ . 'OlderDate' } = undef;
            }
            else {
                $GetParam{ $_ } = 'checked="checked"';
                if (
                    $GetParam{ $_ . 'StartDay' }
                    && $GetParam{ $_ . 'StartMonth' }
                    && $GetParam{ $_ . 'StartYear' }
                    )
                {
                    $GetParam{ $_ . 'NewerDate' }
                        = $GetParam{ $_ . 'StartYear' } . '-'
                        . $GetParam{ $_ . 'StartMonth' } . '-'
                        . $GetParam{ $_ . 'StartDay' }
                        . ' 00:00:01';
                }
                if (
                    $GetParam{ $_ . 'StopDay' }
                    && $GetParam{ $_ . 'StopMonth' }
                    && $GetParam{ $_ . 'StopYear' }
                    )
                {
                    $GetParam{ $_ . 'OlderDate' }
                        = $GetParam{ $_ . 'StopYear' } . '-'
                        . $GetParam{ $_ . 'StopMonth' } . '-'
                        . $GetParam{ $_ . 'StopDay' }
                        . ' 23:59:59';
                }
            }
        }

        # focus of "From To Cc Subject Body Title"
        for (qw(From To Cc Subject Body Title)) {
            if ( defined( $GetParam{$_} ) && $GetParam{$_} ne '' ) {
                $GetParam{$_} = "*$GetParam{$_}*";
            }
        }

        # lookup customer
        if ($GetParam{CustomerID}) {
            my $NewCustomerID = $Self->{CustomerUserObject}->CustomerIDTranslate(
                CustomerID => $GetParam{CustomerID}
            );
            if ($NewCustomerID) {
                $GetParam{CustomerID} = $NewCustomerID;
            }
        }

        # perform ticket search
        my $Counter     = 0;
        my @ViewableIDs = $Self->{TicketObjectSearch}->TicketSearch(
            Result          => 'ARRAY',
            SortBy          => $Self->{SortBy},
            OrderBy         => $Self->{Order},
            Limit           => $Self->{SearchLimit},
            UserID          => $Self->{UserID},
            ConditionInline => $Self->{Config}->{ExtendedSearchCondition},
            FullTextIndex   => 1,
            $GetParam{TicketNumber} =~ /^[^*?_%]+$/ ?
                (TicketNumber => $GetParam{TicketNumber})
                : %GetParam,
        );

        my @CSVHead = ();
        my @CSVData = ();
        my @PDFData = ();
        for (@ViewableIDs) {
            $Counter++;

            # build search result
            if (
                $Counter >= $Self->{StartHit}
                && $Counter < ( $Self->{SearchPageShown} + $Self->{StartHit} )
                )
            {

                # get first article data
                my %Data = $Self->{TicketObjectSearch}->ArticleFirstArticle( TicketID => $_ );

                my %TicketData = $Self->{TicketObjectSearch}->TicketGet( TicketID => $_, );
                $Data{Priority} = $TicketData{Priority};
                for ( 1 .. 16 ) {
                    #$Data{ 'TicketFreeKey' . $_ }  = $TicketData{ 'TicketFreeKey' . $_ };
                    $Data{ 'TicketFreeText' . $_ } = $TicketData{ 'TicketFreeText' . $_ };
                }
                for ( 1 .. 6 ) {
                    #$Data{ 'TicketFreeTimeKey' . $_ } = $TicketData{ 'TicketFreeTimeKey' . $_ };
                    $Data{ 'TicketFreeTime' . $_ }    = $TicketData{ 'TicketFreeTime' . $_ };
                }


                # get whole article (if configured!)
                if ( $Self->{Config}->{'SearchArticleCSVTree'} && $GetParam{ResultForm} eq 'CSV' ) {
                    my @Article = $Self->{TicketObjectSearch}->ArticleGet( TicketID => $_ );
                    for my $Articles (@Article) {
                        if ( $Articles->{Body} ) {
                            $Data{ArticleTree}
                                .= "\n-->||$Articles->{ArticleType}||$Articles->{From}||"
                                . $Articles->{Created}
                                . "||<--------------\n"
                                . $Articles->{Body};
                        }
                    }
                }

                # customer info
                my %CustomerData = ();
                if ( $Data{CustomerUserID} ) {
                    %CustomerData = $Self->{CustomerUserObject}->CustomerUserDataGet(
                        User => $Data{CustomerUserID},
                    );
                }

                # customer info (customer name)
                if ( $CustomerData{UserLogin} ) {
                    $Data{CustomerName} = $Self->{CustomerUserObject}->CustomerName(
                        UserLogin => $CustomerData{UserLogin},
                    );
                }

                # user info
                my %UserInfo = $Self->{UserObject}->GetUserData(
                    User   => $Data{Owner},
                    Cached => 1
                );

                # get age
                $Data{Age} = $Self->{LayoutObject}->CustomerAgeHumanReadable( Age => $Data{Age}, Space => ' ' );

                # customer info string
                $UserInfo{CustomerName} = '(' . $UserInfo{CustomerName} . ')'
                    if ( $UserInfo{CustomerName} );

                # generate ticket result
                if ( $GetParam{ResultForm} eq 'Preview' ) {

                    # check if just a only html email
                    if (
                        my $MimeTypeText
                        = $Self->{LayoutObject}->CheckMimeType(
                            %Data, Action => 'AgentTicketZoom',
                        )
                        )
                    {
                        $Data{TextNote} = $MimeTypeText;
                        $Data{Body}     = '';
                    }
                    else {

                        # do some text quoting
                        $Data{Body} = $Self->{LayoutObject}->Ascii2Html(
                            NewLine => $Self->{ConfigObject}->Get('DefaultViewNewLine') || 85,
                            Text    => $Data{Body},
                            VMax    => $Self->{Config}->{SearchViewableTicketLines}     || 15,
                            StripEmptyLines => 1,
                            HTMLResultMode  => 1,
                        );

                        # do charset check
                        if (
                            my $CharsetText = $Self->{LayoutObject}->CheckCharset(
                                Action         => 'AgentTicketZoom',
                                ContentCharset => $Data{ContentCharset},
                                TicketID       => $Data{TicketID},
                                ArticleID      => $Data{ArticleID}
                            )
                            )
                        {
                            $Data{TextNote} = $CharsetText;
                        }
                    }

                    # customer info string
                    $UserInfo{CustomerTable} = $Self->{LayoutObject}->AgentCustomerViewTable(
                        Data => \%CustomerData,
                        Max  => $Self->{ConfigObject}->Get(
                            'Ticket::Frontend::CustomerInfoQueueMaxSize'
                        ),
                    );

                    # do some html highlighting
                    my $HighlightStart = '<font color="orange"><b><i>';
                    my $HighlightEnd   = '</i></b></font>';
                    if (%GetParam) {
                        for (qw(Body From To Subject)) {
                            if ( $GetParam{$_} ) {
                                $GetParam{$_} =~ s/(\*|\%)//g;
                                my @Parts = split( '%', $GetParam{$_} );
                                if ( $Data{$_} ) {
                                    for my $Part (@Parts) {
                                        $Data{$_} =~ s/($Part)/$HighlightStart$1$HighlightEnd/gi;
                                    }
                                }
                            }
                        }
                    }
                    for (qw(From To Subject)) {
                        if ( !$GetParam{$_} ) {
                            $Data{$_}
                                = $Self->{LayoutObject}->Ascii2Html( Text => $Data{$_}, Max => 80 );
                        }
                    }

                    # add ticket block
                    $Self->{LayoutObject}->Block(
                        Name => 'Record',
                        Data => { %Data, %UserInfo, },
                    );

                    # add ticket title
                    if ( $Self->{ConfigObject}->Get('Ticket::Frontend::Title') ) {
                        $Self->{LayoutObject}->Block(
                            Name => 'Title',
                            Data => { %Data, %UserInfo, },
                        );
                    }
                }
                elsif ( $GetParam{ResultForm} eq 'Print' ) {
                    use Kernel::System::PDF;
                    $Self->{PDFObject} = Kernel::System::PDF->new( %{$Self} );
                    if ( $Self->{PDFObject} ) {
                        my %Info = ( %Data, %UserInfo ),
                            my $Created = $Self->{LayoutObject}->Output(
                            Template => '$TimeLong{"$Data{"Created"}"}',
                            Data     => \%Data,
                            );
                        my $Owner = $Self->{LayoutObject}->Output(
                            Template =>
                                '$QData{"Owner","30"} ($Quote{"$Data{"UserFirstname"} $Data{"UserLastname"}","30"})',
                            Data => \%Info
                        );
                        my $Customer = $Self->{LayoutObject}->Output(
                            Template => '$QData{"CustomerID","15"} $QData{"CustomerName","15"}',
                            Data     => \%Data
                        );

                        my @PDFRow;
                        push( @PDFRow, $Data{TicketNumber} );
                        push( @PDFRow, $Created );
                        push( @PDFRow, $Data{From} );
                        push( @PDFRow, $Data{Subject} );
                        push( @PDFRow, $Data{State} );
                        push( @PDFRow, $Data{Queue} );
                        push( @PDFRow, $Owner );
                        push( @PDFRow, $Customer );

                        push( @PDFData, \@PDFRow );
                    }
                    else {

                        # add table block
                        $Self->{LayoutObject}->Block(
                            Name => 'Record',
                            Data => { %Data, %UserInfo, },
                        );
                    }
                }
                elsif ( $GetParam{ResultForm} eq 'CSV' ) {

                    # merge row data
                    my %Info = (
                        %Data, %UserInfo,
                        AccountedTime =>
                            $Self->{TicketObjectSearch}->TicketAccountedTimeGet( TicketID => $_ ),
                    );

                    # csv quote
                    if ( !@CSVHead ) {
                        @CSVHead = @{ $Self->{Config}->{SearchCSVData} };
                    }
                    my @Data = ();
                    for (@CSVHead) {
                        push( @Data, $Info{$_} );
                    }
                    push( @CSVData, \@Data );
                }
                else {

                    # condense down the subject
                    my $Subject = $Self->{TicketObject}->TicketSubjectClean(
                        TicketNumber => $Data{TicketNumber},
                        Subject => $Data{Subject} || '',
                    );

                    # seperate each searchresult line by using several css
                    if ( $Counter % 2 ) {
                        $Data{css} = "searchpassive";
                    }
                    else {
                        $Data{css} = "searchactive";
                    }

                    # add table block
                    $Self->{LayoutObject}->Block(
                        Name => 'Record',
                        Data => {
                            %Data,
                            Subject => $Subject,
                            %UserInfo,
                        },
                    );

                    # Add table Columns
                    for ( qw(Age From Subject Title State Queue CustomerID CustomerName Priority) ) {
                        if ( $GetParam{"ShowTicket$_"} ) {
                            $Self->{LayoutObject}->Block(
                                Name => 'ShowTicket' . $_,
                                Data => { $_ => $Data{$_} }
                            );
                        }
                    }
                    if ( $GetParam{ShowTicketOwner} ) {
                        my $IsTicketLocked = $Self->{TicketObject}->LockIsTicketLocked( TicketID => $_ );
                        $Self->{LayoutObject}->Block(
                            Name => 'ShowTicketOwner',
                            Data => {
                                'tooltip-extra' => ($IsTicketLocked)
                                  ? ''
                                  : $Self->{LayoutObject}->{LanguageObject}->Get('last owner: '),
                                'css-style' => ($IsTicketLocked)
                                  ? ''
                                  : 'style="color:#ccc"',
                                'Owner'         => $Data{Owner},
                                'UserFirstname' => $UserInfo{UserFirstname},
                                'UserLastname'  => $UserInfo{UserLastname},
                            }
                        );
                    }

                    for ( 1 .. 16 ) {
                        if ( $GetParam{ 'ShowFreeText' . $_ } ) {
                            $Self->{LayoutObject}->Block(
                                Name => 'ShowFreeText',
                                Data => {
                                    'TicketFreeText' => $Data{ 'TicketFreeText' . $_ },
                                }
                            );
                        }
                    }
                    for ( 1 .. 16 ) {
                        if ( $GetParam{ 'ShowFreeTime' . $_ } ) {
                            $Self->{LayoutObject}->Block(
                                Name => 'ShowFreeTime',
                                Data => {
                                    'TicketFreeTime' => $Data{ 'TicketFreeTime' . $_ },
                                }
                            );
                        }
                    }
                }
            }
        }
        $Self->{LayoutObject}->Block(
            Name => 'ShowColCount',
            Data => { 'ShowColumnsCount' => $ShowColumnsCount },
        );

        # start html page
        $Param{PageTitle} = $Self->{LayoutObject}->{LanguageObject}->Get('Search Result')
          . ': "' . $Self->{LayoutObject}->{LanguageObject}->Get( $Self->{Profile} ) . '"';
        my $Output = $Self->{LayoutObject}->Header( Title => $Param{PageTitle} );

        $Output .= $Self->{LayoutObject}->NavigationBar();

        # Wird gebraucht um zwischen den SuchSeiten zu springen
        my $SortOptionsURL =
          ( $GetParam{Sort} && $GetParam{Order} )
          ? 'Sort=' . $GetParam{Sort} . '&Order=' . $GetParam{Order} . '&'
          : '';

        # build search navigation bar
        my %PageNav = $Self->{LayoutObject}->PageNavBar(
            Limit     => $Self->{SearchLimit},
            StartHit  => $Self->{StartHit},
            PageShown => $Self->{SearchPageShown},
            AllHits   => $Counter,
            Action    => "Action=AgentTicketNorisSearch&Subaction=Search",
            Link =>
                "Profile=$Self->{Profile}&TakeLastSearch=1&" . $SortOptionsURL,
        );

        # build shown ticket
        if ( $GetParam{ResultForm} eq 'Preview' ) {
            $Output .= $Self->{LayoutObject}->Output(
                TemplateFile => 'AgentTicketNorisSearchResult',
                Data => { %Param, %PageNav, Profile => $Self->{Profile}, },
            );
        }
        elsif ( $GetParam{ResultForm} eq 'Print' ) {

            # PDF Output
            if ( $Self->{PDFObject} ) {
                my $Title = $Self->{LayoutObject}->{LanguageObject}->Get('Ticket') . ' '
                    . $Self->{LayoutObject}->{LanguageObject}->Get('Search');
                my $PrintedBy = $Self->{LayoutObject}->{LanguageObject}->Get('printed by');
                my $Page      = $Self->{LayoutObject}->{LanguageObject}->Get('Page');
                my $Time      = $Self->{LayoutObject}->Output( Template => '$Env{"Time"}' );
                my $Url       = '';
                if ( $ENV{REQUEST_URI} ) {
                    $Url
                        = $Self->{ConfigObject}->Get('HttpType') . '://'
                        . $Self->{ConfigObject}->Get('FQDN')
                        . $ENV{REQUEST_URI};
                }

                # get maximum number of pages
                my $MaxPages = $Self->{ConfigObject}->Get('PDF::MaxPages');
                if ( !$MaxPages || $MaxPages < 1 || $MaxPages > 1000 ) {
                    $MaxPages = 100;
                }

                # create the header
                my $CellData;
                $CellData->[0]->[0]->{Content} = $Self->{ConfigObject}->Get('Ticket::Hook');
                $CellData->[0]->[0]->{Font}    = 'ProportionalBold';
                $CellData->[0]->[1]->{Content}
                    = $Self->{LayoutObject}->{LanguageObject}->Get('Created');
                $CellData->[0]->[1]->{Font} = 'ProportionalBold';
                $CellData->[0]->[2]->{Content}
                    = $Self->{LayoutObject}->{LanguageObject}->Get('From');
                $CellData->[0]->[2]->{Font} = 'ProportionalBold';
                $CellData->[0]->[3]->{Content}
                    = $Self->{LayoutObject}->{LanguageObject}->Get('Subject');
                $CellData->[0]->[3]->{Font} = 'ProportionalBold';
                $CellData->[0]->[4]->{Content}
                    = $Self->{LayoutObject}->{LanguageObject}->Get('State');
                $CellData->[0]->[4]->{Font} = 'ProportionalBold';
                $CellData->[0]->[5]->{Content}
                    = $Self->{LayoutObject}->{LanguageObject}->Get('Queue');
                $CellData->[0]->[5]->{Font} = 'ProportionalBold';
                $CellData->[0]->[6]->{Content}
                    = $Self->{LayoutObject}->{LanguageObject}->Get('Owner');
                $CellData->[0]->[6]->{Font} = 'ProportionalBold';
                $CellData->[0]->[7]->{Content}
                    = $Self->{LayoutObject}->{LanguageObject}->Get('CustomerID');
                $CellData->[0]->[7]->{Font} = 'ProportionalBold';

                # create the content array
                my $CounterRow = 1;
                for my $Row (@PDFData) {
                    my $CounterColumn = 0;
                    for my $Content ( @{$Row} ) {
                        $CellData->[$CounterRow]->[$CounterColumn]->{Content} = $Content;
                        $CounterColumn++;
                    }
                    $CounterRow++;
                }

                # output 'No Result', if no content was given
                if ( !$CellData->[0]->[0] ) {
                    $CellData->[0]->[0]->{Content}
                        = $Self->{LayoutObject}->{LanguageObject}->Get('No Result!');
                }

                # page params
                my %PageParam;
                $PageParam{PageOrientation} = 'landscape';
                $PageParam{MarginTop}       = 30;
                $PageParam{MarginRight}     = 40;
                $PageParam{MarginBottom}    = 40;
                $PageParam{MarginLeft}      = 40;
                $PageParam{HeaderRight}     = $Title;
                $PageParam{FooterLeft}      = $Url;
                $PageParam{HeadlineLeft}    = $Title;
                $PageParam{HeadlineRight}   = $PrintedBy . ' '
                    . $Self->{UserFirstname} . ' '
                    . $Self->{UserLastname} . ' ('
                    . $Self->{UserEmail} . ') '
                    . $Time;

                # table params
                my %TableParam;
                $TableParam{CellData}            = $CellData;
                $TableParam{Type}                = 'Cut';
                $TableParam{FontSize}            = 6;
                $TableParam{Border}              = 0;
                $TableParam{BackgroundColorEven} = '#AAAAAA';
                $TableParam{BackgroundColorOdd}  = '#DDDDDD';
                $TableParam{Padding}             = 1;
                $TableParam{PaddingTop}          = 3;
                $TableParam{PaddingBottom}       = 3;

                # create new pdf document
                $Self->{PDFObject}->DocumentNew(
                    Title  => $Self->{ConfigObject}->Get('Product') . ': ' . $Title,
                    Encode => $Self->{LayoutObject}->{UserCharset},
                );

                # start table output
                $Self->{PDFObject}->PageNew( %PageParam, FooterRight => $Page . ' 1', );
                for ( 2 .. $MaxPages ) {

                    # output table (or a fragment of it)
                    %TableParam = $Self->{PDFObject}->Table( %TableParam, );

                    # stop output or another page
                    if ( $TableParam{State} ) {
                        last;
                    }
                    else {
                        $Self->{PDFObject}->PageNew( %PageParam, FooterRight => $Page . ' ' . $_, );
                    }
                }

                # return the pdf document
                my $Filename = 'ticket_search';
                my ( $s, $m, $h, $D, $M, $Y )
                    = $Self->{TimeObject}->SystemTime2Date(
                    SystemTime => $Self->{TimeObject}->SystemTime(),
                    );
                $M = sprintf( "%02d", $M );
                $D = sprintf( "%02d", $D );
                $h = sprintf( "%02d", $h );
                $m = sprintf( "%02d", $m );
                my $PDFString = $Self->{PDFObject}->DocumentOutput();
                return $Self->{LayoutObject}->Attachment(
                    Filename    => $Filename . "_" . "$Y-$M-$D" . "_" . "$h-$m.pdf",
                    ContentType => "application/pdf",
                    Content     => $PDFString,
                    Type        => 'attachment',
                );
            }
            else {
                $Output = $Self->{LayoutObject}->PrintHeader( Width => 800 );
                if ( @ViewableIDs == $Self->{SearchLimit} ) {
                    $Param{Warning} = '$Text{"Reached max. count of %s search hits!", "'
                        . $Self->{SearchLimit} . '"}';
                }
                $Output .= $Self->{LayoutObject}->Output(
                    TemplateFile => 'AgentTicketNorisSearchResultPrint',
                    Data         => \%Param,
                );

                # add footer
                $Output .= $Self->{LayoutObject}->PrintFooter();

                # return output
                return $Output;
            }
        }
        elsif ( $GetParam{ResultForm} eq 'CSV' ) {
            my $CSV = $Self->{LayoutObject}->OutputCSV(
                Head => \@CSVHead,
                Data => \@CSVData,
            );

            # return csv to download
            my $CSVFile = 'ticket_search';
            my ( $s, $m, $h, $D, $M, $Y ) = $Self->{TimeObject}->SystemTime2Date(
                SystemTime => $Self->{TimeObject}->SystemTime(),
            );
            $M = sprintf( "%02d", $M );
            $D = sprintf( "%02d", $D );
            $h = sprintf( "%02d", $h );
            $m = sprintf( "%02d", $m );
            return $Self->{LayoutObject}->Attachment(
                Filename    => $CSVFile . "_" . "$Y-$M-$D" . "_" . "$h-$m.csv",
                ContentType => "text/csv; charset=" . $Self->{LayoutObject}->{UserCharset},
                Content     => $CSV,
            );
        }
        else {
            $Output .= $Self->{LayoutObject}->Output(
                TemplateFile => 'AgentTicketNorisSearchResultShort',
                Data => { %Param, %PageNav, Profile => $Self->{Profile}, },
            );
        }

        # build footer
        $Output .= $Self->{LayoutObject}->Footer();
        return $Output;
    }
    # Noris Fast Search
    elsif ( $Self->{Subaction} eq 'FastSearch' ) {
        my $FSP = $Self->{ParamObject}->GetParam( Param => 'FastSearchPattern' );

        # if ticket number exists
        if ( my $TicketID = $Self->{TicketObject}->TicketCheckNumber( Tn => $FSP ) ) {
   
            # redirect to the ticket
            return $Self->{LayoutObject}->Redirect( OP => "Action=AgentTicketZoom&TicketID=$TicketID" );
        }

        # if not look if there are ticket numbers that look like this
        my @ViewableIDs = $Self->{TicketObjectSearch}->TicketSearch(
            Result       => 'ARRAY',
            Limit        => "2",
            UserID       => $Self->{UserID},
            TicketNumber => $FSP,
        );

        $Self->{Subaction} = 'Search';

        # if Tickets found
        if (@ViewableIDs) {

            # run the 'TicketNumber' Search
            return $Self->Run( %Param, FastSearch => 1, TicketNumber => $FSP, Title => '' );
        }

        # else run a 'Title' Search
        return $Self->Run( %Param, FastSearch => 1, TicketNumber => '', Title => $FSP );
    }
    # empty search site
    else {

        # delete profile
        if ( $Self->{EraseTemplate} && $Self->{Profile} ) {

            # remove old profile stuff
            $Self->{SearchProfileObject}->SearchProfileDelete(
                Base      => 'TicketSearch',
                Name      => $Self->{Profile},
                UserLogin => $Self->{UserLogin},
            );
            %GetParam = ();
            $Self->{Profile} = '';
        }

        # generate search mask
        my $Output = $Self->{LayoutObject}->Header(NoCache => 0, BodyTags => ' onload="search_onload()"');
        my %LockedData = $Self->{TicketObject}->GetLockedCount( UserID => $Self->{UserID} );

        # get free text config options
        my %TicketFreeText = ();
        for ( 1 .. 16 ) {
            $TicketFreeText{"TicketFreeKey$_"} = $Self->{TicketObject}->TicketFreeTextGet(
                Type   => "TicketFreeKey$_",
                FillUp => 0,
                Action => $Self->{Action},
                UserID => $Self->{UserID},
            );
            $TicketFreeText{"TicketFreeText$_"} = $Self->{TicketObject}->TicketFreeTextGet(
                Type   => "TicketFreeText$_",
                FillUp => 0,
                Action => $Self->{Action},
                UserID => $Self->{UserID},
            );
        }
        my %TicketFreeTextHTML = $Self->{LayoutObject}->AgentFreeText(
            NullOption => 1,
            Ticket     => \%GetParam,
            Config     => \%TicketFreeText,
        );
        $Output .= $Self->{LayoutObject}->NavigationBar( LockData => \%LockedData );
        $Output .= $Self->MaskForm( %GetParam, %TicketFreeTextHTML, Profile => $Self->{Profile}, );
        $Output .= $Self->{LayoutObject}->Footer();
        return $Output;
    }
}

sub MaskForm {
    my ( $Self, %Param ) = @_;

    # get user of own groups
    my %ShownUsers = $Self->{UserObject}->UserList(
        Type  => 'Long',
        Valid => 1,
    );
    if ( !$Self->{ConfigObject}->Get('Ticket::ChangeOwnerToEveryone') ) {
        my %Involved = $Self->{GroupObject}->GroupMemberInvolvedList(
            UserID => $Self->{UserID},
            Type   => 'ro',
        );
        for my $UserID ( keys %ShownUsers ) {
            if ( !$Involved{$UserID} ) {
                delete $ShownUsers{$UserID};
            }
        }
    }
    $Param{'UserStrg'} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data               => \%ShownUsers,
        Name               => 'OwnerIDs',
        Multiple           => 1,
        Size               => 5,
        SelectedIDRefArray => $Param{OwnerIDs},
        OnChange           => "clear_checkboxes('usr_chk')",
    );
    $Param{'CreatedUserStrg'} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data               => \%ShownUsers,
        Name               => 'CreatedUserIDs',
        Multiple           => 1,
        Size               => 5,
        SelectedIDRefArray => $Param{CreatedUserIDs},
        OnChange           => "clear_checkboxes('cui_chk')",
    );
    $Param{'JS_me_ID'} = "{'" . $Self->{UserID} . "':1 }";
    $Param{'ResultFormStrg'} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data => {
            Preview => 'Preview',
            Normal  => 'Normal',
            Print   => 'Print',
            CSV     => 'CSV',
        },
        Name => 'ResultForm',
        SelectedID => $Param{ResultForm} || 'Normal',
    );

    my %ProfilesList = $Self->{SearchProfileObject}->SearchProfileList(
        Base      => 'TicketSearch',
        UserLogin => $Self->{UserLogin},
    );

    {
        my @List;
        #$Param{'JS_newstate_IDs'} = "{'" . join( "':1, '",  @List) . "':1}";

        @List = $Self->{StateObject}->StateGetStatesByType(
            StateType => [ 'new', 'open', 'pending over', 'seen', ],
            Result    => 'ID',
        );
        $Param{'JS_openstate_IDs'} = "{'" . join( "':1, '",  @List) . "':1}";

        @List = $Self->{StateObject}->StateGetStatesByType(
            StateType => [ 'closed', ],
            Result    => 'ID',
        );
        $Param{'JS_closedstate_IDs'} = "{'" . join( "':1, '",  @List) . "':1}";

        @List = $Self->{StateObject}->StateGetStatesByType(
            StateType => [ 'pending auto', 'hidden', ],
            Result    => 'ID',
        );
        $Param{'JS_waitstate_IDs'} = "{'" . join( "':1, '",  @List) . "':1}";
    }

    $Param{'StatesStrg'} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data => {
            $Self->{StateObject}->StateList(
                UserID => $Self->{UserID},
                Action => $Self->{Action},
            ),
        },
        Name               => 'StateIDs',
        Multiple           => 1,
        Size               => 10,
        SelectedIDRefArray => $Param{StateIDs},
        OnChange           => "clear_checkboxes('sid_chk')",
    );
    my %AllQueues = $Self->{QueueObject}->GetAllQueues(
        UserID => $Self->{UserID},
        Type   => 'ro',
    );

    my @UserQueues = $Self->{QueueObject}->GetAllCustomQueues( UserID => $Self->{UserID}) ;
    if ( ! defined $Param{QueueIDs}  && $Self->{Subaction} ne 'LoadProfile' ) {
        $Param{QueueIDs} = \@UserQueues;
    }

    $Param{'JS_myQueue_IDs'} = "{'" . join( "':1, '", @UserQueues ) . "':1}";
    $Param{'QueuesStrg'} = $Self->{LayoutObject}->AgentQueueListOption(
        Data               => \%AllQueues,
        Size               => 10,
        Multiple           => 1,
        Name               => 'QueueIDs',
        SelectedIDRefArray => $Param{QueueIDs},
        OnChangeSubmit     => 0,
        OnChange           => "clear_checkboxes('qid_chk')",
    );
    $Param{'CreatedQueuesStrg'} = $Self->{LayoutObject}->AgentQueueListOption(
        Data               => \%AllQueues,
        Size               => 5,
        Multiple           => 1,
        Name               => 'CreatedQueueIDs',
        SelectedIDRefArray => $Param{CreatedQueueIDs},
        OnChangeSubmit     => 0,
    );
    $Param{'PrioritiesStrg'} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data => {
            $Self->{PriorityObject}->PriorityList(
                UserID => $Self->{UserID},
                Action => $Self->{Action},
            ),
        },
        Name               => 'PriorityIDs',
        Multiple           => 1,
        Size               => 10,
        SelectedIDRefArray => $Param{PriorityIDs},
        OnChange           => "clear_checkboxes('pid_chk')",
    );
    $Param{'JS_00_09_IDs'} = "{'" . join( "':1, '", ( 91 .. 100 ) ) . "':1}";
    $Param{'JS_10_19_IDs'} = "{'" . join( "':1, '", ( 81 .. 90 ) ) . "':1}";
    $Param{'JS_20_29_IDs'} = "{'" . join( "':1, '", ( 71 .. 80 ) ) . "':1}";
    $Param{'JS_30_39_IDs'} = "{'" . join( "':1, '", ( 61 .. 70 ) ) . "':1}";
    $Param{'JS_40_99_IDs'} = "{'" . join( "':1, '", ( 01 .. 60 ) ) . "':1}";

    for my $Key (qw(Create Close Escalation)) {
        $Param{"Ticket${Key}TimePoint"} = $Self->{LayoutObject}->OptionStrgHashRef(
            Data => {
                1  => ' 1',
                2  => ' 2',
                3  => ' 3',
                4  => ' 4',
                5  => ' 5',
                6  => ' 6',
                7  => ' 7',
                8  => ' 8',
                9  => ' 9',
                10 => '10',
                11 => '11',
                12 => '12',
                13 => '13',
                14 => '14',
                15 => '15',
                16 => '16',
                17 => '17',
                18 => '18',
                19 => '19',
                20 => '20',
                21 => '21',
                22 => '22',
                23 => '23',
                24 => '24',
                25 => '25',
                26 => '26',
                27 => '27',
                28 => '28',
                29 => '29',
                30 => '30',
                31 => '31',
                32 => '32',
                33 => '33',
                34 => '34',
                35 => '35',
                36 => '36',
                37 => '37',
                38 => '38',
                39 => '39',
                40 => '40',
                41 => '41',
                42 => '42',
                43 => '43',
                44 => '44',
                45 => '45',
                46 => '46',
                47 => '47',
                48 => '48',
                49 => '49',
                50 => '50',
                51 => '51',
                52 => '52',
                53 => '53',
                54 => '54',
                55 => '55',
                56 => '56',
                57 => '57',
                58 => '58',
                59 => '59',
            },
            Name       => "Ticket${Key}TimePoint",
            SelectedID => $Param{"Ticket${Key}TimePoint"},
        );
        $Param{"Ticket${Key}TimePointFormat"} = $Self->{LayoutObject}->OptionStrgHashRef(
            Data => {
                minute => 'minute(s)',
                hour   => 'hour(s)',
                day    => 'day(s)',
                week   => 'week(s)',
                month  => 'month(s)',
                year   => 'year(s)',
            },
            Name       => "Ticket${Key}TimePointFormat",
            SelectedID => $Param{"Ticket${Key}TimePointFormat"},
        );
    }

    # Create n Close
    for my $Key (qw(Create Close)) {
        $Param{"Ticket${Key}TimePointStart"} = $Self->{LayoutObject}->OptionStrgHashRef(
            Data => {
                'Last'   => 'last',
                'Before' => 'before',
            },
            Name => "Ticket${Key}TimePointStart",
            SelectedID => $Param{"Ticket${Key}TimePointStart"} || 'Last',
        );
        $Param{"Ticket${Key}TimeStart"} = $Self->{LayoutObject}->BuildDateSelection(
            %Param,
            Prefix   => "Ticket${Key}TimeStart",
            Format   => 'DateInputFormat',
            DiffTime => -( ( 60 * 60 * 24 ) * 30 ),
        );
        $Param{"Ticket${Key}TimeStop"} = $Self->{LayoutObject}->BuildDateSelection(
            %Param,
            Prefix => "Ticket${Key}TimeStop",
            Format => 'DateInputFormat',
        );
    }

    # Escalation
    $Param{"TicketEscalationTimePointStart"} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data => {
            'Last'   => 'last',
            'Before' => 'before',
            'Next'   => 'next',
            'After' => 'after',
        },
        Name => "TicketEscalationTimePointStart",
        SelectedID => $Param{"TicketEscalationTimePointStart"} || 'Next',
    );
    $Param{"TicketEscalationTimeStart"} = $Self->{LayoutObject}->BuildDateSelection(
        %Param,
        Prefix   => "TicketEscalationTimeStart",
        Format   => 'DateInputFormat',
    );
    $Param{"TicketEscalationTimeStop"} = $Self->{LayoutObject}->BuildDateSelection(
        %Param,
        Prefix => "TicketEscalationTimeStop",
        Format => 'DateInputFormat',
        DiffTime => ( 60 * 60 * 24 ) * 30,
    );

    for ( qw(TicketFreeTime1 TicketFreeTime2 TicketFreeTime3
             TicketFreeTime4 TicketFreeTime5 TicketFreeTime6
            ) ) {
        $Param{ $_ . 'Start' } = $Self->{LayoutObject}->BuildDateSelection(
            %Param,
            Prefix   => $_ . 'Start',
            Format   => 'DateInputFormat',
            DiffTime => -( ( 60 * 60 * 24 ) * 30 ),
        );
        $Param{ $_ . 'Stop' } = $Self->{LayoutObject}->BuildDateSelection(
            %Param,
            Prefix   => $_ . 'Stop',
            Format   => 'DateInputFormat',
            DiffTime => +( ( 60 * 60 * 24 ) * 30 ),
        );
    }

    my %ExtraOptions = (
        ( $Self->{Subaction} eq 'LoadProfile' && $Param{Profile} )
        ? (
            BoxTitle => $Self->{LayoutObject}->{LanguageObject}->Get('Ticket Search'),
            SaveProfileRadioOptions => 'checked="checked"',
          )
        : (
            BoxTitle => $Self->{LayoutObject}->{LanguageObject}->Get('Search-Template'),
            SaveProfileRadioOptions => '',
        )
    );
    # html search mask output
    $Self->{LayoutObject}->Block(
        Name => 'Search',
        Data => {
            %Param,
            %ExtraOptions,
          },
    );

    # add ticket title
    if ( $Self->{ConfigObject}->Get('Ticket::Frontend::Title') ) {
        $Self->{LayoutObject}->Block(
            Name => 'TicketTitle',
            Data => {%Param},
        );
    }

    # build type string
    if ( $Self->{ConfigObject}->Get('Ticket::Type') ) {
        my %Type = $Self->{TypeObject}->TypeList( UserID => $Self->{UserID}, );
        $Param{'TypesStrg'} = $Self->{LayoutObject}->BuildSelection(
            Data        => \%Type,
            Name        => 'TypeIDs',
            SelectedID  => $Param{TypeIDs},
            Sort        => 'AlphanumericValue',
            Size        => 10,
            Multiple    => 1,
            Translation => 0,
        );
        $Self->{LayoutObject}->Block(
            Name => 'TicketType',
            Data => {%Param},
        );
    }

    # build service string
    if ( $Self->{ConfigObject}->Get('Ticket::Service') ) {

        # get list type
        my $TreeView = 0;
        if ( $Self->{ConfigObject}->Get('Ticket::Frontend::ListType') eq 'tree' ) {
            $TreeView = 1;
        }
        my %Service = $Self->{ServiceObject}->ServiceList( UserID => $Self->{UserID}, );
        $Param{'ServicesStrg'} = $Self->{LayoutObject}->BuildSelection(
            Data        => \%Service,
            Name        => 'ServiceIDs',
            SelectedID  => $Param{ServiceIDs},
            TreeView    => $TreeView,
            Sort        => 'TreeView',
            Size        => 5,
            Multiple    => 1,
            Translation => 0,
            Max         => 200,
        );
        my %SLA = $Self->{SLAObject}->SLAList( UserID => $Self->{UserID}, );
        $Param{'SLAsStrg'} = $Self->{LayoutObject}->BuildSelection(
            Data        => \%SLA,
            Name        => 'SLAIDs',
            SelectedID  => $Param{SLAIDs},
            Sort        => 'AlphanumericValue',
            Size        => 5,
            Multiple    => 1,
            Translation => 0,
            Max         => 200,
        );
        $Self->{LayoutObject}->Block(
            Name => 'TicketService',
            Data => {%Param},
        );
    }
    if (
        $Self->{ConfigObject}->Get('Ticket::Watcher')
        || $Self->{ConfigObject}->Get('Ticket::Responsible')
        )
    {
        $Self->{LayoutObject}->Block( Name => 'TicketResponsibleWatcher', );
        if ( $Self->{ConfigObject}->Get('Ticket::Watcher') ) {
            $Self->{LayoutObject}->Block(
                Name => 'TicketResponsibleWatcherHeaderOn',
                Data => { Headline => 'Watcher', },
            );
            my $SelectStrg = $Self->{LayoutObject}->BuildSelection(
                Data        => \%ShownUsers,
                Name        => 'WatchUserIDs',
                SelectedID  => $Param{WatchUserIDs},
                Sort        => 'AlphanumericValue',
                Size        => 5,
                Multiple    => 1,
                Translation => 0,
            );
            $Self->{LayoutObject}->Block(
                Name => 'TicketResponsibleWatcherBodyOn',
                Data => { %Param, SelectStrg => $SelectStrg, },
            );
        }
        else {
            $Self->{LayoutObject}->Block( Name => 'TicketResponsibleWatcherHeaderOff', );
            $Self->{LayoutObject}->Block( Name => 'TicketResponsibleWatcherBodyOff', );
        }
        if ( $Self->{ConfigObject}->Get('Ticket::Responsible') ) {
            $Self->{LayoutObject}->Block(
                Name => 'TicketResponsibleWatcherHeaderOn',
                Data => { Headline => 'Responsible', },
            );
            my $SelectStrg = $Self->{LayoutObject}->BuildSelection(
                Data        => \%ShownUsers,
                Name        => 'ResponsibleIDs',
                SelectedID  => $Param{ResponsibleIDs},
                Sort        => 'AlphanumericValue',
                Size        => 5,
                Multiple    => 1,
                Translation => 0,
            );
            $Self->{LayoutObject}->Block(
                Name => 'TicketResponsibleWatcherBodyOn',
                Data => { %Param, SelectStrg => $SelectStrg, },
            );
        }
        else {
            $Self->{LayoutObject}->Block( Name => 'TicketResponsibleWatcherHeaderOff', );
            $Self->{LayoutObject}->Block( Name => 'TicketResponsibleWatcherBodyOff', );
        }
    }
    for my $Count ( 1 .. 16 ) {
        if ( $Self->{Config}->{'TicketFreeText'}->{$Count} ) {
            $Self->{LayoutObject}->Block(
                Name => 'TicketFreeText',
                Data => {
                    TicketFreeKeyField  => $Param{ 'TicketFreeKeyField' . $Count },
                    TicketFreeTextField => $Param{ 'TicketFreeTextField' . $Count },
                    Count               => $Count,
                },
            );
            $Self->{LayoutObject}->Block(
                Name => 'TicketFreeText' . $Count,
                Data => { %Param, },
            );
        }
    }
    for my $Count ( 1 .. 6 ) {
        if ( $Self->{Config}->{'TicketFreeTime'}->{$Count} ) {
            $Self->{LayoutObject}->Block(
                Name => 'TicketFreeTime',
                Data => {
                    TicketFreeTimeKey => $Self->{ConfigObject}->Get( 'TicketFreeTimeKey' . $Count ),
                    TicketFreeTime    => $Param{ 'TicketFreeTime' . $Count },
                    TicketFreeTimeStart => $Param{ 'TicketFreeTime' . $Count . 'Start' },
                    TicketFreeTimeStop  => $Param{ 'TicketFreeTime' . $Count . 'Stop' },
                    Count               => $Count,
                },
            );
            $Self->{LayoutObject}->Block(
                Name => 'TicketFreeTime' . $Count,
                Data => { %Param, Count => $Count, },
            );
        }
    }

    # Show Columns
    my $ColConfig = $Self->{ConfigObject}->Get('Ticket::Frontend::AgentTicketNorisSearch')->{'ShowColumns'};

    foreach my $RKey ( @{$ColConfig} ) {
        $Self->{LayoutObject}->Block( Name => 'ShowCol', );
        foreach my $CKey ( @{$RKey} ) {
            my $IS_Checked = $Param{ $CKey->{Var} };
            $IS_Checked = $CKey->{Checked} unless defined $IS_Checked;

            $Self->{LayoutObject}->Block(
                Name => 'ShowColField',
                Data => {
                    Var     => $CKey->{Var},
                    Text    => $CKey->{Text},
                    Checked => ( $IS_Checked )
                      ? ' checked="checked" '
                      : '',
                }
            );
        }
    }
    {   # Sorting Options
        my $SortOptions =$Self->{ConfigObject}->Get("NorisSearch::SortOptions");
        for ( 1 .. 3 ) {
            my $SortByStrg = $Self->{LayoutObject}->BuildSelection(
                Data         => $SortOptions,
                Name         => 'SortBy' . $_,
                SelectedID => ( $Self->{SortBy}[ $_ - 1 ]
                    ? $Self->{SortBy}[ $_ - 1 ]
                    : '' ),
                Title        => $Self->{LayoutObject}->{LanguageObject}->Get($_ . ". Step"),
                PossibleNone => 1,
            );
            my $OrderStrg = $Self->{LayoutObject}->BuildSelection(
                Data       => [ 'Up', 'Down', ],
                Name       => 'Order' . $_,
                SelectedID => ( $Self->{Order}[ $_ - 1 ]
                    ? $Self->{Order}[ $_ - 1 ]
                    : '' ),
                Title      => 'Order',
            );
            $Self->{LayoutObject}->Block(
                Name => 'SortOptions',
                Data => {
                    SortByStrg => $SortByStrg,
                    OrderStrg  => $OrderStrg,
                    Count      => $_,
                }
            );

        }
    }

    my $Output = $Self->{LayoutObject}->Output(
        TemplateFile => 'AgentTicketNorisSearch',
        Data         => \%Param,
    );
    return $Output;
}

1;
