# --
# Kernel/Modules/AgentTicketComposeInternal.pm - compose an internal email reply
# Copyright (C) 2001-2008 OTRS AG, http://otrs.org/
# Copyright (C) 2008 noris network AG
# --
# This software comes with ABSOLUTELY NO WARRANTY. For details, see
# the enclosed file COPYING for license information (AGPL). If you
# did not receive this file, see http://www.gnu.org/licenses/agpl.txt.
# --

package Kernel::Modules::AgentTicketComposeInternal;

use strict;
use warnings;

use Kernel::System::CheckItem;
use Kernel::System::State;
use Kernel::System::CustomerUser;
use Kernel::System::Web::UploadCache;
use Kernel::System::SystemAddress;
use Mail::Address;

use vars qw($VERSION);
$VERSION = qw($Revision: 0.0.1 $) [1];

sub new {
    my ( $Type, %Param ) = @_;

    # allocate new hash for object
    my $Self = {%Param};
    bless( $Self, $Type );

    $Self->{Debug} = $Param{Debug} || 0;

#    # get common opjects
#    foreach (keys %Param) {
#        $Self->{$_} = $Param{$_};
#    }
#
    # check all needed objects
    for (qw(TicketObject ParamObject DBObject QueueObject LayoutObject ConfigObject LogObject)) {
        if ( !$Self->{$_} ) {
            $Self->{LayoutObject}->FatalError( Message => "Got no $_!" );
        }
    }

    # some new objects
    $Self->{CustomerUserObject}  = Kernel::System::CustomerUser->new(%Param);
    $Self->{CheckItemObject}     = Kernel::System::CheckItem->new(%Param);
    $Self->{StateObject}         = Kernel::System::State->new(%Param);
    $Self->{UploadCachObject}    = Kernel::System::Web::UploadCache->new(%Param);
    $Self->{SystemAddress}       = Kernel::System::SystemAddress->new(%Param);

    # get params
    foreach (qw(From To Cc Bcc Reply-To Subject Body InReplyTo ResponseID ReplyArticleID References ComposeStateID ArticleTypeID
        NewPriorityID
        ArticleID TimeUnits Year Month Day Hour Minute AttachmentUpload
        AttachmentDelete1 AttachmentDelete2 AttachmentDelete3 AttachmentDelete4
        AttachmentDelete5 AttachmentDelete6 AttachmentDelete7 AttachmentDelete8
        AttachmentDelete9 AttachmentDelete10 AttachmentDelete11 AttachmentDelete12
        AttachmentDelete13 AttachmentDelete14 AttachmentDelete15 AttachmentDelete16
        FormID)
    ) {
        my $Value = $Self->{ParamObject}->GetParam(Param => $_);
        if (defined($Value)) {
            $Self->{GetParam}->{$_} = $Value;
        }
    }
    
    # create form id
    if ( !$Self->{GetParam}->{FormID} ) {
        $Self->{GetParam}->{FormID} = $Self->{UploadCachObject}->FormIDCreate();
    }
    # Is the current user owner of the ticket?
    $Self->{isOwner} = $Self->{TicketObject}->OwnerCheck(
                    TicketID => $Self->{TicketID},
                    OwnerID => $Self->{UserID},
    );


    $Self->{Config} = $Self->{ConfigObject}->Get("Ticket::Frontend::$Self->{Action}");
    $Self->{InternalEmailDomainRegExp} = $Self->{ConfigObject}->Get('InternalEmailDomainRegExp');

    return $Self;
}

sub Run {
    my ( $Self, %Param ) = @_;

    # check needed stuff
    if ( !$Self->{TicketID} ) {
        return $Self->{LayoutObject}->ErrorScreen(
            Message => 'No TicketID is given!',
            Comment => 'Please contact the admin.',
        );
    }

    my $Output;

    my %Ticket = $Self->{TicketObject}->TicketGet( TicketID => $Self->{TicketID} );
    my $TicketFreeTextOptions = $Self->{QueueObject}->GetFreeTextOptions(
        Config => $Self->{Config}->{TicketFreeText},
        Queue  => $Ticket{Queue},
        Type   => $Ticket{Type},
    );
    my $TicketFreeTimeOptions = $Self->{QueueObject}->GetFreeTimeOptions(
        Config => $Self->{Config}->{TicketFreeTime},
        Queue  => $Ticket{Queue},
        Type   => $Ticket{Type},
    );

    if ($Self->{Subaction} eq 'SendEmail') {
        $Output = $Self->SendEmail(TicketFreeTextOptions => $TicketFreeTextOptions, TicketFreeTimeOptions => $TicketFreeTimeOptions);
    }
    else {
        $Output = $Self->Form(
            TicketFreeTextOptions => $TicketFreeTextOptions,
            TicketFreeTimeOptions => $TicketFreeTimeOptions,
            QueueID               => $Ticket{QueueID},
        );
    }
    return $Output;
}

sub Form {
    my $Self = shift;
    my %Param = @_;
    my $Output;
    my %Error = ();
    my %GetParam = %{$Self->{GetParam}};

    my $TicketFreeTextOptions = $Param{TicketFreeTextOptions};
    my $TicketFreeTimeOptions = $Param{TicketFreeTimeOptions};
    # check needed stuff
    if (!$Self->{TicketID}) {
        return $Self->{LayoutObject}->ErrorScreen(
            Message => "Got no TicketID!",
            Comment => 'System Error!',
        );
    }
    # get ticket data
    my %Ticket = $Self->{TicketObject}->TicketGet(TicketID => $Self->{TicketID});

    # check permissions
    if (!$Self->{TicketObject}->Permission(
        Type => $Self->{Config}->{Permission},
        TicketID => $Self->{TicketID},
        UserID => $Self->{UserID})) {
        # error screen, don't show ticket
        return $Self->{LayoutObject}->NoPermission(WithHeader => 'yes');
    }

    # get lock state
    if ($Self->{Config}->{RequiredLock}) {
        # first check if the User has Queue-rw-Permissions
        my %CheckQueues = $Self->{QueueObject}->GetAllQueues(
            UserID => $Self->{UserID},
            Type   => 'rw'
        );
        if (!$Self->{TicketObject}->LockIsTicketLocked(TicketID => $Self->{TicketID})
             && defined $CheckQueues{$Param{QueueID}} ) {
            # set owner
            $Self->{TicketObject}->OwnerSet(
                TicketID => $Self->{TicketID},
                UserID => $Self->{UserID},
                NewUserID => $Self->{UserID},
            );
            # set lock
            if ($Self->{TicketObject}->LockSet(
                TicketID => $Self->{TicketID},
                Lock => 'lock',
                UserID => $Self->{UserID}
            )) {
                # show lock state
                $Self->{LayoutObject}->Block(
                    Name => 'PropertiesLock',
                    Data => {
                        %Param,
                        TicketID => $Self->{TicketID},
                    },
                );
            }
        }
        else {
            $Self->{LayoutObject}->Block(
                Name => 'TicketBack',
                Data => {
                    %Param,
                    TicketID => $Self->{TicketID},
                },
            )
        }
    }
    # get last customer article or selecte article ...
    my %Data = ();
    if ($GetParam{ArticleID}) {
        %Data = $Self->{TicketObject}->ArticleGet(
            ArticleID => $GetParam{ArticleID},
        );
    }
    else {
        %Data = $Self->{TicketObject}->ArticleLastCustomerArticle(
            TicketID => $Self->{TicketID},
        );
    }
    # add attachmens to upload cache
    my %AttachmentIndex = $Self->{TicketObject}->ArticleAttachmentIndex(
        %Data,
    );
    foreach (keys %AttachmentIndex) {
        my %Attachment = $Self->{TicketObject}->ArticleAttachment(
            ArticleID => $Data{ArticleID},
            FileID => $_,
        );
        $Self->{UploadCachObject}->FormIDAddFile(
            FormID => $GetParam{FormID},
            %Attachment,
        );
    }
    # get all attachments meta data
    my @Attachments = $Self->{UploadCachObject}->FormIDGetAllFilesMeta(
        FormID => $GetParam{FormID},
    );

    # current user
    my %User = $Self->{UserObject}->GetUserData(
        UserID => $Self->{UserID},
        Cached => 1,
    );    
    # get customer data
    my %Customer = ();
    if ($Ticket{CustomerUserID}) {
        %Customer = $Self->{CustomerUserObject}->CustomerUserDataGet(
            User => $Ticket{CustomerUserID},
        );
    }
    # prepare salutation
    $Data{Salutation} = $Self->{QueueObject}->GetSalutation(%Ticket);
    # prepare signature
    $Data{Signature} = $Self->{QueueObject}->GetSignature(%Ticket);
    foreach (qw(Signature Salutation)) {
        # get and prepare realname
        if ($Data{$_} =~ /<OTRS_CUSTOMER_REALNAME>/) {
            my $From = '';
            if ($Ticket{CustomerUserID}) {
                $From = $Self->{CustomerUserObject}->CustomerName(UserLogin => $Ticket{CustomerUserID});
            }
            if (!$From) {
                $From = $Data{OrigFrom} || '';
                $From =~ s/<.*>|\(.*\)|\"|;|,//g;
                $From =~ s/( $)|(  $)//g;
            }
            $Data{$_} =~ s/<OTRS_CUSTOMER_REALNAME>/$From/g;
        }
        foreach my $UserKey (keys %User) {
            if ($User{$UserKey}) {
                $Data{$_} =~ s/<OTRS_Agent_$UserKey>/$User{$UserKey}/gi;
                $Data{$_} =~ s/<OTRS_CURRENT_$UserKey>/$User{$UserKey}/gi;
            }
        }
        # replace other needed stuff
        $Data{$_} =~ s/<OTRS_FIRST_NAME>/$Self->{UserFirstname}/g;
        $Data{$_} =~ s/<OTRS_LAST_NAME>/$Self->{UserLastname}/g;
        # cleanup
        $Data{$_} =~ s/<OTRS_Agent_.+?>/-/gi;
        $Data{$_} =~ s/<OTRS_CURRENT_.+?>/-/gi;

        # owner user
        my %OwnerUser = $Self->{UserObject}->GetUserData(
            UserID => $Ticket{OwnerID},
            Cached => 1,
        );
        foreach my $UserKey (keys %OwnerUser) {
            if ($OwnerUser{$UserKey}) {
                $Data{$_} =~ s/<OTRS_OWNER_$UserKey>/$OwnerUser{$UserKey}/gi;
            }
        }
        # cleanup
        $Data{$_} =~ s/<OTRS_OWNER_.+?>/-/gi;

        # responsible user
        my %ResponsibleUser = $Self->{UserObject}->GetUserData(
            UserID => $Ticket{ResponsibleID},
            Cached => 1,
        );
        foreach my $UserKey (keys %ResponsibleUser) {
            if ($ResponsibleUser{$UserKey}) {
                $Data{$_} =~ s/<OTRS_RESPONSIBLE_$UserKey>/$ResponsibleUser{$UserKey}/gi;
            }
        }
        # cleanup
        $Data{$_} =~ s/<OTRS_RESPONSIBLE_.+?>/-/gi;

        # replace ticket data
        foreach my $TicketKey (keys %Ticket) {
            if ($Ticket{$TicketKey}) {
                $Data{$_} =~ s/<OTRS_TICKET_$TicketKey>/$Ticket{$TicketKey}/gi;
            }
        }
        # cleanup all not needed <OTRS_TICKET_ tags
        $Data{$_} =~ s/<OTRS_TICKET_.+?>/-/gi;
        # replace customer data
        foreach my $CustomerKey (keys %Customer) {
            if ($Customer{$CustomerKey}) {
                $Data{$_} =~ s/<OTRS_CUSTOMER_$CustomerKey>/$Customer{$CustomerKey}/gi;
                $Data{$_} =~ s/<OTRS_CUSTOMER_DATA_$CustomerKey>/$Customer{$CustomerKey}/gi;
            }
        }
        # cleanup all not needed <OTRS_CUSTOMER_ tags
        $Data{$_} =~ s/<OTRS_CUSTOMER_.+?>/-/gi;
        $Data{$_} =~ s/<OTRS_CUSTOMER_DATA.+?>/-/gi;
        # replace config options
        $Data{$_} =~ s{<OTRS_CONFIG_(.+?)>}{$Self->{ConfigObject}->Get($1)}egx;
        $Data{$_} =~ s/<OTRS_CONFIG_.+?>/-/gi;
    }
    # check if original content isn't text/plain or text/html, don't use it
    if ($Data{'ContentType'}) {
        if($Data{'ContentType'} =~ /text\/html/i) {
            $Data{Body} =~ s/\<.+?\>//gs;
        }
        elsif ($Data{'ContentType'} !~ /text\/plain/i) {
            $Data{Body} = "-> no quotable message <-";
        }
    }
    # prepare body, subject, Reply-To ...
    my $NewLine = $Self->{ConfigObject}->Get('Ticket::Frontend::TextAreaEmail') || 75;
    $Data{Body} =~ s/(^>.+|.{4,$NewLine})(?:\s|\z)/$1\n/gm;
    $Data{Body} =~ s/\t/ /g;
    my $Quote = $Self->{ConfigObject}->Get('Ticket::Frontend::Quote');
    if ($Quote) {
        $Data{Body} =~ s/\n/\n$Quote /g;
        $Data{Body} = "\n$Quote " . $Data{Body};
    }
    else {
        $Data{Body} = "\n".$Data{Body};
    }
    $Data{Body} = "Hi!\n\n$Data{From} schrieb am $Data{Created}:$Data{Body}";
    $Data{Subject} = $Self->{TicketObject}->TicketSubjectBuild(
        TicketNumber => $Ticket{TicketNumber},
        Subject => $Data{Subject} || '',
    );
    
    my $TicketAddress = $Self->{TicketObject}->TicketEnvelopeSender( TicketID => $Ticket{TicketID} );
    my %Recipients = ();
    $Recipients{To} = [Mail::Address->parse($Data{ReplyTo} || $Data{From})];
    push @{$Recipients{To}}, Mail::Address->parse($Data{To});
    $Recipients{Cc} = $Self->{ConfigObject}->Get('Ticket::Frontend::ComposeExcludeCcRecipients') 
        ? [] : [Mail::Address->parse($Data{Cc})];
        
    
    $Data{From} = "$User{UserFirstname} $User{UserLastname} ($Ticket{Queue}) <$TicketAddress>";
    $Data{Email} = $User{UserEmail};
    $Data{RealName} = "$User{UserFirstname} $User{UserLastname}";
    
    # Remove queue names, duplicates and addresses referring 
    # to this particular queue
    # At the end, all addresses for real use are in %FinalRecipients
    my %Seen = ();
    $Seen{lc $TicketAddress} = 1;
    my $TicketRTAddress = $Self->{TicketObject}->TicketRTEnvelopeSender( TicketID => $Ticket{TicketID} );
    $Seen{lc $TicketRTAddress} = 1 if defined $TicketRTAddress;

    if ($Self->{TicketObject}->OwnerCheck(
            TicketID => $Ticket{TicketID},
            OwnerID => $Self->{UserID}
        )
    ) {
        $Seen{lc $User{UserEmail}} = 1;
        $Seen{lc $User{UserLogin}} = 1;
    }
    my %FinalRecipients = ( Cc => [], To => [] );

    for my $Type (qw(To Cc)) {
        for my $Email (@{$Recipients{$Type}}) {
            my $address = lc $Email->address();
            if (! (! $Email->host() && $Self->{QueueObject}->QueueLookup(
                        Queue => $address))
                && ! exists $Seen{$address}
                && ! $Self->{SystemAddress}->SystemAddressIsLocalAddress(
                        Address => $address) 
            ) {
                if ((! defined $Self->{InternalEmailDomainRegExp})
                    || ! $Email->host()
                    || $Email->host() =~ $Self->{InternalEmailDomainRegExp}) {
                    push @{$FinalRecipients{$Type}}, $Email;
                }
                else {
                    $Self->{LayoutObject}->Block(
                        Name => 'RemovedAddress',
                        Data => { Address => $Email->address() },
                    );
                }
                $Seen{$address} = 1;
            }
        }
    }
    
    # Create the recipient headers
    for my $Type (qw(To Cc)) {
        $Data{$Type} = join ', ', (map $_->format(), @{$FinalRecipients{$Type}});
    }
        
    # run compose modules
    if (ref($Self->{ConfigObject}->Get('Ticket::Frontend::ArticleComposeModule')) eq 'HASH') {
        my %Jobs = %{$Self->{ConfigObject}->Get('Ticket::Frontend::ArticleComposeModule')};
        foreach my $Job (sort keys %Jobs) {
            # load module
            if ($Self->{MainObject}->Require($Jobs{$Job}->{Module})) {
                my $Object = $Jobs{$Job}->{Module}->new(
                    %{$Self},
                    Debug => $Self->{Debug},
                );
                # get params
                foreach ($Object->Option(%Data, %GetParam, Config => $Jobs{$Job})) {
                    $GetParam{$_} = $Self->{ParamObject}->GetParam(Param => $_);
                }
                # run module
                $Object->Run(%Data, %GetParam, Config => $Jobs{$Job});
                # get errors
                %Error = (%Error, $Object->Error(%GetParam, Config => $Jobs{$Job}));
            }
            else {
                return $Self->{LayoutObject}->FatalError();
            }
        }
    }
    # get free text config options
    my %TicketFreeText = ();
    foreach (1..16) {
        $TicketFreeText{"TicketFreeKey$_"} = $Self->{TicketObject}->TicketFreeTextGet(
            TicketID => $Self->{TicketID},
            Type => "TicketFreeKey$_",
            Action => $Self->{Action},
            UserID => $Self->{UserID},
        );
        $TicketFreeText{"TicketFreeText$_"} = $Self->{TicketObject}->TicketFreeTextGet(
            TicketID => $Self->{TicketID},
            Type => "TicketFreeText$_",
            Action => $Self->{Action},
            UserID => $Self->{UserID},
        );
    }
    my %TicketFreeTextHTML = $Self->{LayoutObject}->AgentFreeText(
        Ticket => \%Ticket,
        Config => \%TicketFreeText,
    );
    # get ticket free time params
    foreach (1..6) {
        if (!$Self->{ConfigObject}->Get('TicketFreeTimeOptional'.$_)) {
            $GetParam{'TicketFreeTime'.$_.'Used'} = 1;
        }
    }
    # free time
    my %TicketFreeTime = ();
    foreach (1..6) {
        $TicketFreeTime{"TicketFreeTime".$_.'Optional'} = 1;
        $TicketFreeTime{"TicketFreeTime".$_.'Used'} = $GetParam{'TicketFreeTime'.$_.'Used'};

        if ($Ticket{"TicketFreeTime".$_}) {
            (
                $TicketFreeTime{"TicketFreeTime".$_.'Secunde'},
                $TicketFreeTime{"TicketFreeTime".$_.'Minute'},
                $TicketFreeTime{"TicketFreeTime".$_.'Hour'},
                $TicketFreeTime{"TicketFreeTime".$_.'Day'},
                $TicketFreeTime{"TicketFreeTime".$_.'Month'},
                $TicketFreeTime{"TicketFreeTime".$_.'Year'}
            ) = $Self->{TimeObject}->SystemTime2Date(
                SystemTime => $Self->{TimeObject}->TimeStamp2SystemTime(
                    String => $Ticket{"TicketFreeTime".$_},
                ),
            );
            $TicketFreeTime{"TicketFreeTime".$_.'Used'} = 1;
        }
    }
    my %TicketFreeTimeHTML = $Self->{LayoutObject}->AgentFreeDate(
        Ticket => \%TicketFreeTime,
    );
    $Data{'ComposeState'} = 'open';
    # build view ...
    # start with page ...
    $Output .= $Self->{LayoutObject}->Header(
        Value => $Ticket{TicketNumber} . " - " .$Ticket{Title},
        NoCache => 0
    );
    $Output .= $Self->_Mask(
        TicketNumber => $Ticket{TicketNumber},
        TicketID => $Self->{TicketID},
        QueueID => $Ticket{QueueID},
        NextStates => $Self->_GetNextStates(),
        Errors => \%Error,
        Attachments => \@Attachments,
        %Data,
        %GetParam,
        InReplyTo    => $Data{MessageID},
        References   => "$Data{References} $Data{MessageID}",
        TicketFreeTextOptions => $TicketFreeTextOptions,
        TicketFreeTimeOptions => $TicketFreeTimeOptions,
        %TicketFreeTextHTML,
        %TicketFreeTimeHTML,
    );
    $Output .= $Self->{LayoutObject}->Footer();

    return $Output;
}

sub SendEmail {
    my $Self = shift;
    my %Param = @_;
    my %Error = ();
    my %GetParam = %{$Self->{GetParam}};
    my $Output = '';
    my $QueueID = $Self->{QueueID};
    my %StateData = $Self->{TicketObject}->{StateObject}->StateGet(
        ID => $GetParam{ComposeStateID},
    );
    my $NextState = $StateData{Name};

    # check pending date
    if ($StateData{TypeName} && $StateData{TypeName} =~ /^pending/i) {
        if (!$Self->{TimeObject}->Date2SystemTime(%GetParam, Second => 0)) {
            $Error{"Date invalid"} = 'invalid';
        }
    }
    # attachment delete
    foreach (1..16) {
        if ($GetParam{"AttachmentDelete$_"}) {
            $Error{AttachmentDelete} = 1;
            $Self->{UploadCachObject}->FormIDRemoveFile(
                FormID => $GetParam{FormID},
                FileID => $_,
            );
        }
    }
    # attachment upload
    if ($GetParam{AttachmentUpload}) {
        $Error{AttachmentUpload} = 1;
        my %UploadStuff = $Self->{ParamObject}->GetUploadAll(
            Param => "file_upload",
            Source => 'string',
        );
        $Self->{UploadCachObject}->FormIDAddFile(
            FormID => $GetParam{FormID},
            %UploadStuff,
        );
    }
    # get all attachments meta data
    my @Attachments = $Self->{UploadCachObject}->FormIDGetAllFilesMeta(
        FormID => $GetParam{FormID},
    );
    # check some values
    foreach (qw(From To Cc Bcc)) {
        if ($GetParam{$_}) {
            foreach my $Email (Mail::Address->parse($GetParam{$_})) {
                if (!$Self->{CheckItemObject}->CheckEmail(Address => $Email->address())) {
                    $Error{"$_ invalid"} .= $Self->{CheckItemObject}->CheckError();
                }
            }
        }
    }
    # prepare subject
    my $Tn = $Self->{TicketObject}->TicketNumberLookup(TicketID => $Self->{TicketID});
    $GetParam{Subject} = $Self->{TicketObject}->TicketSubjectBuild(
        TicketNumber => $Tn,
        Subject => $GetParam{Subject} || '',
    );
    # rewrap body if exists
    # deactivate it (#10068755)
    #if ($GetParam{Body}) {
    #    my $NewLine = $Self->{ConfigObject}->Get('Ticket::Frontend::TextAreaEmail') || 75;
    #    $GetParam{Body} =~ s/(^>.+|.{4,$NewLine})(?:\s|\z)/$1\n/gm;
    #    my $Quote = $Self->{ConfigObject}->Get('Ticket::Frontend::Quote');
    #}
    # prepare free text
    my %TicketFree = ();
    foreach (1..16) {
        $TicketFree{"TicketFreeKey$_"} = $Self->{ParamObject}->GetParam(Param => "TicketFreeKey$_");
        $TicketFree{"TicketFreeText$_"} = $Self->{ParamObject}->GetParam(Param => "TicketFreeText$_");
    }
    # get free text config options
    my $TicketFreeTextOptions = $Param{TicketFreeTextOptions};
    my %TicketFreeText = ();
    foreach (1..16) {
        $TicketFreeText{"TicketFreeKey$_"} = $Self->{TicketObject}->TicketFreeTextGet(
            TicketID => $Self->{TicketID},
            Type => "TicketFreeKey$_",
            Action => $Self->{Action},
            UserID => $Self->{UserID},
        );
        $TicketFreeText{"TicketFreeText$_"} = $Self->{TicketObject}->TicketFreeTextGet(
            TicketID => $Self->{TicketID},
            Type => "TicketFreeText$_",
            Action => $Self->{Action},
            UserID => $Self->{UserID},
        );
        # check required FreeTextField (if configured)
        if ($TicketFreeTextOptions->[$_-1] == 2 && $TicketFree{"TicketFreeText$_"} eq '') {
            $Error{"TicketFreeTextField$_ invalid"} = 'invalid';
        }
    }
    my %TicketFreeTextHTML = $Self->{LayoutObject}->AgentFreeText(
        Config => \%TicketFreeText,
        Ticket => \%TicketFree,
    );
    # get ticket free time params
    my $TicketFreeTimeOptions = $Param{TicketFreeTimeOptions};
    my %TicketFreeTime = ();
    foreach (1..6) {
        foreach my $Type (qw(Used Year Month Day Hour Minute)) {
            $GetParam{"TicketFreeTime".$_.$Type} = $Self->{ParamObject}->GetParam(Param => "TicketFreeTime".$_.$Type);
        }
        $GetParam{'TicketFreeTime'.$_.'Optional'} = 1;
        if (!$Self->{ConfigObject}->Get('TicketFreeTimeOptional'.$_)) {
            $GetParam{'TicketFreeTime'.$_.'Used'} = 1;
        }
        if ($TicketFreeTimeOptions->[$_-1] == 2 && $GetParam{'TicketFreeTime'.$_.'Used'} == 0) {
            $Error{"TicketFreeTimeField$_ invalid"} = 'invalid';
        }
    }
    my %TicketFreeTimeHTML = $Self->{LayoutObject}->AgentFreeDate(
        Ticket => \%GetParam,
    );
    # check some values
    foreach (qw(To Cc Bcc)) {
        if ($GetParam{$_}) {
            foreach my $Email (Mail::Address->parse($GetParam{$_})) {
                if (!$Self->{CheckItemObject}->CheckEmail(Address => $Email->address())) {
                    $Error{"$_ invalid"} .= $Self->{CheckItemObject}->CheckError();
                }
            }
        }
    }

    my %ArticleParam = ();
    # run compose modules
    if (ref($Self->{ConfigObject}->Get('Ticket::Frontend::ArticleComposeModule')) eq 'HASH') {
        my %Jobs = %{$Self->{ConfigObject}->Get('Ticket::Frontend::ArticleComposeModule')};
        foreach my $Job (sort keys %Jobs) {
            # load module
            if ($Self->{MainObject}->Require($Jobs{$Job}->{Module})) {
                my $Object = $Jobs{$Job}->{Module}->new(
                    %{$Self},
                    Debug => $Self->{Debug},
                );
                # get params
                foreach ($Object->Option(%GetParam, Config => $Jobs{$Job})) {
                    $GetParam{$_} = $Self->{ParamObject}->GetParam(Param => $_);
                }
                # run module
                $Object->Run(%GetParam, Config => $Jobs{$Job});
                # ticket params
                %ArticleParam = (%ArticleParam, $Object->ArticleOption(%GetParam, Config => $Jobs{$Job}));
                # get errors
                %Error = (%Error, $Object->Error(%GetParam, Config => $Jobs{$Job}));
            }
            else {
                return $Self->{LayoutObject}->FatalError();
            }
        }
    }
    # check if there is an error
    if (%Error) {
        my $QueueID = $Self->{TicketObject}->TicketQueueID(TicketID => $Self->{TicketID});
        my $Output = $Self->{LayoutObject}->Header(Value => $Tn);
        $Output .= $Self->_Mask(
            TicketNumber => $Tn,
            TicketID => $Self->{TicketID},
            QueueID => $QueueID,
            NextStates => $Self->_GetNextStates(),
            Errors => \%Error,
            Attachments => \@Attachments,
            TicketFreeTextOptions => $TicketFreeTextOptions,
            TicketFreeTimeOptions => $TicketFreeTimeOptions,
            %TicketFreeTextHTML,
            %TicketFreeTimeHTML,
            %GetParam,
        );
        $Output .= $Self->{LayoutObject}->Footer();
        return $Output;
    }
    # replace <OTRS_TICKET_STATE> with next ticket state name
    if ($NextState) {
        $GetParam{Body} =~ s/<OTRS_TICKET_STATE>/$NextState/g;
    }
    # get pre loaded attachments
    my @AttachmentData = $Self->{UploadCachObject}->FormIDGetAllFilesData(
        FormID => $GetParam{FormID},
    );
    # get submit attachment
    my %UploadStuff = $Self->{ParamObject}->GetUploadAll(
        Param => 'file_upload',
        Source => 'String',
    );
    if (%UploadStuff) {
        push (@AttachmentData, \%UploadStuff);
    }

    my $MimeType = 'text/plain';
    if ( $Self->{ConfigObject}->Get('Frontend::RichText') ) {
        $MimeType = 'text/html';

        # remove unused inline images
        my @NewAttachmentData = ();
        REMOVEINLINE:
        for my $TmpAttachment (@AttachmentData) {
            next REMOVEINLINE if $TmpAttachment->{ContentID}
                    && $TmpAttachment->{ContentID} =~ /^inline/
                    && $GetParam{Body} !~ /$TmpAttachment->{ContentID}/;
            push @NewAttachmentData, \%{$TmpAttachment};
        }
        @AttachmentData = @NewAttachmentData;

        # verify html document
        $GetParam{Body} = $Self->{LayoutObject}->RichTextDocumentComplete(
            String => $GetParam{Body},
        );
    }

    # send email
    my $ArticleID = $Self->{TicketObject}->ArticleSend(
        ArticleTypeID => $Self->{GetParam}->{ArticleTypeID},
        SenderType    => 'agent',
        TicketID      => $Self->{TicketID},
        HistoryType   => 'SendAnswer',
        HistoryComment => "\%\%$GetParam{To}, $GetParam{Cc}, $GetParam{Bcc}",
        From       => $GetParam{From},
        To         => $GetParam{To},
        Cc         => $GetParam{Cc},
        Bcc        => $GetParam{Bcc},
        'Reply-To' => $GetParam{'Reply-To'},
        Subject    => $GetParam{Subject},
        UserID     => $Self->{UserID},
        Body       => $GetParam{Body},
        InReplyTo  => $GetParam{InReplyTo},
        References => $GetParam{References},
        Charset    => $Self->{LayoutObject}->{UserCharset},
        MimeType   => $MimeType,
        Attachment => \@AttachmentData,
        %ArticleParam,
    );

    # error page
    if (! $ArticleID ) {
        return $Self->{LayoutObject}->ErrorScreen( Comment => 'Please contact the admin.', );
    }

    # time accounting
    if ( $GetParam{TimeUnits} ) {
        $Self->{TicketObject}->TicketAccountTime(
            TicketID  => $Self->{TicketID},
            ArticleID => $ArticleID,
            TimeUnit  => $GetParam{TimeUnits},
            UserID    => $Self->{UserID},
        );
    }

    # set priority
    if ( $Self->{Config}->{Priority} && $GetParam{NewPriorityID} ) {
        $Self->{TicketObject}->PrioritySet(
            TicketID   => $Self->{TicketID},
            PriorityID => $GetParam{NewPriorityID},
            UserID     => $Self->{UserID},
        );
    }

    # update ticket free text
    for ( 1 .. 16 ) {
        my $FreeKey = $Self->{ParamObject}->GetParam(Param => "TicketFreeKey$_");
        my $FreeValue = $Self->{ParamObject}->GetParam(Param => "TicketFreeText$_");
        if (defined($FreeKey) && defined($FreeValue)) {
            $Self->{TicketObject}->TicketFreeTextSet(
                Key => $FreeKey,
                Value => $FreeValue,
                Counter => $_,
                TicketID => $Self->{TicketID},
                UserID => $Self->{UserID},
            );
        }
    }

    # set ticket free time
    for ( 1 .. 6 ) {
        if (
            defined $GetParam{ "TicketFreeTime" . $_ . "Year" }
            && defined $GetParam{ "TicketFreeTime" . $_ . "Month" }
            && defined $GetParam{ "TicketFreeTime" . $_ . "Day" }
            && defined $GetParam{ "TicketFreeTime" . $_ . "Hour" }
            && defined $GetParam{ "TicketFreeTime" . $_ . "Minute" }
            )
        {
            my %Time;
            $Time{ "TicketFreeTime" . $_ . "Year" }    = 0;
            $Time{ "TicketFreeTime" . $_ . "Month" }   = 0;
            $Time{ "TicketFreeTime" . $_ . "Day" }     = 0;
            $Time{ "TicketFreeTime" . $_ . "Hour" }    = 0;
            $Time{ "TicketFreeTime" . $_ . "Minute" }  = 0;
            $Time{ "TicketFreeTime" . $_ . "Secunde" } = 0;

            if ( $GetParam{ "TicketFreeTime" . $_ . "Used" } ) {
                %Time = $Self->{LayoutObject}->TransfromDateSelection(
                    %GetParam,
                    Prefix => "TicketFreeTime" . $_,
                );
            }
            $Self->{TicketObject}->TicketFreeTimeSet(
                %Time,
                Prefix   => "TicketFreeTime",
                TicketID => $Self->{TicketID},
                Counter  => $_,
                UserID   => $Self->{UserID},
            );
        }
    }
    # set state
    $Self->{TicketObject}->StateSet(
        TicketID  => $Self->{TicketID},
        ArticleID => $ArticleID,
        State     => $NextState,
        UserID    => $Self->{UserID},
    );

    # should I set an unlock?
    if ( $StateData{TypeName} =~ /^close/i ) {
        $Self->{TicketObject}->LockSet(
            TicketID => $Self->{TicketID},
            Lock     => 'unlock',
            UserID   => $Self->{UserID},
        );
    }

    # set pending time
    elsif ( $StateData{TypeName} =~ /^pending/i ) {
        $Self->{TicketObject}->TicketPendingTimeSet(
            UserID   => $Self->{UserID},
            TicketID => $Self->{TicketID},
            Year     => $GetParam{Year},
            Month    => $GetParam{Month},
            Day      => $GetParam{Day},
            Hour     => $GetParam{Hour},
            Minute   => $GetParam{Minute},
        );
    }

    # remove pre submited attachments
    $Self->{UploadCachObject}->FormIDRemove(FormID => $GetParam{FormID});

    return $Self->{LayoutObject}->Redirect(OP => "Action=AgentTicketZoom&TicketID=$Self->{TicketID}&ArticleID=$ArticleID");
}

sub _GetNextStates {
    my $Self = shift;
    my %Param = @_;
    # get next states
    my %NextStates = $Self->{TicketObject}->StateList(
        Action => $Self->{Action},
        TicketID => $Self->{TicketID},
        UserID => $Self->{UserID},
    );
    $NextStates{''} = '-' if $Self->{isOwner};
    return \%NextStates;
}

sub _Mask {
    my $Self = shift;
    my %Param = @_;
    # build next states string
    my %State = ();
    if ($Self->{isOwner}) {
        if ( !$Param{ComposeStateID} ) {
            $State{Selected} = $Self->{Config}->{StateDefault};
        }
        else {
            $State{SelectedID} = $Param{ComposeStateID};
        }
    }
    else {
        $State{Selected} = 'open';
    }

    # waittime only when needed. (#10057256)
    my $JS_pendingstate_IDs = '';
    my $DisabledPendingTime = 0;
    {
        if ( $State{Selected} ) {
            my %TStateID = $Self->{TicketObject}->{StateObject}->StateGet( Name => $State{Selected} );
            $DisabledPendingTime = 1 if $TStateID{TypeName} !~ /pending/i;
        }
        elsif ( $State{SelectedID} ) {
            my %TStateID = $Self->{TicketObject}->{StateObject}->StateGet( ID => $State{SelectedID} );
            $DisabledPendingTime = 1 if $TStateID{TypeName} !~ /pending/i;
        }
        elsif ( $Param{State} ) {
            $DisabledPendingTime = 1 if $Param{State} !~ /pending/i;
        }
        my @List = $Self->{StateObject}->StateGetStatesByType(
            StateType => [ 'pending auto', ],
            Result => 'ID',
        );
        $JS_pendingstate_IDs = "{'" . join( "':1, '",  @List) . "':1}";
    }
    $Param{'NextStatesStrg'} = $Self->{LayoutObject}->OptionStrgHashRef(
        Data => $Param{NextStates},
        Name => 'ComposeStateID',
        %State,
        OnChange => 'toggle_pendingtime(' . $JS_pendingstate_IDs . ', \'ComposeStateID\')',
    );

    my %ArticleTypes = ();
    my @ArticleTypesPossible = @{$Self->{Config}->{ArticleTypes}};
    foreach (@ArticleTypesPossible) {
        $ArticleTypes{$Self->{TicketObject}->ArticleTypeLookup(ArticleType => $_)} = $_;
    }
    if ($Self->{GetParam}->{ArticleTypeID}) {
        $Param{'ArticleTypesStrg'} = $Self->{LayoutObject}->OptionStrgHashRef(
            Data => \%ArticleTypes,
            Name => 'ArticleTypeID',
            SelectedID => $Self->{GetParam}->{ArticleTypeID},
        );
    }
    else {
        $Param{'ArticleTypesStrg'} = $Self->{LayoutObject}->OptionStrgHashRef(
            Data => \%ArticleTypes,
            Name => 'ArticleTypeID',
            Selected => $Self->{Config}->{ArticleTypeDefault},
        );
    }

    # prepare errors!
    if ($Param{Errors}) {
        foreach (keys %{$Param{Errors}}) {
            $Param{$_} = "* ".$Self->{LayoutObject}->Ascii2Html(Text => $Param{Errors}->{$_});
        }
    }
    # pending data string
    my $Initial;
    if ($Param{RealTillTimeNotUsed} 
        && $Param{RealTillTimeNotUsed} > $Self->{TimeObject}->SystemTime())
    {
        $Initial = $Param{RealTillTimeNotUsed};
    }
    $Param{PendingDateString} = $Self->{LayoutObject}->BuildDateSelection(
        %Param,
        Format => 'DateInputFormatLong',
        DiffTime => $Self->{ConfigObject}->Get('Ticket::Frontend::PendingDiffTime') || 0,
        Initial => $Initial,
        Disabled => $DisabledPendingTime,
    );

    my %Ticket = $Self->{TicketObject}->TicketGet( TicketID => $Self->{TicketID} );

    if ( $Self->{Config}->{Priority} ) {
        my %Priority;
        my %PriorityList = $Self->{TicketObject}->PriorityList(
            UserID   => $Self->{UserID},
            TicketID => $Self->{TicketID},
        );
        if ( !$Self->{Config}->{PriorityDefault} ) {
            $PriorityList{''} = '-';
        }
        if ( $Param{NewPriorityID} ) {
            $Priority{SelectedID} = $Param{NewPriorityID};
        }
        elsif ( $Ticket{PriorityID} ) {
            $Priority{SelectedID} = $Ticket{PriorityID};
        }
        else {
            if ( $Self->{Config}->{PriorityDefault} ) {
                $Priority{Selected} = $Self->{Config}->{PriorityDefault};
            }
        }
        $Param{PriorityStrg} = $Self->{LayoutObject}->OptionStrgHashRef(
            Data => \%PriorityList,
            Name => 'NewPriorityID',
            %Priority,
        );
        $Self->{LayoutObject}->Block(
            Name => 'Priority',
            Data => \%Param,
        );
    }
    # ticket free text
    my $TicketFreeTextOptions = $Param{TicketFreeTextOptions};
    my $Count = 0;
    foreach (1..16) {
        $Count++;
        if ( $TicketFreeTextOptions->[ $Count - 1 ] > 0 ) {
            $Self->{LayoutObject}->Block(
                Name => 'TicketFreeText',
                Data => {
                    TicketFreeKeyField => $Param{'TicketFreeKeyField'.$Count},
                    TicketFreeTextField => $Param{'TicketFreeTextField'.$Count},
                    %Param,
                },
            );
            $Self->{LayoutObject}->Block(
                Name => 'TicketFreeText'.$Count,
                Data => {
                    %Param,
                    Count => $Count,
                },
            );
        }
    }
    # ticket free time
    my $TicketFreeTimeOptions = $Param{TicketFreeTimeOptions};
    $Count = 0;
    foreach (1..6) {
        $Count++;
        if ( $TicketFreeTimeOptions->[ $Count - 1 ] > 0 ) {
            $Self->{LayoutObject}->Block(
                Name => 'TicketFreeTime',
                Data => {
                    TicketFreeTimeKey => $Self->{ConfigObject}->Get('TicketFreeTimeKey'.$Count),
                    TicketFreeTime => $Param{'TicketFreeTime'.$Count},
                    Count => $Count,
                    %Param,
                },
            );
            $Self->{LayoutObject}->Block(
                Name => 'TicketFreeTime'.$Count,
                Data => {
                    %Param,
                    Count => $Count,
                },
            );
        }
    }
    # show time accounting box
    if ($Self->{ConfigObject}->Get('Ticket::Frontend::AccountTime')) {
        $Self->{LayoutObject}->Block(
            Name => 'TimeUnitsJs',
            Data => \%Param,
        );
        $Self->{LayoutObject}->Block(
            Name => 'TimeUnits',
            Data => \%Param,
        );
    }
    # show spell check
    if ($Self->{ConfigObject}->Get('SpellChecker') && $Self->{LayoutObject}->{BrowserJavaScriptSupport}) {
        $Self->{LayoutObject}->Block(
            Name => 'SpellCheck',
            Data => {},
        );
    }
    # show address book
    if ($Self->{LayoutObject}->{BrowserJavaScriptSupport}) {
        $Self->{LayoutObject}->Block(
            Name => 'AddressBook',
            Data => {},
        );
    }
    # show attachments
    foreach my $DataRef (@{$Param{Attachments}}) {
        $Self->{LayoutObject}->Block(
            Name => 'Attachment',
            Data => $DataRef,
        );
    }
    # jscript check freetextfields by submit
    foreach my $Key (keys %{$Self->{Config}->{TicketFreeText}}) {
        if ( $TicketFreeTextOptions->[ $Key - 1 ] == 2 ) {
            $Self->{LayoutObject}->Block(
                Name => 'TicketFreeTextCheckJs',
                Data => {
                    TicketFreeTextField => "TicketFreeText$Key",
                    TicketFreeKeyField => "TicketFreeKey$Key",
                },
            );
        }
    }
    # create & return output
    return $Self->{LayoutObject}->Output(TemplateFile => 'AgentTicketComposeInternal', Data => \%Param);
}

1;
