#include <stdlib.h>
#include <errno.h>
#include <assert.h>

#include <net-snmp/net-snmp-config.h>
#include <net-snmp/net-snmp-includes.h>
#include <net-snmp/agent/net-snmp-agent-includes.h>

#include <mysql/mysql.h>

#include "capman_mysql.h"

#ifndef DEBUG
#define DEBUG 0
#endif

/*
 * Macros
 */
#define SFREE(ptr) if ((ptr) != NULL) { free (ptr); ptr = NULL; }

/*
 * My data structures
 */
#define SSL_NAME_LEN 52
struct show_status_list_s;
typedef struct show_status_list_s show_status_list_t;
struct show_status_list_s
{
	int                 ssl_id;
	char                ssl_name[SSL_NAME_LEN];
	int                 ssl_value;
	show_status_list_t *ssl_next;
};

show_status_list_t *g_show_status_list = NULL;

/*
 * Module global variables
 */
static char *mysql_host_g = NULL;
static char *mysql_user_g = NULL;
static char *mysql_pass_g = NULL;
static char *mysql_db_g   = NULL;

/*
 * Intern (static) functions
 */
static MYSQL *db_get_connection (void)
{
	static MYSQL *con;
	static int    state;

	if (state != 0)
	{
		int err;
		if ((err = mysql_ping (con)) != 0)
		{
			snmp_log (LOG_WARNING, "mysql_ping failed: %s", mysql_error (con));
			state = 0;
		}
		else
		{
			state = 1;
			return (con);
		}
	}

	if ((con = mysql_init (con)) == NULL)
	{
		snmp_log (LOG_ERR, "mysql_init failed: %s", mysql_error (con));
		state = 0;
		return (NULL);
	}

	if (mysql_real_connect (con, mysql_host_g, mysql_user_g,
				mysql_pass_g, mysql_db_g, 0, NULL, 0)
			== NULL)
	{
		snmp_log (LOG_ERR, "mysql_real_connect failed: %s", mysql_error (con));
		state = 0;
		return (NULL);
	}
	else
	{
		state = 1;
		return (con);
	}
} /* static MYSQL *db_get_connection (void) */

static netsnmp_variable_list *ti_get_next_data_point (void **loop_context,
		void **data_context,
		netsnmp_variable_list *var_list,
		netsnmp_iterator_info *iinfo)
{
	show_status_list_t    *ssl_ptr;
	netsnmp_variable_list *var_ptr;

	if (*loop_context == NULL)
	{
#if DEBUG
		snmp_log (LOG_DEBUG, "ti_get_next_data_point: *loop_context == NULL\n");
#endif
		return (NULL);
	}

	ssl_ptr = (show_status_list_t *) *loop_context;

	var_ptr = var_list;
	assert (var_ptr != NULL);
	snmp_set_var_value (var_ptr, (char *) &ssl_ptr->ssl_id, sizeof (ssl_ptr->ssl_id));

	*data_context = (void *) ssl_ptr;
	*loop_context = (void *) ssl_ptr->ssl_next;

	return (var_list);
}

static netsnmp_variable_list *ti_get_first_data_point (void **loop_context,
		void **data_context,
		netsnmp_variable_list *var_list,
		netsnmp_iterator_info *iinfo)
{
	*data_context = (void *) g_show_status_list;
	*loop_context = *data_context;

	return (ti_get_next_data_point (loop_context, data_context, var_list, iinfo));
}

static int ti_handler (netsnmp_mib_handler *mib_handler,
		netsnmp_handler_registration *reginfo,
		netsnmp_agent_request_info *reqinfo,
		netsnmp_request_info *requests)
{
	void                       *loop_context;
	netsnmp_request_info       *request;
	netsnmp_variable_list      *var_list;
	netsnmp_table_request_info *table_info;
	show_status_list_t         *ssl_ptr;

#if DEBUG
	snmp_log (LOG_DEBUG, "ti_handler (%p, %p, %p, %p);\n",
			(void *) mib_handler,
			(void *) reginfo,
			(void *) reqinfo,
			(void *) requests);
#endif
	DEBUGMSGTL (("capman_mysql", "begin ti_handler"));

	if (reqinfo->mode != MODE_GET)
	{
		snmp_log (LOG_WARNING, "ti_handler: Mode %i is not implemented yet.\n",
				(int) reqinfo->mode);
		return (SNMP_ERR_NOERROR);
	}

	for (request = requests; request != NULL; request = request->next)
	{
		if ((loop_context = netsnmp_extract_iterator_context (request)) == NULL)
		{
			/* 
			 * This happends every time, so I suspect it's normal.
			 * Don't know for sure though. -octo
			snmp_log (LOG_ERR, "ti_handler: netsnmp_extract_iterator_context failed.\n");
			 */
			continue;
		}
		ssl_ptr = (show_status_list_t *) loop_context;
#if DEBUG
		snmp_log (LOG_DEBUG, "ti_handler: ssl_ptr = %p\n", (void *) ssl_ptr);
#endif

		if ((table_info = netsnmp_extract_table_info (request)) == NULL)
		{
			snmp_log (LOG_ERR, "ti_handler: netsnmp_extract_table_info failed.\n");
			continue;
		}

		if ((var_list = request->requestvb) == NULL)
		{
			snmp_log (LOG_ERR, "ti_handler: request->requestvb == NULL.\n");
			continue;
		}

		switch (table_info->colnum)
		{
			case 1:
				snmp_set_var_typed_value (var_list, ASN_OCTET_STR,
						ssl_ptr->ssl_name,
						strlen (ssl_ptr->ssl_name));
				break;

			case 2:
				snmp_set_var_typed_value (var_list, ASN_INTEGER,
						(const char *) &ssl_ptr->ssl_value,
						sizeof (ssl_ptr->ssl_value));
				break;

#if DEBUG
			default:
				snmp_log (LOG_DEBUG, "ti_handler: table_info->colnum = %i\n",
						(int) table_info->colnum);
#endif
		} /* switch (table_info->colnum) */
	} /* for (requests) */

	DEBUGMSGTL (("capman_mysql", "end ti_handler"));

	return (SNMP_ERR_NOERROR);
}

static void cache_free (netsnmp_cache *cache, void *vmagic)
{
	show_status_list_t *ptr;

#if DEBUG
	snmp_log (LOG_DEBUG, "cache_free (%p, %p);\n",
			(void *) cache, vmagic);
#endif

	while (g_show_status_list != NULL)
	{
		ptr = g_show_status_list;
		g_show_status_list = g_show_status_list->ssl_next;
		free (ptr);
	}
	g_show_status_list = NULL;
}

static int cache_load (netsnmp_cache *cache, void *vmagic)
{
	MYSQL     *con;
	MYSQL_RES *res;
	MYSQL_ROW  row;
	char      *query;
	int        query_len;
	int        field_num;
	show_status_list_t *ssl_tail;
	show_status_list_t *ssl_new;
	int subid;

#if DEBUG
	snmp_log (LOG_DEBUG, "cache_load (%p, %p);\n",
			(void *) cache, vmagic);
#endif

	cache_free (cache, vmagic);

	assert (g_show_status_list == NULL);
	ssl_tail = NULL;
	subid = 0;

	if ((con = db_get_connection ()) == NULL)
		return (-1);

	query = "SHOW STATUS";
	if (mysql_get_server_version (con) >= 50002)
		query = "SHOW GLOBAL STATUS";
	query_len = strlen (query);

	if (mysql_real_query (con, query, query_len))
	{
		snmp_log (LOG_ERR, "mysql_real_query failed: %s\n",
				mysql_error (con));
		return (-1);
	}

	if ((res = mysql_store_result (con)) == NULL)
	{
		snmp_log (LOG_ERR, "mysql_store_result failed: %s\n",
				mysql_error (con));
		return (-1);
	}

	field_num = mysql_num_fields (res);
	while ((row = mysql_fetch_row (res)))
	{
		char *key;
		int   val;

		subid++;

		key = row[0];
		val = atol (row[1]);

		if ((ssl_new = (show_status_list_t *) malloc (sizeof (show_status_list_t))) == NULL)
		{
			snmp_log (LOG_ERR, "malloc failed: %s", strerror (errno));
			continue;
		}
		memset ((void *) ssl_new, '\0', sizeof (show_status_list_t));

		ssl_new->ssl_id = subid;
		strncpy (ssl_new->ssl_name, key, SSL_NAME_LEN);
		ssl_new->ssl_value = val;
		ssl_new->ssl_next = NULL;

		if (g_show_status_list == NULL)
		{
			g_show_status_list = ssl_new;
			ssl_tail = ssl_new;
		}
		else
		{
			assert (ssl_tail != NULL);
			ssl_tail->ssl_next = ssl_new;
			ssl_tail = ssl_new;
		}
	}
	mysql_free_result (res); res = NULL;

	return (0);
}

static void config_handler (const char *key, char *value)
{
#if DEBUG
	snmp_log (LOG_DEBUG, "config_handler (key = %s, value = %s);\n",
			key, value);
#endif

	if (strcmp (key, "mysql_host") == 0)
	{
		SFREE (mysql_host_g);
		mysql_host_g = strdup (value);
	}
	else if (strcmp (key, "mysql_user") == 0)
	{
		SFREE (mysql_user_g);
		mysql_user_g = strdup (value);
	}
	else if ((strcmp (key, "mysql_password") == 0)
			|| (strcmp (key, "mysql_pass") == 0))
	{
		SFREE (mysql_pass_g);
		mysql_pass_g = strdup (value);
	}
	else if (strcmp (key, "mysql_db") == 0)
	{
		SFREE (mysql_db_g);
		mysql_db_g = strdup (value);
	}
	else
	{
		snmp_log (LOG_WARNING, "Unknown configuration token: %s = %s\n",
				key, value);
	}
}

static void config_free (void)
{
#if DEBUG
	snmp_log (LOG_DEBUG, "config_free ();\n");
#endif

	SFREE (mysql_host_g);
	SFREE (mysql_user_g);
	SFREE (mysql_pass_g);
	SFREE (mysql_db_g);
}


/*
 * Extern (public) functions
 */
void init_capman_mysql (void) /* Must match filename! */
{
	/*
	 * enterprises.12114 = noris network AG 
	 *            .42    = CapMan
	 *            .1     = MySQL
	 */
	static oid capman_mysql_oid[] = { 1, 3, 6, 1, 4, 1, 12114, 42, 1 };

	netsnmp_table_registration_info *table_info;
	netsnmp_iterator_info           *iinfo;
	netsnmp_handler_registration    *reginfo;
	netsnmp_mib_handler             *cache_handler;

#if DEBUG
	DEBUGMSGTL (("capman_mysql", "begin init_capman_mysql"));
	snmp_log (LOG_DEBUG, "init_capman_mysql ();\n");
#endif

	/*
	 * Create the table data structure, and define the indexing....
	 */
	if ((table_info = SNMP_MALLOC_TYPEDEF (netsnmp_table_registration_info)) == NULL)
	{
		snmp_log (LOG_ERR, "Allocating %u bytes failed.\n",
				(unsigned int) sizeof (netsnmp_table_registration_info));
		return;
	}
	netsnmp_table_helper_add_indexes (table_info, ASN_INTEGER, 0);
	table_info->min_column = 1;
	table_info->max_column = 2;

	/*
	 * .... and iteration information ....
	 */
	if ((iinfo = SNMP_MALLOC_TYPEDEF (netsnmp_iterator_info)) == NULL)
	{
		snmp_log (LOG_ERR, "Allocating %u bytes failed.\n",
				(unsigned int) sizeof (netsnmp_iterator_info));
		return;
	}
	iinfo->get_first_data_point = ti_get_first_data_point;
	iinfo->get_next_data_point  = ti_get_next_data_point;
	iinfo->table_reginfo        = table_info;
	iinfo->flags               |= NETSNMP_ITERATOR_FLAG_SORTED;


	/*
	 * .... and register the table with the agent.
	 */
	reginfo = netsnmp_create_handler_registration ("capman_mysql",
			ti_handler,
			capman_mysql_oid, OID_LENGTH (capman_mysql_oid),
			HANDLER_CAN_DEFAULT); /* GET and GETNEXT only */
	netsnmp_register_table_iterator (reginfo, iinfo);

	/*
	 * .... with a local cache
	 */
	if ((cache_handler = netsnmp_get_cache_handler (60,
					cache_load, cache_free,
					capman_mysql_oid, OID_LENGTH (capman_mysql_oid))) == NULL)
	{
		snmp_log (LOG_ERR, "init_capman_mysql: netsnmp_get_cache_handler failed.\n");
		return;
	}
	netsnmp_inject_handler (reginfo, cache_handler);

	snmpd_register_config_handler ("mysql_host",
			config_handler, config_free,
			"MySQL hostname");
	snmpd_register_config_handler ("mysql_user",
			config_handler, config_free,
			"MySQL username");
	snmpd_register_config_handler ("mysql_pass",
			config_handler, config_free,
			"Synonym for `mysql_password'");
	snmpd_register_config_handler ("mysql_password",
			config_handler, config_free,
			"Password for the MySQL user");
	snmpd_register_config_handler ("mysql_db",
			config_handler, config_free,
			"MySQL database to select");
} /* void init_capman_mysql */
