# -*- encoding: utf-8 -*-

"""Sonderfunktionen für Kunden auch ohne Service-Flag.
"""

import os.path, urllib2

from django.core.urlresolvers import reverse
from django.http import HttpResponse, Http404, HttpResponseRedirect, urlencode
from django.template import loader, RequestContext
from django.shortcuts import render_to_response
from django.conf import settings
from django.utils.encoding import force_unicode
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext_lazy

from kundebunt import newforms
from kundebunt.popkern import models, updatelog
from kundebunt.popkern.utils import any
from kundebunt.kunde_auth.decorators import service_flag_required


_bad_auth_message = ugettext_lazy(u"Der Benutzername oder das Passwort sind falsch.")
_password_is_username_message = ugettext_lazy(
    u"Das neue Passwort darf aus Sicherheitsgründen nicht identisch mit dem Benutzernamen sein. "
    u"Bitte wählen Sie in Ihrem eigenen Interesse ein sicheres Passwort.")
_password_confirm_failed_message = ugettext_lazy(u"Das Bestätigungs-Passwort stimmt nicht mit dem neuen Passwort überein.")
_bad_flag_message = mark_safe(ugettext_lazy(
    u'Eine Änderung des Passworts ist bei diesem Zugang nur durch unseren '
    u'<a href="mailto:support@noris.net">Support</a> möglich.'))
_no_password_kunde_message = _bad_flag_message


def _clean_css(css):
    """Überprüft eine mögliche css-URL. Wenn die URL von einem Host stammt, dessen
    Domain (oder eine parent domain) in der POP-DB eingetragen ist, wird die css
    zuirückgegeben, anderenfalls nur u'' Die Domain muss mindestens zwei Labels umfassen.
    """
    if css:
        css_host = urllib2.Request(css).get_origin_req_host().strip(u'.')
        if css_host:
            css_labels = css_host.split(u'.')
            css_domains = [u".".join(css_labels[i:]) for i in range(len(css_labels)-1)]
            if not css_domains or not models.Domainkunde.objects.active().filter(domain__in=css_domains).exists():
                css = u''
    return css


class PasswordChangeForm(newforms.Form):
    username = newforms.CharField(label=ugettext_lazy(u"Benutzername"), required=True, max_length=32)
    old_password = newforms.CharField(label=ugettext_lazy(u"Altes Passwort"), widget=newforms.PasswordInput,
                                      required=True, max_length=255, min_length=3)
    new_password = newforms.CharField(label=ugettext_lazy(u"Neues Passwort"), widget=newforms.PasswordInput,
                                      required=True, max_length=255, min_length=8)
    confirmation_password = newforms.CharField(label=ugettext_lazy(u"Neues Passwort zur Bestätigung"),
                                               widget=newforms.PasswordInput, required=True, max_length=255,
                                               min_length=6)

    def clean(self):
        """Prüft:

        a) username: ob der Personeintrag existiert und das richtige Passwort angegeben hat. Außerdem
           dürfen keine Flags außerhalb der Gruppe "passwordchange" der Person zugeordnet sein, und
           wenn der primäre Kunde das Flag "no_passwordchange" hat, geht gar nichts.

        b) passwords: ob das neue Passwort brauchbar ist, und ob es mit dem Bestätigungs-Passwort übereinstimmt.

        Außerdem wird self.kunde gesetzt.
        """
        data = self.cleaned_data
        if "username" in data and "old_password" in data:
            try:
                user = models.Person.objects.active().get(user=data["username"])
            except models.Person.DoesNotExist:
                raise newforms.ValidationError(_bad_auth_message)
            if user.pass_field != data["old_password"]:
                raise newforms.ValidationError(_bad_auth_message)
            permitted_flags = set(models.Person.get_pwuse_descr_in_group("passwordchange"))
            if any(fl not in permitted_flags for fl in user.pwuse_set()):
                raise newforms.ValidationError(_bad_flag_message)
            if user.kunde.has_flags_flag("no_passwordchange"):
                raise newforms.ValidationError(_no_password_kunde_message)
            self.user = user
        return data

    def clean_new_password(self):
        data = self.cleaned_data
        new_password = data["new_password"]
        if new_password == data["username"]:
            raise newforms.ValidationError(_password_confirm_failed_message)
        newforms.validate_only_latin1_characters(new_password)
        return new_password

    def clean_confirmation_password(self):
        data = self.cleaned_data
        confirmation_password = data["confirmation_password"]
        if "new_password" in data and data["new_password"] != confirmation_password:
            raise newforms.ValidationError(_password_confirm_failed_message)
        return confirmation_password


def password_change(request):
    """
    Änderung eines Passworts. Auch für Benutzer ohne service-Flag.
    """
    if request.method == 'POST':
        form = PasswordChangeForm(request.POST)
        if form.is_valid():
            user = form.user
            updatelog.set_person(user)
            user.pass_field = form.cleaned_data["new_password"]
            user.save()
            redirect_url = "%s?%s" % (reverse(password_confirmation), request.META.get('QUERY_STRING', ''))
            return HttpResponseRedirect(redirect_url)
    else:
        form = PasswordChangeForm()
    template = loader.get_template("user/password_change.html")
    content = template.render(RequestContext(
            request,
            {'form': form,
             "nologo": "nologo" in request.GET,
             "css": _clean_css(request.GET.get('css', u'')),
            }))
    return HttpResponse(content)


def password_confirmation(request):
    """
    Bestätigung der erfolgten Passwortänderung. Auch für Benutzer ohne service-Flag.
    """
    template = loader.get_template("user/password_confirmation.html")
    content = template.render(RequestContext(request,
            {"nologo": "nologo" in request.GET,
             "css": _clean_css(request.GET.get('css', u''))}))
    return HttpResponse(content)


@service_flag_required
def download_header(request):
    path = ''
    path_list = []
    for part in request.META.get('SCRIPT_URL', request.path).split('/'):
        if part:
            path = os.path.join(path, part)
            path_list.append((part, path))
    if len(path_list) > 1:
        parent_dir = path_list[-2][1]
    else:
        parent_dir = None
    content = loader.render_to_string('user/download.html',
            context_instance = RequestContext(request,
                {'path_list': path_list,
                 'parent_dir': parent_dir,
                }))
    content = content.split(u'**CONTENT**',1)[0]
    return HttpResponse(content)


@service_flag_required
def download_footer(request):
    content = loader.render_to_string('user/download.html', context_instance = RequestContext(request, {}))
    content = content.split(u'**CONTENT**',1)[1]
    return HttpResponse(content)

