"""Create doc-testable, evaluatable representations of objects with these properties:

    * eval(repr(t_repr(a))) == t_repr(a)
      i.e., t_repr has an evaluatable repr()
    * a==b  ==>  t_repr(a)) == t_repr(b)
    * t_repr(a) == t_repr(b) is a strong indication for a == b
    * if a and b are instances of the same class:
       eval(t_repr(a)) == eval(t_repr(b))  ==> a==b
    * easy to read and write

    Note that t_repr(obj) is not necessarily a string.

    For containers, such as lists, dicts, sets, prepr is the same type of container
    consisting of the prepr(x) for all elements x of the container.
    For simple builtin types, it's the type itself, with the exception that all
    integers are normalized to longs. For other types, there's either a special representation
    function registered, or `repr()` is used as fallback, but removing trailing "object at ..."

    For classes with too simplistic repr(), this means that t_repr(obj1)==t_repr(obj2) for
    all instances obj1, obj2 of this class.

    >>> t_repr("abc")
    'abc'

    >>> t_repr(range(5))
    [0, 1, 2, 3, 4]

    >>> import sys
    >>> d=t_repr({"a": [type(1), type("s")], "b": set([Ellipsis])})
    >>> d["a"]
    ["<type 'int'>", "<type 'str'>"]
    >>> d["b"]
    set(['Ellipsis'])

    >>> from django.utils.datastructures import MultiValueDict
    >>> md = t_repr(MultiValueDict({"a": [type(1), type("s")], "b": set([Ellipsis])}))
    >>> md
    MultiValueDict({'a': ["<type 'int'>", "<type 'str'>"], 'b': set(['Ellipsis'])})
    >>> md.getlist("a")
    ["<type 'int'>", "<type 'str'>"]
    >>> md["a"]
    "<type 'str'>"

    It's main use is for automatically creating tests. If you want to compare
    that a variable ``x`` in a test case evaluates to a complex value ``v``, but you
    cannot (automatically) find out how to create this complex value, you can
    generate this code::
        assert drepr(x) == %s
    substituting ``repr(drepr(value))`` for %s at code generation time.
"""

import types, re
from Cookie import BaseCookie
from django.utils.datastructures import MultiValueDict
from django.utils import functional
from django.utils.safestring import SafeString, SafeUnicode
from django.db.models.query import QuerySet
from django.oldforms import FormWrapper
from django.http import HttpResponse
from django.template.context import Context




# ignore these keys in context objects
context_ignore = ["LANGUAGES","LANGUAGE_CODE", "LANGUAGE_BIDI","sql_queries", "context", "debug", "perms", "MAILADMIN_ROOTURL", "AUTH_ROOTURL", 'RELEASENOTES_ROOTURL', 'POPKERN_ROOTURL', 'HARDWARE_ROOTURL', 'HISTORY_ROOTURL', 'KUNDE_ROOTURL', 'BASE_URL', 'USER_ROOTURL', 'SERVERSTATS_ROOTURL', 'ADD_SERVICE_LINKS', 'version', "staging", "otrs_enabled"]

# regulare expression, this is replaced in a repr() of unknown objects
t_repr_others_re = re.compile(r" object at 0x[0-f]+>$")



def t_repr(src):
    """returns a representation object for testing purposes. See module comments.
    This is the main dispatcher.
    """
    return _dispatch.get(type(src), t_repr_others)(src)

def t_repr_others(src):
    """dispatches objects"""
    if isinstance(src, QuerySet):
        return t_repr_list(src)
    elif isinstance(src, MultiValueDict):
        return t_repr_multi_value_dict(src)
    elif isinstance(src, (FormWrapper)):
        return t_repr_dict(src.__dict__)
    elif isinstance(src, BaseCookie):
        return t_repr_cookie(src)
    elif isinstance(src, HttpResponse):
        return t_repr_response(src)
    elif isinstance(src, Context):
        return t_repr_context(src)
    elif isinstance(src, functional.Promise):
        if src._delegate_unicode:
            return src.encode('utf-8')
        else:
            return str(src)
    elif isinstance(src, dict):
        return t_repr_dict(src)
    else:
        return t_repr_others_re.sub(" object>",repr(src))


def t_repr_response(response):
    data = dict(status_code=response.status_code,
                cookies=response.cookies)
    if response.has_header("Location"):
        data["location"]=response["Location"]
    return t_repr_dict(data)

def t_repr_context(context):
    result = {}
    for d in context:
        result.update(d)
    for k in context_ignore:
        if k in result:
            del result[k]
        if "csrfmiddlewaretoken" in result:
            result["csrfmiddlewaretoken"] = "self.csrfmiddlewaretoken"
    return t_repr_dict(result)


def identity(src):
    return src

def t_repr_tuple(src):
    return tuple([t_repr(x) for x in src])

def t_repr_list(src):
    return [t_repr(x) for x in src]

def t_repr_dict(src):
    return dict([(k, t_repr(v)) for k,v in src.iteritems()])

def t_repr_multi_value_dict(src):
    return MultiValueDictRepr(dict([(k, t_repr(v)) for k,v in src.lists()]))

def t_repr_set(src):
    return set([t_repr(x) for x in src])

def t_repr_cookie(src):
    return CookieRepr(src)

class CookieRepr(object):
    def __init__(self, src):
        self.cookie = src
    def __repr__(self):
        return "%s(%r)" % (self.cookie.__class__.__name__, self.cookie.output([], header=""))

class MultiValueDictRepr(MultiValueDict):
    def __repr__(self):
        return "MultiValueDict(%s)" % dict.__repr__(self)

_dispatch = {
    types.IntType: identity,
    types.LongType: identity,
    types.StringType: identity,
    types.UnicodeType: identity,
    types.BooleanType: identity,
    types.FloatType: identity,
    types.NoneType: identity,
    types.TupleType: t_repr_tuple,
    types.ListType: t_repr_list,
    types.DictionaryType: t_repr_dict,
    type(set()): t_repr_set,
    SafeString: identity,
    SafeUnicode: identity,
}


def _test():
    import doctest
    doctest.testmod()

if __name__ == "__main__":
    _test()



