# -*- encoding: UTF-8 -*-

import types, os, tempfile, re
from pprint import pprint, pformat
from django.dispatch import dispatcher
from django.core import signals
from django.core.handlers.base import BaseHandler
from django.http import HttpRequest, urlencode, SimpleCookie
from django.utils.datastructures import MergeDict
from django.utils.encoding import smart_str
from t_repr import t_repr

__all__ = ['TestRunner']

HEADER = "*" * 60
ignoreable = ([None, 0, '', [], {}, ' ', False, [None], [0], [''], [[]], [{}], [' '], [False]])

def _unicode_to_str(u):
    if isinstance(u, unicode):
        return u.encode('utf-8')
    else:
        return u
    
class TestRunner(object):
    def __init__(self):
        from django.test.client import Client
        self.client = Client()
        self.cookie_stack = []

    def get_response(self, path, method, GET, POST):
        """deprecated ... use get() or post() instead if possible"""
        if method=="GET":
            return self.get(path, GET)
        elif method=="POST":
            return self.post(path, POST, GET)
        else:
            raise NotImplementedError

    def get(self, path, data):
        response = self.client.get(path, data)
        self._set_context(response)
        return (response, self.context)

    def post(self, path, post_data, get_data=None):
        if get_data:
            extra = {'QUERY_STRING': urlencode(get_data)}
        else:
            extra = {}
        response = self.client.post(path, post_data, **extra)
        self._set_context(response)
        return (response, self.context)

    def _set_context(self, response):
        if response.context == None:
            self.context = None
        else:
            try:
                self.context = response.context[0]
            except KeyError:
                self.context = response.context

    def push_cookies(self):
        self.cookie_stack.append(self.client.cookies)

    def pop_cookies(self):
        from django.test.client import Client
        self.client = Client()
        self.client.cookies = self.cookie_stack.pop()

    def remove_cookies(self):
        self.client.cookies = SimpleCookie()


    def check_context(self, response, ist, soll):
        ist = _unicode_to_str(t_repr(ist))
        soll = _unicode_to_str(soll)
        
        # remove all the keys that evaluate to False and are not in soll:
        if ist == None:
            ist = {}
        try:
            if ist == soll:
                return True
        except UnicodeDecodeError:
            pass
        ist, soll, proposal = remove_ellipsis(ist, soll)
        if ist==soll:
            return True
        else:
            print "response is:"
            self.print_html(response)
            print HEADER
            print "context differs:"
            print HEADER
            print_diffs("context",ist,soll)
            print "Testvorschlag:\nassert runner.check_context(response, context, %s)" % pformat(t_repr(proposal))
            #print "IST"
            #pprint (ist)
            #print "\nSOLL"
            #pprint (soll)
            return False

    def check_response(self, ist, soll):
        result = ist.status_code == soll["status_code"]
        if not result:
            print "response is:"
            self.print_html(ist)
            print HEADER
            if ist.status_code != soll["status_code"]:
                print "* status_code ist=%d, soll=%d" % (ist.status_code, soll["status_code"])
            print "context:"
            pprint(self.context)
            print HEADER
        return result

    def check_datasource(self, response, soll):
        ist=response.content
        if ist == None:
            ist = {}
        else:
            ist = eval(ist)
        if ist == soll:
            return True
        else:
            remove_ellipsis(ist, soll)
            print "response is:"
            self.print_html(response)
            print HEADER
            print "datasource differs:"
            print HEADER
            print_diffs("content",ist,soll)
            return False

    def body_contains(self, response, soll):
        """Prüft, ob soll im body von response (als Substring) enthalten ist."""
        result = smart_str(soll) in response.content
        if not result:
            print "Body does not contain: %r" % soll
            self.print_html(response)
        return result

    def body_without(self, response, soll):
        """Prüft, ob soll im body von response (als Substring) nicht enthalten ist."""
        result = not smart_str(soll) in response.content
        if not result:
            print "Body *does* contain: %r" % soll
            self.print_html(response)
        return result

    def body_contains_link(self, response, url):
        """Prüft, ob der Body einen Link zur angegebenen url enthält"""
        regex = re.compile(r'''<[aA]([^>]|'[^']*'])*href\s*=\s*"%s"''' % re.escape(smart_str(url)), re.DOTALL)
        result = regex.search(response.content)
        if not result:
            print "Body does not contain link to url %r" % url
            self.print_html(response)
        return result

    def body_without_link(self, response, url):
        """Prüft, ob der Body keinen Link zur angegebenen url enthält."""
        regex = re.compile(r'''<[aA]([^>]|'[^']*'])*href\s*=\s*"%s"''' % re.escape(smart_str(url)), re.DOTALL)
        result = not regex.search(response.content)
        if result:
            print "Body *does* contain link to url %r" % url
            self.print_html(response)
        return result


    def print_html(self, response):
        fd, filename = tempfile.mkstemp('.html','viewtest-',text=True)
        f = os.fdopen(fd,"w")
        print "... writing html output to %s" % filename
        f.write(smart_str(response))
        f.close()
        #print HEADER
        #f = popen("w3m -dump -T text/html","w")
        #if f:
            #f.write(str(response))
            #f.close()


def print_diffs(prefix, ist, soll):
    if ist==None and soll!=None:
        print "* %s: missing" % prefix
    else:
        if hasattr(ist, "getlist"):
            get = ist.getlist
        else:
            get = ist.get
        for key, soll_value in soll.iteritems():
            key = _unicode_to_str(key)
            soll_value = _unicode_to_str(soll_value)
            if key not in ist:
                print "* %s.%s: missing, soll=%r" % (prefix,key, soll_value)
            else:
                ist_value = _unicode_to_str(get(key))
                if ist_value != soll_value:
                    if isinstance(soll_value,dict) and isinstance(ist_value,dict):
                        print_diffs(prefix+"."+key, ist_value, soll_value)
                    elif isinstance(soll_value,list) and isinstance(ist_value, list):
                        print_list_diffs(prefix+"."+key, ist_value, soll_value)
                    else:
                        print "* %s.%s:  ist=%r,\n* %s  soll=%r" % (prefix, key, ist_value, " " * (len(prefix)+len(key)+1), soll_value)
        for key, value in ist.iteritems():
            key = _unicode_to_str(key)
            value = _unicode_to_str(value)
            if key not in soll:
                print "* %s.%s: additional, ist=%r" % (prefix, key, value)

def print_list_diffs(prefix, ist, soll):
    if ist==None and soll!=None:
        print "* %s: missing" % prefix
    else:
        for key, (soll_value, ist_value) in enumerate(zip(soll, ist)):
            soll_value = _unicode_to_str(soll_value)
            ist_value = _unicode_to_str(ist_value)
            if ist_value != soll_value:
                if isinstance(soll_value,dict) and isinstance(ist_value,dict):
                    print_diffs("%s[%d]" % (prefix, key), ist_value, soll_value)
                elif isinstance(soll_value,list) and isinstance(ist_value, list):
                    print_list_diffs("%s[%d]" % (prefix, key), ist_value, soll_value)
                else:
                    print "* %s.%s:  ist=%r,\n* %s  soll=%r" % (prefix, key, ist_value, " " * (len(prefix)+len(str(key))+1), soll_value)
        for key, value in enumerate(ist[len(soll):]):
            value = _unicode_to_str(value)
            if key not in soll:
                print "* %s.%s: additional, ist=%r" % (prefix, key + len(soll), value)

def remove_ellipsis(ist, soll):
    """if soll is a dict, remove all items with a value of `Ellipsis`, from both ist and soll.
    Also remove all elements from ist that evaluate to False.
    If soll is a list, remove all elemens that equal `Ellipsis`, from both ist and soll.
    Then apply this recursively to all elements of ist/soll.

    It also converts all unicode values to bytestrings (deeply)

    It returns three values: ist, soll, proposal. proposal is a proposal for a new valid test
    that contains all the values from ist that passed Ellipsis removal, plus all the matching Ellipsis values.

    FIXME: This should be renamed to 'normalize()' and probably better modify values in-place only
           and only return the proposal

    >>> remove_ellipsis([1,2,3], [4,5, Ellipsis])
    ([1, 2], [4, 5], [1, 2, Ellipsis])

    >>> remove_ellipsis([],[])
    ([], [], [])

    >>> remove_ellipsis([{"bla":1, "blo":"asdasd"}], [Ellipsis])
    ([], [], [Ellipsis])

    >>> remove_ellipsis({"eins":1, "zwei":2}, {"drei": 3, "eins": Ellipsis})
    ({'zwei': 2}, {'drei': 3}, {'eins': Ellipsis, 'zwei': 2})

    >>> remove_ellipsis([1,2,{3:3, 4:4, 5:[6,7]}], [0,1,{5:[Ellipsis,2]}])
    ([1, 2, {3: 3, 4: 4, 5: [7]}], [0, 1, {5: [2]}], [1, 2, {3: 3, 4: 4, 5: [Ellipsis, 7]}])

    >>> remove_ellipsis(1,Ellipsis)
    (1, Ellipsis, 1)

    >>> remove_ellipsis(None, [Ellipsis])
    (None, [Ellipsis], None)

    >>> (a,b,c) = remove_ellipsis({1: None, 2: "", 3:True}, {1:None})
    >>> (1 in a, 2 in a, 3 in a, 1 in b, 2 in b, 3 in b, 1 in c, 2 in c, 3 in c)
    (True, False, True, True, False, False, True, True, True)
    """
    if isinstance(soll, dict) and isinstance(ist, dict):
        for key, value in ist.items():
            if isinstance(value, unicode):
                ist[key] = smart_str(value)
        proposal = ist.copy()
        for key, value in soll.items():
            if value == Ellipsis:
                del soll[key]
                try:
                    del ist[key]
                except KeyError:
                    pass
                proposal[key] = Ellipsis
            elif isinstance(value, (dict,types.ListType, dict)) and  key in ist:
                _1, _2, proposal[key] = remove_ellipsis(ist[key], value)
            elif isinstance(value, unicode):
                    value = smart_str(value)
                    soll[key] = value
        for key, value in ist.items():
            if value in ignoreable and key not in soll:
                del ist[key]
    elif isinstance(soll, types.ListType) and isinstance(ist, types.ListType):
        proposal = []
        deleteables = []
        for i, (soll_value, ist_value) in enumerate(zip(soll, ist)):
            if soll_value == Ellipsis:
                deleteables.insert(0,i)
                proposal.append(Ellipsis)
            elif isinstance(soll_value, (dict,types.ListType, dict)):
                _1, _2, p = remove_ellipsis(ist_value, soll_value)
                proposal.append(p)
            else:
                if isinstance(ist_value, unicode):
                    ist_value = smart_str(ist_value)
                    ist[i] = ist_value
                if isinstance(soll_value, unicode):
                    soll_value = smart_str(soll_value)
                    soll[i] = soll_value
                proposal.append(ist_value)
        for i in deleteables:
            del soll[i]
            del ist[i]
    else:
        proposal = ist
    return (ist,soll,proposal)

def _test():
    import doctest
    doctest.testmod()

if __name__ == "__main__":
    _test()

