# -*- encoding: UTF-8 -*-

from django.db import connection
from kundebunt.tests.fixtures import base_setup

class AsFany(object):
    def setup_class(cls):
        cls.get_runner().push_cookies()
        base_setup.login(cls.get_runner(), "fany")

    def teardown_class(cls):
        cls.get_runner().pop_cookies()

class WithManyDomains(object):
    """Testen mit vielen Domains für checkts"""
    def setup_class(cls):
        from kundebunt.popkern.models import Person, Kunde, Domainkunde, Mailrule
        from kundebunt.popkern import updatelog
        checkts = Person.objects.get(user='checkts')
        updatelog._log_person.person = checkts
        kunde_id = 277
        for i in range(100):
            Domainkunde.objects.create(kunde_id=kunde_id, domain='_test_%02d' % i,
                                       beginn=0, status_id=0, flags=0)

    def teardown_class(cls):
        from kundebunt.popkern.models import Person, Kunde, Domainkunde, Mailrule
        from kundebunt.popkern import updatelog
        Domainkunde.objects.filter(domain__startswith='_test_').delete()


class WithoutDomains(object):
    """Testen ohne Domains fuer checkts"""
    def setup_class(cls):
        from kundebunt.popkern.models import Person, Kunde, Domainkunde, Mailrule
        from kundebunt.popkern import updatelog
        checkts = Person.objects.get(user='checkts')
        updatelog._log_person.person = checkts
        cls.domains = list(Domainkunde.objects.all())
        for domain in cls.domains:
            domain.delete()

    def teardown_class(cls):
        from kundebunt.popkern.models import Person, Kunde, Domainkunde, Mailrule
        from kundebunt.popkern import updatelog
        checkts = Person.objects.get(user='checkts')
        for domain in cls.domains:
            domain.save()
        del cls.domains

class WithManyMailboxes(object):
    """Testen mit vielen Postfächern für checkts"""
    def setup_class(cls):
        from kundebunt.popkern.models import Person, Kunde, Domainkunde, Mailrule
        from kundebunt.popkern import updatelog
        checkts = Person.objects.get(user='checkts')
        updatelog._log_person.person = checkts
        kunde_id = 277
        for i in range(100):
            Person.objects.create(kunde_id=kunde_id, name='_test_%03d' % i,
                                  user='_test_%03d' % i,
                                  pwuse=Person.get_pwuse_bitmask('mail'))

    def teardown_class(cls):
        from kundebunt.popkern.models import Person, Kunde, Domainkunde, Mailrule
        from kundebunt.popkern import updatelog
        Person.objects.filter(user__startswith='_test_').delete()

class WithoutMailboxes(object):
    """Testen ohne mailbox fuer checkts"""
    def setup_class(cls):
        from kundebunt.popkern.models import Person, Kunde, Domainkunde, Mailrule
        from kundebunt.popkern import updatelog
        checkts = Person.objects.get(user='checkts')
        updatelog._log_person.person = checkts
        cls.mailboxes = list(Person.objects.has_flag('pwuse', 'mail'))
        for mailbox in cls.mailboxes:
            mailbox.pwuse = mailbox.pwuse & ~ Person.get_pwuse_bitmask("mail")
            mailbox.save()

    def teardown_class(cls):
        from kundebunt.popkern.models import Person, Kunde, Domainkunde, Mailrule
        from kundebunt.popkern import updatelog
        checkts = Person.objects.get(user='checkts')
        kunde_id = checkts.kunde.id
        for mailbox in cls.mailboxes:
            mailbox.pwuse = mailbox.pwuse | Person.get_pwuse_bitmask("mail")
            mailbox.save()
        del cls.mailboxes

class WithoutRules(object):
    """Testen ohne irgendwelche mailrules fuer checkts"""
    def setup_class(cls):
        from kundebunt.popkern.models import Person, Kunde, Domainkunde, Mailrule
        from kundebunt.popkern import updatelog
        checkts = Person.objects.get(user='checkts')
        updatelog._log_person.person = checkts
        cls.rules = list(Mailrule.objects.all())
        for rule in cls.rules:
            rule.delete()

    def teardown_class(cls):
        from kundebunt.popkern.models import Person, Kunde, Domainkunde, Mailrule
        from kundebunt.popkern import updatelog
        checkts = Person.objects.get(user='checkts')
        kunde_id = checkts.kunde.id
        for rule in cls.rules:
            rule.save()
        del cls.rules

class WithManyRules(object):
    """Testen mit vielen mailrules für checkts"""
    def setup_class(cls):
        from kundebunt.popkern.models import Person, Kunde, Domainkunde, Mailrule
        from kundebunt.popkern import updatelog
        checkts = Person.objects.get(user='checkts')
        updatelog._log_person.person = checkts
        kunde_id = 277  # irgendein kunde, zu dem checkts assoziiert ist. checkts.kunde_id ist nicht unbedingt assoziiert.
        for i in range(100):
            Mailrule.objects.create(kunde_id=kunde_id, quelle='_test_%02d' % i,
                                    typ_id=Mailrule.get_typ_code('virt'),
                                    ziel='_test_%02d' % i)

    def teardown_class(cls):
        from kundebunt.popkern.models import Person, Kunde, Domainkunde, Mailrule
        from kundebunt.popkern import updatelog
        Mailrule.objects.filter(quelle__startswith='_test_').delete()

class WithMultipleCustomers(object):
    """Testen mit mehreren Unterkunden für checkts"""
    def setup_class(cls):
        from django.db import connection, transaction
        cursor = connection.cursor()
        cursor.execute("insert into kundemail (dienst, kunde, person) values (1254, 744,3381)")
        cursor.execute("insert into kundemail (dienst, kunde, person) values (1254, 1014,3381)")
        transaction.commit()


    def teardown_class(cls):
        from django.db import connection, transaction
        cursor = connection.cursor()
        cursor.execute("delete from kundemail where person=3381 and kunde in (744, 1014)")
        transaction.commit()

class WithoutCustomers(object):
    """Testen ohne Unterkunden"""
    def setup_class(cls):
        from django.db import connection, transaction
        from kundebunt.popkern.models import Kundemail
        cls.kundemails = list(Kundemail.objects.all())
        cursor = connection.cursor()
        cursor.execute("delete from kundemail")
        transaction.commit()

    def teardown_class(cls):
        for kundemail in cls.kundemails:
            kundemail.save()
        del cls.kundemails

class AsStaff(object):
    """Testen mit checkts als staff"""
    def setup_class(cls):
        from kundebunt.popkern.models import Person, Kunde, Domainkunde, Mailrule
        from kundebunt.popkern import updatelog
        checkts = Person.objects.get(user='checkts')
        updatelog._log_person.person = checkts
        cls.kunde_id = checkts.kunde_id
        cls.pwuse = checkts.pwuse
        checkts.kunde_id = 1
        checkts.pwuse |= Person.get_pwuse_bitmask("pop")
        checkts.save()

    def teardown_class(cls):
        from kundebunt.popkern.models import Person, Kunde, Domainkunde, Mailrule
        from kundebunt.popkern import updatelog
        checkts = Person.objects.get(user='checkts')
        checkts.kunde_id = cls.kunde_id
        checkts.pwuse = cls.pwuse
        checkts.save()
        del cls.kunde_id

class WithoutServiceFlag(object):
    def setup_class(cls):
        from kundebunt.popkern.models import Person
        from kundebunt.popkern import updatelog
        checkts = Person.objects.get(user='checkts')
        updatelog._log_person.person = checkts
        cls.old_pwuse = checkts.pwuse
        checkts.pwuse = 0
        checkts.save()

    def teardown_class(cls):
        from kundebunt.popkern.models import Person
        from kundebunt.popkern import updatelog
        checkts = Person.objects.get(user='checkts')
        updatelog._log_person.person = checkts
        checkts.pwuse = cls.old_pwuse
        checkts.save()

class WithoutServiceAssociation(object):
    def setup_class(cls):
        from kundebunt.popkern.models import Kundemail
        cursor = connection.cursor()
        cls.service_dienst_id = Kundemail.get_dienst_code("service")
        cursor.execute('update kundemail set dienst=0 where dienst=%s', [cls.service_dienst_id])

    def teardown_class(cls):
        cursor = connection.cursor()
        cursor.execute('update kundemail set dienst=%s where dienst=0', [cls.service_dienst_id])
