# -*- encoding: utf-8 -*-
"""Utilities zum Erzeugen der Testdaten"""

import sys
from copy import copy
from itertools import chain
from kundebunt.popkern import models
from kundebunt.tests.fixtures.serializer import serialize_json


def model_data(model):
    """returns a dict with the field data of model, ready to inject again by manipulator.save()"""
    modeldict = model.__dict__
    return dict([(f.name, modeldict[f.attname]) for f in model._meta.fields])

def dump_queryset(qset, f=sys.stdout):
    """writes to f a list of dicts that represent the models in the query set"""
    from pprint import pprint
    f.write("[\n")
    for model in qset:
        pprint(model_data(model),f)
        f.write(",\n")
    f.write("]\n")

def dump_data(qset1, qset2, f=sys.stdout):
    """schreibt direkt den ganzen BLABLA_DATA=(ModelClass, initial_data, rest_data)-Kram"""
    f.write("%s_DATA=(models.%s,\n" %
             (qset1.model._meta.object_name.upper(), qset1.model._meta.object_name))
    dump_queryset(qset1,f)
    f.write(",\n")
    dump_queryset(qset2,f)
    f.write(")\n")


def dump_all_data(f=sys.stdout):
    """Schreibt die Daten raus. Das Minimalset enthält:

    * Alle Descriptoren
    * Kunden POP, sluka, test, wally
    * Person fany
    * Kundemail sluka<->fany
    * Domainkunde checkts.net
    * Mailrule (mx) sluka.de -> mx.sluka.de
    """
    f.write("from popkern import models\n")
    f.write("import datetime\n")
    dump_data(models.DescrTyp.objects.all(), [], f)
    dump_data(models.Descr.objects.all(), [], f)
    dump_data(models.Kunde.objects.filter(name__in=["POP","sluka","test","wally"]), [], f)
    dump_data(models.Person.objects.filter(user="fany"),
              models.Person.objects.filter(kunde__name__in=["POP","test","wally"]).exclude(user="sluka"), f)
    dump_data(models.Kundemail.objects.filter(kunde__name__exact="sluka", person__user__exact="fany"),
              models.Kundemail.objects.filter(kunde__name__in=["POP","sluka","test","wally"])
                                      .exclude(kunde__name__exact="sluka", person__user__exact="fany"), f)
    dump_data(models.Domainkunde.objects.filter(domain="checkts.net"),
              models.Domainkunde.objects.filter(kunde__name__in=["POP","sluka","test","wally"]).exclude(domain="checkts.net"), f)
    dump_data(models.Mailrule.objects.filter(kunde__name__exact="sluka", quelle="sluka.de"),
              models.Mailrule.objects.filter(kunde__name__in=["POP","test","wally","fany"]).exclude(quelle="sluka.de"), f)

def serialize(qset, f):
    f.write(serializers.serialize(qset, indent=4))
    
def dump_customer(customer_ids, filename):
    def _add_mperson(cand):
        """Fügt fehlende über mperson referenzierte Personen hinzu. Rekursiv."""
        mperson_ids = [p.mperson_id for p in cand if p.mperson_id and p.mperson_id not in person_ids]
        if mperson_ids:
            person_ids.update(mperson_ids)
            neue_personen = list(models.Person.objects.filter(id__in=mperson_ids).order_by("meperson"))
            personen[0:0] = neue_personen
            _add_mperson(neue_personen)

    # init_kunden enthält erst mal die Kunden ohne hauptkunde und billc,
    # mysql prüft ja dummerweise foreign key constraints beim Insert.
    kunden = list(models.Kunde.objects.filter(id__in=customer_ids))
    init_kunden = []
    for k in kunden:
        k2 = copy(k)
        k2.hauptperson = k2.billc = k2.ap_vertrieb = k2.ap_technik = k2.adminc_id = None
        init_kunden.append(k2)
    kundemail = list(models.Kundemail.objects.filter(kunde__in=customer_ids))
    domainkunden = list(models.Domainkunde.objects.filter(kunde__in=customer_ids))
    mailrules = list(models.Mailrule.objects.filter(kunde__in=customer_ids))
    tarife = list(models.Dienst.objects.filter(ipkunde__hardware__kunde__in=customer_ids))
    ipregionen = list(models.IpRegion.objects.filter(ipkunden_by_allocation__hardware__kunde__in=customer_ids))
    ipkunden = list(models.Ipkunde.objects.filter(hardware__kunde__in=customer_ids))
    rzs = list(models.RZ.objects.filter(rack__hardware__kunde__in=customer_ids))
    racks = list(models.Rack.objects.filter(hardware__kunde__in=customer_ids))
    hardware = list(models.Hardware.objects.filter(kunde__in=customer_ids))
    
    person_ids = set(models.Person.objects.filter(kunde__in=customer_ids).valuelist("id"))
    person_ids.update(chain(* ((k.hauptperson_id, k.billc_id, k.ap_technik_id, k.ap_vertrieb_id, k.adminc_id) for k in kunden)))
    person_ids.update(km.person_id for km in kundemail)
    person_ids.update(chain(* ((hw.standort_id, hw.eigentuemer_id, hw.verantwortlich_id) for hw in hardware)))
    person_ids.update(chain(* ((d.person_id, d.owner_id, d.adminc_id, d.techc_id, d.zonec_id, d.billc_id) for d in domainkunden)))
    person_ids.update(rz.standort_id for rz in rzs)
    person_ids.discard(None)
    personen = list(models.Person.objects.filter(id__in=person_ids).order_by("mperson"))
    _add_mperson(personen)

    adressen = list(models.Adresse.objects.filter(id__in=[p.adresse_id for p in personen if p.adresse]))
    laender_ids = set(adr.land_id for adr in adressen if adr.land_id)
    laender = list(models.Land.objects.filter(id__in=list(laender_ids)))

    f = open(filename, "w")
    f.write(serialize_json(
        chain(init_kunden, laender, adressen, personen, kunden, kundemail, domainkunden, mailrules,
              tarife, ipregionen, ipkunden, rzs, racks, hardware),
        indent=4))
    f.close()
    
