# -*- coding: UTF-8 -*-

from itertools import chain

__all__ = ['HardwareFixture', 'WithInactiveHardware', 'WartungsvertragFixture']


def setup_log_session():
    from kundebunt.popkern import updatelog, models
    checkts = models.Person.objects.get(user='checkts')
    updatelog._log_person.person = checkts


class HardwareFixture(object):
    """Erzeugt den Rahmen für Hardware-Tests: RZ, Rack, Tarif, Ipkunde, Hardware
    """
    def setup_class(self):
        from kundebunt.popkern import models
        from django import db
        cursor = db.connection.cursor()
        cursor.execute("alter table hardware auto_increment=1")
        cursor.execute("alter table rack auto_increment=1")

        setup_log_session()
        self.ips = []
        self.racks = []
        self.rz_persons = []
        self.rzs = []
        sluka_kunde = models.Kunde.objects.get(name="sluka")
        checkts_person = models.Person.objects.get(user="checkts")
        fany = models.Person.objects.get(user="fany")
        for rz_nr in (1,2):
            rz_person = models.Person.objects.create(kunde_id=1, user='rz%d' % rz_nr, pwuse=0)
            self.rz_persons.append(rz_person)
            rz = models.RZ.objects.create(name='RZ%d' % rz_nr, standort=rz_person)
            self.rzs.append(rz)
            for rack_nr in (1,2):
                self.racks.append(models.Rack.objects.create(rz=rz, name='rack%d' % rack_nr, he=42, kunde_id=1, x=rack_nr, y=1))
        self.tarif = models.Dienst.objects.create(name="dummytarif")
        for ip, name in (("192.168.0.1", "eins"), ("192.168.0.2", "zwei"),
                         ("192.168.128.3", "drei"), ("62.0.0.4", "vier"),
                         ("192.168.3.8", None)):
            self.ips.append(models.Ipkunde.objects.create(
                    kunde_id=1, name=name, ip6=models.Ipkunde.db_value_for_ip6(ip), bits=0,
                    beginn=0, tarif=self.tarif,
                    dest=models.Ipkunde.get_dest_descr('welt'), flags=0))
        hw1 = models.Hardware.objects.create(kunde_id=1, name='Fubrikator 001', hardware_id="D000001", ivnr="000001", ip=self.ips[0],
                seriennr='A1', beginn=0, rack=self.racks[0], unterste_he=0, he=5, flags=0)
        self.hardware = [hw1,
            models.Hardware.objects.create(kunde_id=1, name='Enoraix 002', hardware_id="D000002", ivnr="000002", ip=self.ips[1],
                    seriennr='B2', beginn=0, rack=self.racks[3], unterste_he=5, he=2, flags=0),
            models.Hardware.objects.create(kunde=sluka_kunde, name='Deusexmachina 003',
                    seriennr='C3', beginn=0, standort=fany, eigentuemer=fany,
                    verantwortlich=fany, info="wichtig", flags=0),
            models.Hardware.objects.create(kunde_id=1, name="Caesar 004", hardware_id="D000004", ivnr="000004", ip=self.ips[2],
                    seriennr=None, beginn=0, rack=self.racks[1], unterste_he=7, he=2, verantwortlich=fany,
                    status_id=models.Hardware.get_status_code("reserviert"), flags=0),
            models.Hardware.objects.create(kunde_id=1, name="Brunsbüttel 005", hardware_id=None, ivnr="000005", ip=self.ips[3],
                    seriennr="E5", beginn=0, enthalten_in=hw1,
                    typ_id=models.Hardware.get_typ_code("Adva"), klasse_id=models.Hardware.get_klasse_code("Access"),
                    lieferant_id=models.Hardware.get_lieferant_code("KK Computer"), flags=0)
            ]

    @classmethod
    def teardown_class(self):
        from django import db
        for model in chain(reversed(self.hardware), self.ips, [self.tarif], self.racks, self.rzs, self.rz_persons):
            model.delete()
        db.transaction.commit()


class WithInactiveHardware(object):
    """Setzt eine Hardware (id 1) auf inaktiv. Muss innerhalb einer HardwareFixture geschachtelt vorkommen.
    """
    def setup_class(self):
        from kundebunt.popkern import models
        hw = models.Hardware.objects.get(pk=1)
        self.hw_ende = hw.ende
        hw.ende = 1
        hw.save()

    def teardown_class(self):
        from kundebunt.popkern import models
        hw = models.Hardware.objects.get(pk=1)
        hw.ende = self.hw_ende
        hw.save()

class WartungsvertragFixture(object):
    """Wartungsvertrags(-Templates). Für Hardware-Wartungsverträge wird zusätzlich die
    HardwareFixture benötigt.
    """
    @classmethod
    def setup_class(self):
        from kundebunt.popkern import models
        from django import db
        cursor = db.connection.cursor()
        cursor.execute("alter table wartungsvertrag auto_increment=1")

        setup_log_session()
        dell_ap = models.Person.objects.create(
            kunde_id=1, user="dell_ap", pwuse=0, name="Herr Dell", email="support@dell.local")
        hp_ap = models.Person.objects.create(
            kunde_id=1, pwuse=0, user="hp_ap", name="Herr H. P.", fon="+49 911 999")
        self.dell = models.Wartungsvertrag.objects.create(
            name="Dell Granit Support", sla="1x1", ansprechpartner=dell_ap,
            beschreibung="Dell ... Support ... Blablabla")
        self.hp = models.Wartungsvertrag.objects.create(
            name="HP Gips Support", sla="8x8", ansprechpartner=hp_ap,
            beschreibung="HP ... auch Support ... Blablabla")

    @classmethod
    def teardown_class(self):
        self.dell.delete()
        self.hp.delete()
        del self.dell
        del self.hp
