import os.path, urllib2, datetime, calendar, pytz, threading
from itertools import chain, count
from time import sleep
from urlparse import urlparse

from django.http import HttpResponse, Http404, HttpResponseRedirect, urlencode
from django.template import loader, RequestContext
from django.shortcuts import render_to_response
from django.conf import settings
from django.core.cache import cache
from django.utils import simplejson
from django.utils.translation import ugettext as _, ugettext_lazy as __
#from django.utils.encoding import force_unicode
#from django.utils.safestring import mark_safe
#from django.utils.translation import ugettext_lazy

from kundebunt import newforms
from kundebunt.popkern import models
from kundebunt.popkern.utils import any, all
from kundebunt.kunde_auth.decorators import service_flag_required, person_has_service_association
from kundebunt.kunde_auth.views import login

DAY = 24*3600

EXPLICIT_END = 'DATUM'

DELTA_CHOICES = (
                 (1800, __(u'30 Minuten')),
                 (3600, __(u'1 Stunde')),
                 (6*3600, __(u'6 Stunden')),
                 (DAY, __(u'1 Tag')),
                 (7*DAY, __(u'1 Woche')),
                 (28*DAY, __(u'4 Wochen')),
                 (365*DAY, __(u'1 Jahr')),
                 (EXPLICIT_END, __(u'(bis anderes Datum)')),
                )

local_tz = pytz.timezone(settings.TIME_ZONE)
utc_tz = pytz.utc

def epoch(t):
    """Ermittelt die GMT-Epoch zur lokalen (naiven) Zeit t. Korrigiert eine
    lokale Zeit innerhalb einer Zeitumstellung. Gibt None für None zurück.
    """
    def _epoch(t):
        return calendar.timegm(utc_tz.normalize(local_tz.localize(t, is_dst=True).astimezone(utc_tz)).timetuple())
    if t is None:
        return None
    try:
        return _epoch(t)
    except (ValueError, IndexError), err:
        # Falls wir im Zeitloch einer Zeitumstellung Normal->DST sind,
        # einfach mal 3:00 Uhr probieren.
        t = datetime.datetime(year=t.year, month=t.month, day=t.day,
                              hour=3, minute=0, second=0, tzinfo=t.tzinfo)
        return _epoch(t)


class ServerStatsForm(newforms.Form):
    hostname = newforms.MultipleChoiceField(label=__(u'Server-Name (FQDN)'), required=False)
    plugin = newforms.DynamicMultipleChoiceField(label=__(u'Art'), required=False)
    delta = newforms.ChoiceField(label=__(u'Zeitraum'), choices=DELTA_CHOICES, initial=DAY, required=True)
    #start = newforms.SplitDateTimeField(label=_(u'Start'), required=False, widget=newforms.SplitDateTimeWidget)
    start_quick = newforms.ChoiceField(label=__(u'Beginn'), required=True, initial='heute')
    start_day = newforms.DateField(label=__(u'Beginn'), required=True, initial=datetime.date.today(), widget=newforms.SelectDateWidget)
    start_hour = newforms.TimeField(label=__(u'Uhrzeit (hh:mm)'), required=False, widget=newforms.TextInput(attrs={'style': 'width:5em'}))
    stop_day = newforms.DateField(label=__(u'Ende'),
                                  required=True,
                                  initial=datetime.date.today(),
                                  widget=newforms.SelectDateWidget)
    stop_hour = newforms.TimeField(label=__(u'Uhrzeit (hh:mm)'), required=False, widget=newforms.TextInput(attrs={'style': 'width:5em'}))

    def __init__(self, hostnames, plugins, *args, **kwargs):
        super(ServerStatsForm, self).__init__(*args, **kwargs)
        self.fields['hostname'].choices = ((x,x) for x in hostnames)
        self.fields['plugin'].choices = ((x,x) for x in plugins)
        heute = datetime.date.today()
        tage = (heute - datetime.timedelta(days=i) for i in range(1,8))
        quick_choices = (
                [('today', _(u'heute'))]
                + [(d.isoformat(), d.strftime('%d.%m.%Y')) for d in tage]
                + [('other', _(u'(anderes Datum)'))])
        self.fields['start_quick'].choices = quick_choices
        self.start_quick_other_index = len(quick_choices) - 1
        if len(hostnames) == 1:
            self.fields['hostname'].initial = hostnames


class CollectdHost(object):
    
    @classmethod
    def get_hosts(cls):
        """Return current list of hosts as list of CollectdHost objects.
        Deal with the cache as appropriate """
        for i in [0,1]:
            try:
                entries = cache.get('kundebunt.serverstats.collect_hosts.v2')
            except UnicodeDecodeError:
                # workaround, see RT#464392
                entries = None
            if entries:
                cls._handle_cache_refresh()
            else:
                # no data in cache.
                if cache.add('kundebunt.serverstats.collect_hosts_min', True, settings.SERVERSTATS_HOSTLIST_CACHE_MIN) or i > 0:
                    entries = cls.populate_host_cache()
                else:
                    sleep(15)   # wait for other process to fill cache
            if entries:
                return [cls(*data) for data in entries]

    @classmethod
    def populate_host_cache(cls):
        """Actually fetch host list from collectd server and put it into the cache.
        Return fetched entries as list of tuples (hostname, plugins).
        """
        f = urllib2.urlopen(settings.COLLECTD_LIST_URL)
        host_list_doc = f.read().decode('iso-8859-1','replace')
        f.close()
        entries = simplejson.loads(host_list_doc).items()
        cache.set('kundebunt.serverstats.collect_hosts.v2', entries, settings.SERVERSTATS_HOSTLIST_CACHE_MAX)
        cache.set('kundebunt.serverstats.collect_hosts_min', True, settings.SERVERSTATS_HOSTLIST_CACHE_MIN)
        return entries

    @classmethod
    def _handle_cache_refresh(cls, force=False):
        if cache.add('kundebunt.serverstats.collect_hosts_min', True, settings.SERVERSTATS_HOSTLIST_CACHE_MIN):
            thread = threading.Thread(target=CollectdHost.populate_host_cache, args=())
            thread.start()

    def __init__(self, hostname, plugins):
        self.hostname = hostname
        self.plugins = plugins

    def __repr__(self):
        return "<CollectdHost '%s'>" % self.hostname

    def _get_ipkunden(self):
        """returns the ipkunde entries for this host (acc. to ipkunde).
        Returns None if it does not exist. Cached.
        """
        if not hasattr(self, '_ipkunden'):
            self._ipkunden = list(models.Ipkunde.objects.active().filter(name=self.hostname).select_related(follow_fields=('kunde',)))
        return self._ipkunden

    def _get_kunden(self):
        """returns the kunden for this host, or None if it does not exist. Uses cache of _get_ipkunde."""
        if self.ipkunden is None:
            return None
        else:
            return [ip.kunde for ip in self.ipkunden]

    def get_urls(self, plugin):
        urls = sorted(self.plugins.get(plugin, []))
        if settings.SERVERSTATS_REWRITE_GRAPH_URL:
            return ["%s?%s" % (settings.SERVERSTATS_REWRITE_GRAPH_URL, urlparse(url)[4])
                    for url in urls]
        else:
            return urls

    ipkunden = property(_get_ipkunden, doc="models.Ipkunden for this host, None if it does not exist.")
    kunden = property(_get_kunden, doc="models.Kunden for this host, None if it does not exist.")


@service_flag_required
def overview(request, kunden_name=None):
    if 'force_reload' in request.GET and request.method=='GET':
        CollectdHost.populate_host_cache()
    available_hosts = CollectdHost.get_hosts()
    if kunden_name:
        try:
            kunde = models.Kunde.objects.active().get(name=kunden_name)
        except models.Kunde.DoesNotExist:
            raise Http404
    else:
        kunde = None
    if request.person.is_staff():
        if kunde:
            available_hosts = [h for h in available_hosts if kunde in h.kunden]
    else:
        if kunden_name and not request.person.managed_kunden().filter(id=kunde.id).exists():
            raise Http404
        if kunde:
            available_hosts = [h for h in available_hosts if kunde in h.kunden]
        else:
            kunden = set(request.person.managed_kunden())
            available_hosts = [h for h in available_hosts if any(k in kunden for k in h.kunden)]
    if not available_hosts:
        return render_to_response(
            "serverstats/no_hosts_assigned.html",
            context_instance=RequestContext(request, {'kunden_name': kunden_name}))
    host_dict = dict((h.hostname, h) for h in available_hosts)
    available_hosts.sort(key=lambda h: h.hostname)
    if request.POST.getlist("hostname"):
        plugins = list(sorted(set(chain(* (host_dict[h].plugins.iterkeys() for h in request.POST.getlist("hostname") if h in host_dict)))))
    else:
        plugins = list(sorted(set(chain(* (h.plugins.iterkeys() for h in available_hosts)))))

    if request.POST:
        form = ServerStatsForm(hostnames=[h.hostname for h in available_hosts], plugins=plugins, data=request.POST)
    else:
        form = ServerStatsForm(hostnames=[h.hostname for h in available_hosts], plugins=plugins)
    if form.is_valid():
        data = form.cleaned_data
        # rows: [[[(id,url), ...]]]   (1 row per plugin, 1 column per host, each a list of pairs (id,url))
        #bla = host_dict[data["hostname"][0]].plugins.get('cpu')
        #bla2=data.get("plugin")
        #bla3=data["hostname"]
        selected_plugins = data["plugin"]
        id_counter = count()
        rows = ([[("graph%04d" % id_counter.next(), u) for u in host_dict[hostname].get_urls(plugin)]
                 for hostname in data["hostname"]]
                for plugin in selected_plugins)
        # remove empty rows
        rows = (row for row in rows if any(any(urls for urls in col) for col in row))

        today = datetime.date.today()
        if data['start_quick'] == 'today':
            begin = today
        elif data['start_quick'] == 'other':
            begin = data['start_day']
        else:
            begin = datetime.date(*(int(x) for x in data['start_quick'].split('-')))
        if data.get('start_hour'):
            begin = datetime.datetime.combine(begin, data.get('start_hour'))
        elif begin == today:
            if data.get('delta') == EXPLICIT_END:
                begin = datetime.datetime.combine(begin, datetime.time())
            else:
                begin = datetime.datetime.now()
        else:
            begin = datetime.datetime.combine(begin, datetime.time())
        if data.get('delta') == EXPLICIT_END:
            end = data.get('stop_day')
            if data.get('stop_hour'):
                end = datetime.datetime.combine(end, data.get('stop_hour'))
            elif end == today:
                end = datetime.datetime.now()
            else:
                end = datetime.datetime.combine(end, datetime.time())
            begin_epoch = epoch(begin)
            end_epoch = epoch(end)
            url_suffix = u";begin=%d;end=%d" % (begin_epoch, end_epoch)
        else:
            #delta = datetime.timedelta(seconds=int(data.get('delta')))
            begin_epoch = epoch(begin)
            delta = int(data.get('delta'))
            end = datetime.datetime.now()
            end_epoch = epoch(end)
            if begin_epoch + delta >= end_epoch:
                url_suffix = u";begin=-%s" % data['delta']
                begin_epoch = end_epoch - delta
            else:
                end_epoch = begin_epoch + delta
                url_suffix = u";begin=%d;end=%d" % (begin_epoch, end_epoch)
    else:
        rows = ()
        begin_epoch = end_epoch = data = url_suffix = None
    template = loader.get_template("serverstats/overview.html")
    content = template.render(RequestContext(
            request,
            {'form': form,
             'rows': list(rows),
             'with_table': data and len(data["hostname"]) > 1,
             'hostnames': data and data["hostname"],
             'url_suffix': url_suffix,
             'kunden_name': kunden_name,
             'EXPLICIT_END': len(DELTA_CHOICES) - 1,
             'OTHER_START': form.start_quick_other_index,
             'begin': begin_epoch,
             'end': end_epoch,
            }))
    return HttpResponse(content)


def check_graph_access(request):
    """FastCGIAccessChecker - erlaubt den Zugriff, wenn die Person zur Session zu irgend einem
    Kunden eine service-Assoziierung hat, und die ausgewählten `hostnames` zu assoziierten Kunden
    Zugriff hat. Andernfalls wird Status 403 zurückgegeben.
    In letzterm Fall wird dann sinnvollerweise über ein zu konfigurierendes Apache-ErrorDocument
    die Loginseite aufgerufen.
    """
    if request.person and request.person.is_staff():
        return HttpResponse(u"OK", content_type="text/plain")
    hostnames = request.REQUEST.getlist('hostname')
    if hostnames and person_has_service_association(request.person):
        kunden_ids = request.person.managed_kunden().valuelist("id")
        hosts = [h for h in CollectdHost.get_hosts() if h.hostname in hostnames]
        if hosts and all(any(hk.id in kunden_ids for hk in h.kunden) for h in hosts):
            return HttpResponse(u"OK", content_type="text/plain")
    # Against specification, the apache forwards the content-length header
    # of response of the fastcgi access checker, so it should match the
    # final response (which is the login view, by ErrorDocument)
    response = login(request)
    response.status_code = 403
    return response
