# -*- coding: UTF-8 -*-

from django import db
from django.db import models
from django.db.models.query import Q
from django.conf import settings
from django.utils.translation import gettext, gettext_lazy, ngettext, get_language

import kundebunt.popkern.models
from kundebunt import version
from kundebunt.popkern.navigation import navi_url

NOTE_KIND_CHOICES = ((10, 'hint'),
                     (20, 'feature'),
                     (30, 'bug'),
                     (40, 'critical'))

LANGUAGE_CHOICES = (('intern','intern'),) + settings.LANGUAGES

def current_release():
    return version.version.split('-')[0]

def version_sort_value(version_string):
    if version_string=='UNRELEASED':
        return 99999999
    vlist = (version_string.split("-")[0].split(".") + [0,0,0])[:4]
    res = 0
    for v in vlist:
        res = res * 100 + int(v)
    return res

def get_effective_notes(internal_request):
    version = version_sort_value(kundebunt.version.version)
    notes = Note.objects.select_related().extra(
        joins=['left outer join %s rel1 on rel1.id=%s.release_beginn_id' % (Release._meta.db_table, Note._meta.db_table),
                'left outer join %s rel2  on rel2.id=%s.release_behoben_id' % (Release._meta.db_table, Note._meta.db_table)],
        where=['((%(notes_table)s.release_beginn_id is null or rel1.sort_value <= %(sort_value)d) '
                'and (%(notes_table)s.release_behoben_id is null or rel2.sort_value > %(sort_value)d))'
                % {'sort_value': version, 'notes_table': Note._meta.db_table}],
        ).select_related()
    notes = list(notes)
    delete_list = []
    for i, note in enumerate(notes):
        if not note.text(internal=internal_request):
            delete_list.append(i)
    delete_list.reverse()
    for i in delete_list:
        del notes[i]
    return notes

def get_or_create_release_id(version_string):
    if version_string:
        release, created = get_or_create_release(version_string)
        return release.id
    else:
        return None

def get_or_create_release(version_string):
    vlist = ([int(x) for x in version_string.split("-")[0].split(".")])
    if len(vlist)==4:
        v1,v2,v3,v4 = vlist
        return Release.objects.get_or_create(v1=v1, v2=v2, v3=v3, v4=v4)
    else:
        assert len(vlist)==3
        v1,v2,v3 = vlist
        return Release.objects.get_or_create(v1=v1, v2=v2, v3=v3, v4__isnull=True)


class Release(models.Model):
    """
    Verwendung: Eine Release, zur Anzeige der Versionshinweise in kundebunt.
    
    Index:  id, sort_value
    Aufbau:
    id          int4    !?+ ID
    modified    datetime  -     Zeitpunkt der letzten Änderung
    v1          uint2   -       v1.v2.v3.v4 ist die Versionsnummer
    v2          uint2   -       v1.v2.v3.v4 ist die Versionsnummer
    v3          uint2   -       v1.v2.v3.v4 ist die Versionsnummer
    v4          uint2   !-      v1.v2.v3.v4 ist die Versionsnummer
    sort_value  uint4   -       Sortierwert
    timestamp   datetime    !/+ Zeitstempel
    """
    modified = models.DateTimeField(auto_now=True)
    v1 = models.PositiveSmallIntegerField()
    v2 = models.PositiveSmallIntegerField()
    v3 = models.PositiveSmallIntegerField()
    v4 = models.PositiveSmallIntegerField(null=True, blank=True)
    sort_value = models.PositiveIntegerField(editable=False, unique=True)
    timestamp = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ("sort_value",)
        unique_together = (("v1","v2","v3","v4"), )
        verbose_name_plural='Releases'
    class Admin:
        list_display = ("v1", "v2", "v3", "v4")

    def __str__(self):
        if self.v4 == None:
            v4 = ""
        else:
            v4 = ".%d" % self.v4
        return "%d.%d.%d%s" % (self.v1, self.v2, self.v3, v4)

    def set_sort_value(self):
        sort_value = self.v1*1000000 + self.v2*10000 + self.v3*100
        if self.v4 != None:
            sort_value += self.v4
        self.sort_value = sort_value

    def save(self):
        self.set_sort_value()
        return super(Release, self).save()


class Note(models.Model):
    """
    Verwendung: Versionshinweis zur Anzeige in kundebunt vor dem Einloggen.
    
    Index:  id
    Index:  release_beginn_id, release_behoben_id
    Aufbau:
    id                  int4    !?+
    modified            datetime    !/+                     Zeitpunkt der letzten Modifikation
    created             datetime                            Zeitpunkt des Anlegens
    ticket              uint4   -                           Zugehöriges Ticket
    kind                uint2   -                           Art des Releases
    release_beginn_id   int4    !>releasenotes_release!-    Ab welcher Release der Fehler besteht
    release_behoben_id  int4    !>releasenotes_release!-    Ab welcher der Fehler behoben ist
    last_modified_by    int4    !-                          Wer den Eintrag zuletzt editiert hat (person)
    timestamp           datetime !/+                        Zeitstempel
    """
    modified = models.DateTimeField(auto_now=True)
    created = models.DateTimeField(auto_now_add=True)
    ticket = models.PositiveIntegerField()
    kind = models.PositiveSmallIntegerField(choices = NOTE_KIND_CHOICES)
    release_beginn = models.ForeignKey(Release, null=True, blank=True, related_name='releases_beginn')
    release_behoben = models.ForeignKey(Release, null=True, blank=True, related_name='releases_behoben')
    last_modified_by = models.IntegerField(null=True, blank=True, editable=False) # --> person.id
    timestamp = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name = gettext_lazy("Versionshinweis")
        verbose_name_plural = gettext_lazy('Versionshinweise')
        ordering = ("-kind", "-modified")
        unique_together = (('ticket', 'release_behoben'), ('ticket', 'release_beginn'))
    class Admin:
        list_display = ("ticket", "kind", "release_beginn", "release_behoben", "summary")
        search_fields = ("ticket",)
        list_filter = ("kind", "release_beginn", "release_behoben")
        ordering = ("-ticket",)
        list_select_related = True

    def __repr__(self):
        return "<Note for ticket %d, versions(%s-%s)" % (
            self.ticket, self.release_beginn, self.release_behoben)

    def navi_context(self):
        return {'ticket': self.ticket }

    def summary(self):
        try:
            return self.text(internal=True)[:80]
        except IndexError:
            return ''

    def text(self, internal=False):
        text_dict = dict(((text.language, text) for text in self.notetext_set.all()))
        if internal and 'intern' in text_dict:
            return text_dict['intern'].text
        else:
            if "intern" in text_dict:
                del text_dict["intern"]
            try:
                return text_dict.get(get_language(), text_dict.values()[0]).text
            except IndexError:
                return ''

    def internal_text(self):
        return self.text(internal=True)

    def language_codes(self):
        return ", ".join((text.language for text in self.notetext_set.all()))

    def get_absolute_url(self):
        return navi_url("releasenotes.edit_note", self.navi_context(), True)



class NoteText(models.Model):
    """
    Verwendung: Text zu einem Versionshinweis in einer bestimmten Sprache.

    Index:  id
    Index:  note_id
    Aufbau:
    id          int4    !?+
    modified    datetime   -           Zeitpunkt der letzten Änderung
    language    char7
    text        text2
    note_id     int4    !>releasenotes_note Zugehöriger Hinweis
    timestamp   datetime    !/+ Zeitstempel
    """
    modified = models.DateTimeField(auto_now=True)
    language = models.CharField(max_length=7, choices=LANGUAGE_CHOICES)
    text = models.TextField(core=True)
    note = models.ForeignKey(Note, edit_inline=models.TABULAR, raw_id_admin=True)
    timestamp = models.DateTimeField(auto_now=True)

    class Meta:
        unique_together = (('note', 'language'), )
        ordering = ("language",)

    def __repr__(self):
        return "<NoteText(%s): %s>" % (self.language, self.text)

class LastSync(models.Model):
    """
    Verwendung: Speichert den Zeitpunkt der letzten Synchronisation
                der Versionshinweise zwischen Produktion und QSU.
                Diese Tabelle hat stets nur eine Zeile.
    Index:  id
    Aufbau:
    id          int4    !?+ 
    timestamp   datetime
    """
    timestamp = models.DateTimeField()

    def __repr__(self):
        return "<Sync %s>" % self.timestamp
