# -*- encoding: UTF-8 -*-

from django import oldforms
from django.core import validators
from django.utils.translation import gettext, gettext_lazy, ngettext

from kundebunt.releasenotes import models
#from kundebunt.popkern.formfields import *
#from kundebunt.popkern.navigation import navi_url

_release_help_text = gettext_lazy("Versionsnummern muessen mindestens dreistellig, maximal vierstellig sein, und den Teil ab dem Bindestrich laesst man weg. Beispiele fuer gueltige Versionsnummern sind: 0.7.3 oder 0.7.3.2, nicht aber 0.7, 0.7.8-2 (kein Bindestrich!) und auch nicht 0.7.8.2.8.")

_release_beginn_help_text = gettext_lazy("Hier gibt man die Version ein, ab der der Fehler aufgetreten ist; das wird meist die aktuelle Nummer sein, zur Zeit '%s'. Man kann sie auch leer lassen, das zaehlt als 'schon immer'.\n\n")

_release_behoben_help_text = gettext_lazy("Hier gibt man die Version ein, ab der der Fehler behoben ist; bei der Neueingabe wird laesst man das Feld leer, das bedeutet 'noch nicht behoben'. Die aktuelle Version ist '%s'.\n\n")


class ReleasenotesSearchForm(oldforms.Manipulator):
    def __init__(self):
        self.fields = [
            oldforms.IntegerField("ticket", is_required=False),
            oldforms.CheckboxField("current_only", is_required=False)
            ]

    def search_results(self, data):
        def _sortkey(note):
            rel = note.release_behoben_id
            if rel==None:
                return (-99999999, -note.ticket)
            else:
                return (-rel, note.ticket)
        ticket_id = data.get("ticket")
        if ticket_id:
            notes = models.Note.objects.filter(ticket=ticket_id)
        elif data.get("current_only"):
            notes = models.get_effective_notes(True)
        else:
            notes = models.Note.objects.all()
        if ticket_id:
            notes = models.Note.objects.filter(ticket=ticket_id)
        notes = list(notes)
        notes.sort(key=_sortkey)
        return notes

class ReleasenotesAddForm(oldforms.Manipulator):
    change = False

    def __init__(self):
        self.fields = [
            oldforms.IntegerField("ticket", is_required=True),
            oldforms.SelectField("kind", is_required=True,
                              choices = models.NOTE_KIND_CHOICES),
            oldforms.TextField("release_beginn_text", is_required=False, max_length=16,
                            validator_list = [validators.MatchesRegularExpression('^\d+\.\d+\.\d+(\.\d+)?')]),
            oldforms.TextField("release_behoben_text", is_required=False, max_length=16,
                            validator_list = [validators.MatchesRegularExpression('^\d+\.\d+\.\d+(\.\d+)?')]),
            ]
        self["release_beginn_text"].help_text = (_release_beginn_help_text % models.current_release()) + str(_release_help_text)
        self["release_behoben_text"].help_text = (_release_behoben_help_text % models.current_release()) + str(_release_help_text)
        for i,language in enumerate(models.LANGUAGE_CHOICES):
            self.fields.append(oldforms.LargeTextField("notetext.%d.text" % i))
            self.fields.append(oldforms.TextField("notetext.%d.language" % i))
        self.rel_texts = RelatedTexts(len(models.LANGUAGE_CHOICES))

    def flatten_data(self):
        data = dict([("notetext.%d.language" % i, code) for (i, (code, name)) in enumerate(models.LANGUAGE_CHOICES)])
        if not self.change:
            data["release_beginn_text"] = models.current_release()
        return data

    def save(self, data):
        # Releases
        release_beginn_id = models.get_or_create_release_id(data.get('release_beginn_text'))
        release_behoben_id = models.get_or_create_release_id(data.get('release_behoben_text'))

        # Note
        note = models.Note(
            ticket=data["ticket"],
            kind=data["kind"],
            release_beginn_id = release_beginn_id,
            release_behoben_id = release_behoben_id
            )
        note.save()

        # NoteText
        self._save_texts(note, data)
        return note

    def _save_texts(self, note, data):
        if self.change:
            note.notetext_set.all().delete()
        for (i, (lang_code, lang_name)) in enumerate(models.LANGUAGE_CHOICES):
            prefix = "notetext.%d." % i
            text = data.get(prefix + "text")
            if text:
                note_text = models.NoteText(
                    note_id=note.id,
                    text=text,
                    language=lang_code
                    )
                note_text.save()

    def get_related_objects(self):
        return [self.rel_texts]


class ReleasenotesChangeForm(ReleasenotesAddForm):
    change = True

    def __init__(self, note):
        self.obj_key = note.id
        self.original_object = note
        super(ReleasenotesChangeForm,self).__init__()

    def flatten_data(self):
        note = self.original_object
        texts = note.notetext_set.all()
        data = super(ReleasenotesChangeForm, self).flatten_data()
        data["ticket"] = note.ticket
        data["kind"] = note.kind
        if note.release_beginn:
            data["release_beginn_text"] = str(note.release_beginn)
        if note.release_behoben:
            data["release_behoben_text"] = str(note.release_behoben)
        lang_dict = dict([(text.language, text.text) for text in note.notetext_set.all()])
        for (i, (lang_code, lang_full)) in enumerate(models.LANGUAGE_CHOICES):
            data['notetext.%d.text' % i] = lang_dict.get(lang_code, '')
        return data

    def save(self, data):
        # Releases
        release_beginn_id = models.get_or_create_release_id(data.get('release_beginn_text'))
        release_behoben_id = models.get_or_create_release_id(data.get('release_behoben_text'))

        # Note
        note = self.original_object
        note.ticket = int(data["ticket"])
        note.kind = int(data["kind"])
        note.release_beginn_id = release_beginn_id
        note.release_behoben_id = release_behoben_id
        note.save()

        # NoteText
        self._save_texts(note, data)
        return note


class RelatedTexts(object):
    class DummyField(object):
        """
        Tut so, als ob es ein Datenbank-Feld wär,
        und verweist bei get_manipulator_field_names()
        auf die gewünschten Felder.
        """
        def __init__(self, field_names):
            self.field_names = field_names

        def get_manipulator_field_names(self, dummy):
            return [self.field_names]

    def __init__(self, count):
        self.related_fields = ["text", "language"]
        self.name = 'texts'
        self.opts = models.NoteText._meta
        self._count = count

    def extract_data(self, data):
        """Werte der Datenfelder für die Mailrules.
        Man kann problemlos alles zurückgeben.
        """
        return data

    def get_list(self, original_object):
        """Gibt die Mailrules + zusätzlich Dummies für leere Zeilen zurück"""
        return [None] * self._count  #???

    def editable_fields(self):
        """Hier muss man Plazebos für Datenbank-Felder konstruieren, für
        alle benutzten Felder (nicht nur die editerbaren)."""
        return [self.DummyField(name)
                for name in self.related_fields]


